/*
 * Copyright (C) 2022 user0001
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi;

import java.util.ArrayList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MetaMessage;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.SysexMessage;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.mx30controller.MGStatus;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMessageFactory {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMessageFactory.class);
    
    public static MXMessage createDummy() {
        return new MXMessage();
    }

    public static MXMessage fromClone(MXMessage old) {
        if (old == null) {
            return MXMessageFactory.createDummy();
        }
        MXMessage message = new MXMessage();
        message._extype = old._extype;
        message._port = old._port;
        message._template = old._template.clone();
        message._cached = false;

        message._checksumLength = old._checksumLength;
        message.setValue(old.getValue());
        message.setGate(old.getGate());
        message.setChannel(old.getChannel());
        
        message.setDataentryMSB(old.getDataentryMSB());
        message.setDataentryLSB(old.getDataentryLSB());
        message.setDataentryType(old.getDataentryType());
        message.setValue14bit(old.isValue14bit());

        return message;
    }

    public static MXMessage fromJavaMessage(int port, MidiMessage message) {
        if (message instanceof ShortMessage) {
            ShortMessage msg = (ShortMessage) message;
            return MXMessageFactory.fromShortMessage(port, msg.getStatus(), msg.getData1(), msg.getData2());
        }
        if (message instanceof SysexMessage) {
            MXMessage m = new MXMessage();
            m._port = port;
            byte[] data = ((SysexMessage) message).getData();
            int status = ((SysexMessage) message).getStatus();
            int[] template = new int[data.length + 1];
            template[0] = status;
            for (int i = 0; i < data.length; ++i) {
                template[i + 1] = data[i] & 0xff;
            }
            m._template = template;
            if (status == 240) {
                m._extype = MXMessage.EXTYPE_SYSEX;
            } else if (status == 247) {
                m._extype = MXMessage.EXTYPE_SYSEX_SPECIAL;
            } else {
                _debug.println("Can't inport message " + message.getClass());
            }
            return m;
        }
        if (message instanceof MetaMessage) {
            MetaMessage meta = (MetaMessage)message;
            return fromMeta(port, meta.getType(), meta.getData());
        }
        _debug.println("Unknown " + message.getClass());
        return null;
    }

    public static MXMessage fromMeta(int port, int metaType, byte[] data) {
        MXMessage message = new MXMessage();
        message._port = port;
        message._extype = MXMessage.EXTYPE_META;
        message.setMetaType(metaType);

        String text = null;
        try {
            text = new String(data, "ASCII");
            text = new String(data);
            text = new String(data, "SJIS");
        }catch(Exception e) {
            e.printStackTrace();
        }

        message._metaText = text;
        message._dataBytes = data;
        return message;
    }

    public static MXMessage fromProgramInc(int port, int channel) {
        MXMessage message = new MXMessage();
        message._port = port;
        message.setChannel(channel);
        message._extype = MXMessage.EXTYPE_PROGRAM_INC;
        return message;
    }

    public static MXMessage fromProgramDec(int port, int channel) {
        MXMessage message = new MXMessage();
        message._port = port;
        message.setChannel(channel);
        message._extype = MXMessage.EXTYPE_PROGRAM_DEC;
        return message;
    }

    public static MXMessage fromSysexMessage(int port, int status, byte[] data) {
        MXMessage message = new MXMessage();
        message._port = port;
        int[] template = new int[data.length + 1];
        template[0] = status & 0x0ff;
        for (int i = 0; i < data.length; ++i) {
            template[i + 1] = data[i] & 0x0ff;
        }
        if (status == 240) {
            message._extype = MXMessage.EXTYPE_SYSEX;
        } else {
            message._extype = MXMessage.EXTYPE_SYSEX_SPECIAL;
        }
        message._template = template;
        return message;
    }

    public static MidiMessage toJavaDataentry1MSB(MXMessage msg) throws InvalidMidiDataException {
        if (msg.getCommand() == MXMidi.COMMAND_CONTROLCHANGE && msg.getGate() == MXMidi.DATA1_CC_DATAENTRY) {
            switch (msg.getDataentryType()){
                case MXMidi.DATAENTRY_TYPE_RPN:
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(),  MXMidi.DATA1_RPN_MSB,  msg.getDataentryMSB());
                case MXMidi.DATAENTRY_TYPE_NRPN:
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(),  MXMidi.DATA1_NRPN_MSB,  msg.getDataentryMSB());
            }
        }
        return null;
    }

    public static MidiMessage toJavaDataentry2LSB(MXMessage msg) throws InvalidMidiDataException {
        if (msg.getCommand() == MXMidi.COMMAND_CONTROLCHANGE && msg.getGate() == MXMidi.DATA1_CC_DATAENTRY) {
            switch (msg.getDataentryType()){
                case MXMidi.DATAENTRY_TYPE_RPN:
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(),  MXMidi.DATA1_RPN_LSB,  msg.getDataentryLSB());
                case MXMidi.DATAENTRY_TYPE_NRPN:
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(),  MXMidi.DATA1_NRPN_LSB,  msg.getDataentryLSB());
            }
        }
        return null;
    }

    public static MidiMessage toDataentryValueMSB(MXMessage msg) throws InvalidMidiDataException {
        if (msg._template.length >= 2) {
            if ((msg._template[1] & 0xff00) != 0) {
                int sendValue = msg.getDValue(msg._template[1]);
                if (sendValue >= 0) {
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(), msg.getGate(), sendValue);
                }
            }else {
                int sendValue = msg._template[1] & 0xff;
                return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(), msg.getGate(), sendValue);
            }
        }else {
        }
        return null;
    }

    public static MidiMessage toDataentryValueLSB(MXMessage msg) throws InvalidMidiDataException {
        if (msg._template.length >= 3) {
            if ((msg._template[2] & 0xff00) != 0) {
                int sendValue = msg.getDValue(msg._template[2]);
                if (sendValue >= 0) {
                    return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(), msg.getGate() + 0x20 , sendValue);
                }
            }else {
                int sendValue = msg._template[2] & 0xff;
                return new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + msg.getChannel(), msg.getGate() + 0x20, sendValue);
            }
        }else {
        }
        return null;
    }
    
    public static MidiMessage toJavaMessage(MXMessage msg) throws InvalidMidiDataException {
        byte[] data = msg.getDataBytes();
        int extype = msg._extype;
        if (extype == MXMessage.EXTYPE_DUMMY) {
            return null;
        }
        if (extype == MXMessage.EXTYPE_SHORTMESSAGE) {
            return new ShortMessage(data[0] & 0x0ff, data[1] & 0x0ff, data[2] & 0x0ff);
        }

        byte[] newData = new byte[data.length - 1];
        for (int i = 0; i < data.length - 1; ++ i) {
            newData[i] = data[i + 1];
        }

        switch(extype) {
            case MXMessage.EXTYPE_SYSEX:
                return new SysexMessage(data[0] & 0xff, newData, newData.length);
            case MXMessage.EXTYPE_SYSEX_SPECIAL:
                return new SysexMessage(data[0] & 0xff, newData, newData.length);
            case MXMessage.EXTYPE_META:
                return new MetaMessage(msg.getMetaType(), newData, newData.length);
        }
        throw new InvalidMidiDataException("Not supported.");
    }

    public static MXMessage fromShortMessage(int port, int status, int data1, int data2) {
        MXMessage message = new MXMessage();
        message._port = port;
        message._cached = false;
        int command = status;
        int channel = 0;
        
        if (status >= 0x80 && status <= 0xe0) {
            command = status & 0xf0;
            channel = status & 0x0f;
        }

        message._extype = MXMessage.EXTYPE_SHORTMESSAGE;
        message.setChannel(channel);

        if (command < 0 || command > 255) {
            _debug.println("command = " + command);
            return null;
        }
        if (channel < 0 || channel > 15) {
            _debug.println("channel= " + channel);
            return null;
        }
        if (data1 < 0 || data1 > 127) {
            _debug.println("data1 = " + data1);
            return null;
        }
        if (data2 < 0 || data2 > 127) {
            _debug.println("data2 = " + data2);
            return null;
        }

        message._template[0] = status & 0x0ff;
        message._template[1] = data1 & 0x0ff;
        message._template[2] = data2 & 0x0ff;

        int defValue = 0;

        switch (command) {
            case MXMidi.COMMAND_PROGRAMCHANGE:
                message.setChannel(channel);
                message._template[1] = MXMessage.DTEXT_GL;
                message._template[2] = 0;
                message.setGate(data1);
                break;
            case MXMidi.COMMAND_CONTROLCHANGE:
                message._template[1] = MXMessage.DTEXT_GL;
                message._template[2] = MXMessage.DTEXT_VL;
                message.setGate(data1);
                message.setValue(data2);
                switch(message.getGate()) {
                    case MXMidi.DATA1_CCPANPOT:
                        defValue = 64;
                        break;
                    case MXMidi.DATA1_CC_EXPRESSION:
                        defValue = 127;
                        break;
                }
                break;
            case MXMidi.COMMAND_NOTEON:
            case MXMidi.COMMAND_NOTEOFF:
            case MXMidi.COMMAND_POLYPRESSURE:
                message._template[1] = MXMessage.DTEXT_GL;
                message._template[2] = MXMessage.DTEXT_VL;
                message.setGate(data1);
                message.setValue(data2);
                break;
            case MXMidi.COMMAND_PITCHWHEEL:
                message._template[1] = MXMessage.DTEXT_VL;
                message._template[2] = MXMessage.DTEXT_VH;
                message.setValue((data1 & 127) | (data2 << 7));
                break;
            case MXMidi.COMMAND_CHANNELPRESSURE:
                message._template[1] = MXMessage.DTEXT_VL;
                message.setValue(data1);
                break;
            default:
                if (command >= 240 && command <= 247) {
                    if (command == MXMidi.STATUS_SONGPOSITION) {
                        message._template[1] = MXMessage.DTEXT_VL;
                        message._template[2] = MXMessage.DTEXT_VH;
                        message.setValue((data1 & 127) | (data2 << 7));
                    }
                    if (command == MXMidi.STATUS_SONGSELECT) {
                        message._template[1] = MXMessage.DTEXT_VL;
                        message.setValue(data1);
                    }
                }
                break;
        }
        return message;
    }

    public static MXMessage fromDtext(String text, int port, int channel, int gate, int value)  {
        if (text == null || text.length() == 0) {
            return createDummy();
        }
        
        while (text.startsWith(" ")) {
            text = text.substring(1);
        }
        while (text.endsWith(" ")) {
            text = text.substring(0, text.length() - 1);
        }

        if (text.equals(MXMessage.EXCOMMAND_PROGRAM_INC)) {
            return MXMessageFactory.fromProgramInc(0, 0);
        }
        if (text.equals(MXMessage.EXCOMMAND_PROGRAM_DEC)) {
            return MXMessageFactory.fromProgramDec(0, 0);
        }

        try {
            int rpn_msb;
            int rpn_lsb;
            int nrpn_msb;
            int nrpn_lsb;

            char[] line = text.toCharArray();

            char[] word = new char[line.length];
            int wx = 0;

            int readX = 0;
            ArrayList<String> separated = new ArrayList();

            boolean inChecksum = false;
            int checksumKeep = -1;

            while(readX < line.length) {
                char ch = line[readX ++];
                if (ch == '[') {
                    inChecksum = true;
                    checksumKeep = 0;
                    continue;
                }
                if (ch == ']') {
                    if (inChecksum) {
                        inChecksum = false;
                        if (wx != 0) {
                            separated.add(new String(word, 0, wx));
                        }
                        separated.add("#CHECKSUM");
                        wx = 0;
                    }else {
                        _debug.println("Checksum have not opened");
                        _debug.printStackTrace();
                    }
                    continue;
                }
                if (ch == ' '|| ch == '\t' || ch == ',') {
                    if (wx != 0) {
                        separated.add(new String(word, 0, wx));
                        if (inChecksum) {
                            checksumKeep ++;
                        }
                    }
                    wx = 0;
                    continue;
                }
                word[wx ++] = ch;
            }

            if (wx != 0) {
                separated.add(new String(word, 0, wx));
                if (inChecksum) {
                    checksumKeep ++;
                }
                wx = 0;
            }

            int gatetemp = -1;
            if (text.contains("@")) {
                ArrayList<String> sepa2 = new ArrayList();
                for (int sx = 0; sx < separated.size(); ++ sx) {
                    String str = separated.get(sx);
                    if (str.startsWith("@")) {
                        if (str.equalsIgnoreCase("@PB")) {
                            sepa2.add("#ECH");
                            sepa2.add(separated.get(++ sx));
                            sepa2.add(separated.get(++ sx));
                        }
                        else if (str.equalsIgnoreCase("@CP")) {
                            sepa2.add("#DCH");
                            sepa2.add(separated.get(++ sx));
                            sepa2.add("#NONE");
                        }
                        else if (str.equalsIgnoreCase("@PKP")) {
                            sepa2.add("#ACH");
                            String t = separated.get(++sx);
                            if(t.startsWith("#")) {
                                sepa2.add(t);
                            }else {
                                gatetemp = MXUtil.parseTextForNumber(t);
                                sepa2.add("#GL");
                            }
                            sepa2.add(separated.get(++ sx));
                        }
                        else if (str.equalsIgnoreCase("@CC")) {
                            sepa2.add("#BCH");
                            String t = separated.get(++sx);
                            if(t.startsWith("#")) {
                                sepa2.add(t);
                            }else {
                                gatetemp = MXUtil.parseTextForNumber(t);
                                sepa2.add("#GL");
                            }
                            sx ++;
                            if (separated.size() <= sx) { //TODO ?? is it right?
                                return null;
                            }
                            sepa2.add(separated.get(sx));
                        }
                        else if (str.equalsIgnoreCase("@SYSEX")) {
                            //THRU (no need recompile)
                        }
                        else if (str.equalsIgnoreCase("@RPN")) {
                            int  msb = MXMessage.readAliasText(separated.get(++sx));
                            int  lsb = MXMessage.readAliasText(separated.get(++sx));
                            int val1 = MXMessage.readAliasText(separated.get(++sx));
                            int val2 = -1;
                            if (separated.size() >= sx +2) {
                                val2 = MXMessage.readAliasText(separated.get(++sx));
                            }
                            return fromDataentry(MXMidi.DATAENTRY_TYPE_RPN,  0, channel, msb, lsb, val1, val2);
                        }
                        else if (str.equalsIgnoreCase("@NRPN")) {
                            int  msb = MXMessage.readAliasText(separated.get(++sx));
                            int  lsb = MXMessage.readAliasText(separated.get(++sx));
                            int val1 = MXMessage.readAliasText(separated.get(++sx));
                            int val2 = -1;
                            if (separated.size() >= sx +2) {
                                val2 = MXMessage.readAliasText(separated.get(++sx));
                            }
                            return fromDataentry(MXMidi.DATAENTRY_TYPE_NRPN,  0, channel, msb, lsb, val1, val2);
                        }else {
                            _debug.println("Not Support [" + text + "]");
                            return null;
                        }
                    }else {
                        sepa2.add(str);
                    }
                }
                separated = sepa2;
            }

            // cleanup
            int[] compiled = new int[line.length];
            int cx = 0;
            int px = 0;
            
            for (int sx = 0; sx < separated.size(); ++ sx) {
                String str = separated.get(sx);
                int code = MXMessage.readAliasText(str);
                if (code < 0) {
                    return null;
                }
                compiled[px++] = code;
                continue;
            }
            int[] template = new int[Math.max(px, 3)];
            for (int i = 0; i < px; ++ i) {
                template[i] = compiled[i];
            }
            MXMessage ret = new MXMessage(template, checksumKeep);

            ret.setChannel(channel);
            if (gatetemp > 0) {
                ret.setGate(gatetemp);
            }else {
                ret.setGate(gate);
            }
            ret.setValue(value);
            return ret;
        }catch(Exception e) {
            _debug.printStackTrace(e);
            return null;
        }
    }
    
    public static String toDText(MXMessage message) {
        ArrayList<String> array = message.toDArray();
        
        StringBuffer text = new StringBuffer();
        String last = "]";
        for (String seg : array) {
            if (seg.length() == 0) {
                continue;
            }
            if (text.length() >= 0) {
                if (seg.equals("[") || seg.equals("]") || last.equals("[") || last.equals("]")) {
                    // nothing
                }else {
                    text.append(" ");
                }
            }
            last = seg;
            text.append(seg);
        }
        return text.toString();
    }
    
    public static MXMessage fromDataentry(int type, int port, int channel, int msb, int lsb, int valuemsb, int valuelsb) {
        int[] template = new int[3];
        if (type == MXMidi.DATAENTRY_TYPE_RPN) {
            template[0] = MXMessage.DTEXT_RPN;
        }else if (type == MXMidi.DATAENTRY_TYPE_NRPN) {
            template[0] = MXMessage.DTEXT_NRPN;
        }else {
            return null;
        }
        template[1] = valuemsb;
        template[2] = valuelsb >= 0 ? valuelsb : 0;
        MXMessage message = new MXMessage(template, -1);
        message.setDataentryType(type);
        message.setDataentryMSB(msb);
        message.setDataentryMSB(lsb);
        return message;
    }
}
