/*
 * Copyright (C) 2022 user0001
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public Lficense for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.synthtarou.midimixer.libs.midi;

import java.io.PrintStream;
import java.util.ArrayList;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXWrapList;

/**
 *
 * @author YOSHIDA Shintarou
 * infomation from g200kg Music & Software https://www.g200kg.com/
 */
public final class MXMessage {
    public boolean canPaired14bit() {
        makeSomeCached();
        if (!_cachedHasValue) {
            return false;
        }
        if (_cachedHasHi) {
            return true;
        }

        if (getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            int cc = getGate();
            if (cc >= 0 && cc <= 31) {
                return true;
            }
        }
        return false;
    }
    

    /**
     * @return the it14bitCC
     */
    public boolean isValue14bit() {
        if (_cachedHasHi) {
            return true;
        }
        return _value14bit;
    }
    
    public void setValue14bitAndZoom(boolean value14bit) {
        if (value14bit) {
            if (!canPaired14bit() && !hasValueHiField()) {
                System.out.println("no spot for 14bit value");
                return;
            }
        }
        if (_value14bit != value14bit) {
            int value = getValue();
            if (value14bit) {
                value = value * 128 + value;
            }else {
                value = value / 128;
            }
        }
        this._value14bit = value14bit;
    }

    /**
     * @param value14bit the it14bitCC to set
     */
    public void setValue14bit(boolean value14bit) {
        if (value14bit) {
            if (!canPaired14bit() && !hasValueHiField()) {
                System.out.println("no spot for 14bit value");
                return;
            }
        }
        this._value14bit = value14bit;
    }

    /**
     * @return the _dataentryType
     */
    public int getDataentryType() {
        return _dataentryType;
    }

    /**
     * @param dataentryType the _dataentryType to set
     */
    public void setDataentryType(int dataentryType) {
        if (dataentryType != this._dataentryType) {
            this._dataentryType = dataentryType;
            _cached = false;
        }
    }

    /**
     * @return the _dataentryMSB
     */
    public int getDataentryMSB() {
        return _dataentryMSB;
    }

    /**
     * @param dataentryMSB the _dataentryMSB to set
     */
    public void setDataentryMSB(int dataentryMSB) {
        if (this._dataentryMSB != dataentryMSB) {
            this._dataentryMSB = dataentryMSB;
            _cached = false;
        }
    }

    /**
     * @return the _dataentryLSB
     */
    public int getDataentryLSB() {
        return _dataentryLSB;
    }

    /**
     * @param dataentryLSB the _dataentryLSB to set
     */
    public void setDataentryLSB(int dataentryLSB) {
        if (this._dataentryLSB != dataentryLSB) {            
            this._dataentryLSB = dataentryLSB;
            _cached = false;
        }
    }
    
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMessage.class);
    public static final String EXCOMMAND_PROGRAM_INC = "@PROG_INC";
    public static final String EXCOMMAND_PROGRAM_DEC = "@PROG_DEC";

    public static final int EXTYPE_INIT_ERROR = 0xfe00;
    public static final int EXTYPE_DUMMY = 0xff00;
    public static final int EXTYPE_SHORTMESSAGE = 0x0100;
    public static final int EXTYPE_PROGRAM_INC = 0x0200;
    public static final int EXTYPE_PROGRAM_DEC = 0x0300;
    public static final int EXTYPE_SYSEX = 0x0400;
    public static final int EXTYPE_SYSEX_SPECIAL = 0x0500;
    public static final int EXTYPE_META = 0x0600; //SEQ NUMBER

    public int _extype = EXTYPE_DUMMY;
    int _port;
    public boolean _cached = false;

    private int _value = -1;
    private int _gate = -1;
    private int _channel = 0;
    
    private int _dataentryType = MXMidi.DATAENTRY_TYPE_NONE;
    private int _dataentryMSB = -1;
    private int _dataentryLSB = -1;

    protected byte[] _dataBytes = new byte[3];
    protected boolean _cachedHasHi, _cachedHasValue, _cachedHasGate, _cachedHasGateHi;
    protected int _checksumLength = -1;
    private boolean _value14bit;

    public int[] _template = new int[3];

    public static final int DTEXT_NONE = 0x100;
    public static final int DTEXT_VL = 0x200;
    public static final int DTEXT_VH = 0x300;
    public static final int DTEXT_GL = 0x400;
    public static final int DTEXT_GH = 0x500;
    static final int DTEXT_CH = 0x600;
    static final int DTEXT_1CH = 0x700;
    static final int DTEXT_2CH = 0x800;
    static final int DTEXT_3CH = 0x900;
    static final int DTEXT_PCH = 0xA00;
    static final int DTEXT_1RCH = 0xB00;
    static final int DTEXT_2RCH = 0xC00;
    static final int DTEXT_3RCH = 0xD00;
    static final int DTEXT_4RCH = 0xE00;
    static final int DTEXT_VF1 = 0xF00;
    static final int DTEXT_VF2 = 0x1000;
    static final int DTEXT_VF3 = 0x1100;
    static final int DTEXT_VF4 = 0x1200;
    static final int DTEXT_VPGL = 0x1300;
    static final int DTEXT_VPGH = 0x1400;
    static final int DTEXT_CCNUM = 0x1500;
    static final int DTEXT_RSCTRT1 = 0x1A00;
    static final int DTEXT_RSCTRT2 = 0x1B00;
    static final int DTEXT_RSCTRT3 = 0x1C00;
    static final int DTEXT_RSCTRT1P = 0x1D00;
    static final int DTEXT_RSCTRT2P = 0x1E00;
    static final int DTEXT_RSCTRT3P = 0x1F00;
    static final int DTEXT_RSCTPT1 = 0x2000;
    static final int DTEXT_RSCTPT2 = 0x2100;
    static final int DTEXT_RSCTPT3 = 0x2200;
    static final int DTEXT_RSCTPT1P = 0x2300;
    static final int DTEXT_RSCTPT2P = 0x2400;
    static final int DTEXT_RSCTPT3P = 0x2500;
    public static final int DTEXT_CHECKSUM = 0x2600;

    static final int DTEXT_4CH = 0x2700;
    static final int DTEXT_5CH = 0x2800;
    static final int DTEXT_6CH = 0x2900;
    static final int DTEXT_7CH = 0x2a00;
    static final int DTEXT_8CH = 0x2b00;
    static final int DTEXT_9CH = 0x2c00;
    static final int DTEXT_ACH = 0x2d00;
    static final int DTEXT_BCH = 0x2e00;
    static final int DTEXT_CCH = 0x2f00;
    static final int DTEXT_DCH = 0x3000;
    static final int DTEXT_ECH = 0x3100;
    static final int DTEXT_FCH = 0x3200;

    public static final int DTEXT_RPN = 0x3300;
    public static final int DTEXT_NRPN = 0x3400;
    
    private int _metaType;
    public String _metaText;
    
    static MXWrapList<Integer> textAlias = new MXWrapList();
   
    static {
        textAlias.addNameAndValue("#NONE", DTEXT_NONE);
        textAlias.addNameAndValue("#VL", DTEXT_VL);
        textAlias.addNameAndValue("#VH", DTEXT_VH);
        textAlias.addNameAndValue("#GL", DTEXT_GL);
        textAlias.addNameAndValue("#GH", DTEXT_GH);
        textAlias.addNameAndValue("#CH", DTEXT_CH);
        textAlias.addNameAndValue("#1CH", DTEXT_1CH);
        textAlias.addNameAndValue("#2CH", DTEXT_2CH);
        textAlias.addNameAndValue("#3CH", DTEXT_3CH);
        textAlias.addNameAndValue("#PCH", DTEXT_PCH);
        textAlias.addNameAndValue("#1RCH", DTEXT_1RCH);
        textAlias.addNameAndValue("#2RCH", DTEXT_2RCH);
        textAlias.addNameAndValue("#3RCH", DTEXT_3RCH);
        textAlias.addNameAndValue("#4RCH", DTEXT_4RCH);
        textAlias.addNameAndValue("#VF1", DTEXT_VF1);
        textAlias.addNameAndValue("#VF2", DTEXT_VF2);
        textAlias.addNameAndValue("#VF3", DTEXT_VF3);
        textAlias.addNameAndValue("#VF4", DTEXT_VF4);
        textAlias.addNameAndValue("#VPGL", DTEXT_VPGL);
        textAlias.addNameAndValue("#VPGH", DTEXT_VPGH);
        textAlias.addNameAndValue("#RSCTRT1", DTEXT_RSCTRT1);
        textAlias.addNameAndValue("#RSCTRT2", DTEXT_RSCTRT2);
        textAlias.addNameAndValue("#RSCTRT3", DTEXT_RSCTRT3);
        textAlias.addNameAndValue("#RSCTRT1P", DTEXT_RSCTRT1P);
        textAlias.addNameAndValue("#RSCTRT2P", DTEXT_RSCTRT2P);
        textAlias.addNameAndValue("#RSCTRT3P", DTEXT_RSCTRT3P);
        textAlias.addNameAndValue("#RSCTPT1", DTEXT_RSCTPT1);
        textAlias.addNameAndValue("#RSCTPT2", DTEXT_RSCTPT2);
        textAlias.addNameAndValue("#RSCTPT3", DTEXT_RSCTPT3);
        textAlias.addNameAndValue("#RSCTPT1P", DTEXT_RSCTPT1P);
        textAlias.addNameAndValue("#RSCTPT2P", DTEXT_RSCTPT2P);
        textAlias.addNameAndValue("#RSCTPT3P", DTEXT_RSCTPT3P);
        textAlias.addNameAndValue("#CHECKSUM", DTEXT_CHECKSUM);

        textAlias.addNameAndValue("#4CH", DTEXT_4CH);
        textAlias.addNameAndValue("#5CH", DTEXT_5CH);
        textAlias.addNameAndValue("#6CH", DTEXT_6CH);
        textAlias.addNameAndValue("#7CH", DTEXT_7CH);
        textAlias.addNameAndValue("#8CH", DTEXT_8CH);
        textAlias.addNameAndValue("#9CH", DTEXT_9CH);
        textAlias.addNameAndValue("#ACH", DTEXT_ACH);
        textAlias.addNameAndValue("#BCH", DTEXT_BCH);
        textAlias.addNameAndValue("#CCH", DTEXT_CCH);
        textAlias.addNameAndValue("#DCH", DTEXT_DCH);
        textAlias.addNameAndValue("#ECH", DTEXT_ECH);
        textAlias.addNameAndValue("#FCH", DTEXT_FCH);

        textAlias.addNameAndValue("#RPN", DTEXT_RPN); //@RPN [RPN MSB] [RPN LSB] [Data MSB] [Data LSB]
        textAlias.addNameAndValue("#NRPN", DTEXT_NRPN); //@NRPN [NRPN MSB] [NRPN LSB] [Data MSB] [Data LSB]
    }
    
    public int getDValue(int alias) {
        int newValue = getValue();
        int newGate = getGate();
        switch(alias & 0xff00) {
            case DTEXT_NONE:
                return -1;
            case DTEXT_VL:
                alias = newValue & 0x7f;
                break;
            case DTEXT_VH:
                alias = (newValue >> 7) & 0x7f;
                break;
            case DTEXT_GL:
                alias = newGate & 0x7f;
                break;
            case DTEXT_GH:
                alias = (newGate >> 7) & 0x7f;
                break;
            case DTEXT_CH:
                alias = getChannel();
                break;
            case DTEXT_1CH:
                alias = 0x10 + getChannel();
                break;
            case DTEXT_2CH:
                alias = 0x20 + getChannel();
                break;
            case DTEXT_3CH:
                alias = 0x30 + getChannel();
                break;
            case DTEXT_4CH:
                alias = 0x40 + getChannel();
                break;
            case DTEXT_5CH:
                alias = 0x50 + getChannel();
                break;
            case DTEXT_6CH:
                alias = 0x60 + getChannel();
                break;
            case DTEXT_7CH:
                alias = 0x70 + getChannel();
                break;
            case DTEXT_8CH:
                alias = 0x80 + getChannel();
                break;
            case DTEXT_9CH:
                alias = 0x90 + getChannel();
                break;
            case DTEXT_ACH:
                alias = 0xA0 + getChannel();
                break;
            case DTEXT_BCH:
                alias = 0xB0 + getChannel();
                break;
            case DTEXT_CCH:
                alias = 0xC0 + getChannel();
                break;
            case DTEXT_DCH:
                alias = 0xD0 + getChannel();
                break;
            case DTEXT_ECH:
                alias = 0xE0 + getChannel();
                break;
            case DTEXT_FCH:
                alias = 0xF0 + getChannel();
                break;
            case DTEXT_PCH:
                if (_port >= 0 && _port <= 3) {
                    alias = _port * 0x10 + getChannel();
                }else {
                    alias = 0x30 + getChannel();
                }
                break;
            case DTEXT_1RCH:
            case DTEXT_2RCH:
            case DTEXT_3RCH:
            case DTEXT_4RCH:
                throw new IllegalArgumentException("1RCH, 2RCH, 3RCH, 4RCH not supported.");
                //break;
            case DTEXT_VF1:
                alias = (getValue()) & 0x0f;
                break;
            case DTEXT_VF2:
                alias = (getValue() >> 4) & 0x0f;
                break;
            case DTEXT_VF3:
                alias = (getValue() >> 8) & 0x0f;
                break;
            case DTEXT_VF4:
                alias = (getValue() >> 12) & 0x0f;
                break;
            case DTEXT_VPGL:
                alias = (getValue() + getGate()) & 0x7f;
                break;
            case DTEXT_VPGH:
                alias = ((getValue() + getGate()) >> 7) & 0x7f;
                break;
            case DTEXT_RSCTRT1:
            case DTEXT_RSCTRT2:
            case DTEXT_RSCTRT3:
                throw new IllegalArgumentException("RSCTRT1, RSCTRT2, RSCTRT3 not supported.");
                //break;
            case DTEXT_RSCTRT1P:
            case DTEXT_RSCTRT2P:
            case DTEXT_RSCTRT3P:
                throw new IllegalArgumentException("RSCTRT1P, RSCTRT2P, RSCTRT3P not supported.");
                //break;
            case DTEXT_RSCTPT1:
            case DTEXT_RSCTPT2:
            case DTEXT_RSCTPT3:
                throw new IllegalArgumentException("RSCTPT1, RSCTPT2, RSCTPT3 not supported.");
                 //break;
            case DTEXT_RSCTPT1P:
            case DTEXT_RSCTPT2P:
            case DTEXT_RSCTPT3P:
                throw new IllegalArgumentException("RSCTPT1P, RSCTPT2P, RSCTPT3P not supported.");
                //break;
/*
static final int DTEXT_CCNUM = 0x1500;
*/
            case DTEXT_CHECKSUM:
                //_checksumTo = i;
                break;

            case DTEXT_RPN:
                new IllegalArgumentException("if DTEXT_RPN called shoudl handleit");
                alias = 0;
                break;

            case DTEXT_NRPN:
                new IllegalArgumentException("if DTEXT_NRPN called shoudl handleit");
                alias = 0;
                break;

            case 0:
                return (byte)alias;
                
            default:
                boolean haveEx = false;
                //TODO
                throw new IllegalArgumentException("something wrong " + Integer.toHexString(alias));
        }
        return (byte) alias;
    }

    public void makeSomeCached() {
        if (_cached) { 
            return; 
        }
        synchronized(this) {
            if (_template.length != _dataBytes.length) {
                _dataBytes = new byte[_template.length];
            }
            int _checksumTo = -1;
            _dataentryType = MXMidi.DATAENTRY_TYPE_NONE;
            
            if (_template[0] == DTEXT_RPN || _template[0] == DTEXT_NRPN) {
                if (_template[0] == DTEXT_RPN) {
                    _dataentryType = MXMidi.DATAENTRY_TYPE_RPN;
                }else {
                    _dataentryType = MXMidi.DATAENTRY_TYPE_NRPN;
                }
            }

            for (int i = 0; i < _template.length; ++ i) {
                int x = _template[i];
                if ((x & 0xff00) != 0) {
                    if (x == DTEXT_CHECKSUM) {
                        _checksumTo = i;
                        x = 0;
                    }else {
                        x = getDValue(x);
                    }
                }
                _dataBytes[i] = (byte)(x & 0xff);
            }
            _cachedHasHi = false;
            _cachedHasValue = false;
            _cachedHasGate = false;
            _cachedHasGateHi = false;
            for (int i = 0; i < _template.length; ++ i) {
                if (_template[i] == DTEXT_VL) {
                    _cachedHasValue = true;
                }
                if (_template[i] == DTEXT_VH) {
                    _cachedHasHi = true;
                }
                if (_template[i] == DTEXT_GL) {
                    _cachedHasGate = true;
                }
                if (_template[i] == DTEXT_GH) {
                    _cachedHasGateHi = true;
                }
            }
            if (_dataentryType != MXMidi.DATAENTRY_TYPE_NONE) {
                _dataBytes[0] = (byte)(MXMidi.COMMAND_CONTROLCHANGE + getChannel());
                _dataBytes[1] = MXMidi.DATA1_CC_DATAENTRY;
                _dataBytes[2] = 0;
            }else {
                int command = _dataBytes[0] & 0xf0;
                if (command >= 0x80 && command <= 0xe0) {
                    _dataBytes[0] = (byte)(command + getChannel());
                }
                if (_checksumLength >= 0 && _checksumTo >= 0) {
                    int x128 = 0;
                    for (int x = _checksumTo - _checksumLength; x < _checksumTo; ++ x) {
                        x128 += _dataBytes[x];
                    }
                    x128 = x128 & 0x7f;
                    int r = 128 - x128;
                    _dataBytes[_checksumTo] = (byte)(r & 0x7f);
                }
            }
            
            _cached = true;
        }
    }

    public int getPort() {
        return _port;
    }
    
    public void setPort(int port) {
        if (_port != port) {
            _cached = false;
            _port = port;
        }
    }
 
    public int getStatus() {
        makeSomeCached();
        int status = _dataBytes[0] & 0xff;
        return status;
    }
    
    public int getCommand() {
        makeSomeCached();
        int status = _dataBytes[0] & 0xff;
        if (status >= 0x80 && status <= 0xe0) {
            return status & 0xf0;
        }else {
            return status;
        }
    }
    
    public void setStatus(int status) {
        _cached = false;
        if (_extype == EXTYPE_SHORTMESSAGE) {
            _template[0] = status & 0xff;
        }else {
            throw new IllegalStateException("setStatus on none ShortMessage");
        }
    }
    
    public void setChannel(int channel) {
        if (_channel != channel) {
            _cached = false;
            _channel = channel;
        }
    }
    
    public int getData1FromBytes() {
        if (hasGateLowField()) {
            if (_template[1] == DTEXT_GL && !hasGateHiField()) {
                return getGate();
            }
        }
        makeSomeCached();
        return _dataBytes[1] & 0xff;
    }

    public int getData2FromBytes() {
        makeSomeCached();
        return _dataBytes[2] & 0xff;
    }
 
    public int getNoteNumberFromBytes() {
        switch(getCommand()) {
        case MXMidi.COMMAND_NOTEON:
        case MXMidi.COMMAND_NOTEOFF:
        case MXMidi.COMMAND_POLYPRESSURE:
            return getData1FromBytes();
        }
        _debug.println("Its not note message.");
        return 0;
    }

    public int getVelocityFromBytes() {
        switch(getCommand()) {
        case MXMidi.COMMAND_NOTEON:
        case MXMidi.COMMAND_NOTEOFF:
        case MXMidi.COMMAND_POLYPRESSURE:
            return getData2FromBytes();
        }
        _debug.println("Its not note message.");
        return 0;
    }

    public int getChannel() {
        if (_channel < 0 || _channel > 15) {
            _debug.println("getAsChannel " + _channel);
            return 0;
        }
        return _channel;
    }
    
    public int getValue() {
        return _value;
    }

    public  void setValue(int value) {
        if (value != _value) {
            _value = value;
            _cached = false;
        }
    }

    public int getGate() {
        return _gate;
    }

    public void setGate(int gate) {
        if (gate != _gate) {
            _gate = gate;
            _cached = false;
        }
    }

    public int getMetaType() {
        return _metaType;
    }

    public void setMetaType(int metaCode) {
        if (metaCode != _metaType) {
            _metaType = metaCode;
            _cached = false;
        }
    }

    protected MXMessage() {
        _extype = EXTYPE_DUMMY;
    }

    protected MXMessage(int[] template, int checksumLength) {
        _template = template;
        _checksumLength = checksumLength;
        if (template[0] == 0) {
            _extype = EXTYPE_DUMMY;
        }else if (template[0] == MXMidi.STATUS_SYSEXSTART) {
            _extype = EXTYPE_SYSEX;
        }else if (template[0] == MXMidi.STATUS_SYSEXFIN) {
            _extype = EXTYPE_SYSEX_SPECIAL;
        }else {
            _extype = EXTYPE_SHORTMESSAGE;
        }
        if ((template[0] & 0xf0) == MXMidi.COMMAND_CONTROLCHANGE) {
            if ((template[1] & 0xff00) == 0) {
                _gate = template[1];
                template[1] = DTEXT_GL;
            }
            if ((template[2] & 0xff00) == 0) {
                _value = template[1];
                template[1] = DTEXT_VL;
            }
        }
    }

    public byte[] getDataBytes() {
        makeSomeCached();
        return _dataBytes;        //SYSEXの場合１バイト目は、STATUSに入る
    }

    public boolean isMessageTypeChannel() {
        makeSomeCached();
        if (_extype != EXTYPE_SHORTMESSAGE) {
            return false;
        }
        if (getCommand() >= 0x80 && getCommand() <= 0xe0) {
            return true;
        }
        return false;
    }

    public boolean isMessageTypeSystemKnown() {
        makeSomeCached();

        if (_extype != EXTYPE_SHORTMESSAGE) {
            return false;
        }
        if (getCommand() >= 0x80 && getCommand() <= 0xe0) {
            return true;
        }
        if (getCommand() >= 0xf0 && getCommand() <= 0xf7) {
            return true;
        }
        
        return false;
    }

    public boolean isMessageTypeDataentry() {
        makeSomeCached();

        if (getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            switch (getGate()) {
                case MXMidi.DATA1_CC_DATAENTRY:
                case MXMidi.DATA1_CC_DATAINC:
                case MXMidi.DATA1_CC_DATADEC:
                    return true;
            }
        }
        return false;
    }

    public String toString() {
        String str = toStringHeader();
        if (getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
            return str + "("+ getGate() + ")";
        }else {
            return str + "("+ getValue() + ")";
        }
    }
    
        
    public String toStringHeader(int min, int max) {
        String str = toStringHeader();
        if (min != max) {
            return str + "= Range" + min + " to " + max + "";
        }else {
            return str + "= " + max;
        }
    }
    
    public String toStringHeader() {
        makeSomeCached();
        
        int port = getPort();
        
        switch(getDataentryType()) {
            case MXMidi.DATAENTRY_TYPE_RPN:
                return "RPN[" + MXUtil.toHexFF(getDataentryMSB()) + ":" + MXUtil.toHexFF(getDataentryLSB()) + "]";
            case MXMidi.DATAENTRY_TYPE_NRPN:
                return "NRPN[" + MXUtil.toHexFF(getDataentryMSB()) + ":" + MXUtil.toHexFF(getDataentryLSB()) + "]";
        }
        
        switch (_extype) {
            case MXMessage.EXTYPE_DUMMY:
                return "ZERO";
            case MXMessage.EXTYPE_INIT_ERROR:
                return "ERROR";
            case MXMessage.EXTYPE_META:
                return "Meta " + getMetaType();
            case MXMessage.EXTYPE_SYSEX:
                return "Sysex [" + MXUtil.dumpHexFF(getDataBytes()) + "]";
            case MXMessage.EXTYPE_SYSEX_SPECIAL:
                return "SysexSpecial [" + MXUtil.dumpHexFF(getDataBytes()) + "]";
            case MXMessage.EXTYPE_SHORTMESSAGE:
                break;
            case MXMessage.EXTYPE_PROGRAM_INC:
                return "Program UP Ch:" + (this.getChannel()+1);
            case MXMessage.EXTYPE_PROGRAM_DEC:
                return "Program DEC Ch:" + (this.getChannel()+1);
        }
    
        String chname;
        if (isMessageTypeChannel()) {
            int channel = getChannel();
            chname = "[" + MXUtilMidi.nameOfPort(port) + (channel+1) +"]";
        }else {
            chname = "[" + MXUtilMidi.nameOfPort(port) + "]";
        }

        int command = getCommand();
        
        String name = MXUtilMidi.nameOfMessage(getStatus(), getData1FromBytes(), getData2FromBytes());

        if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            return chname + " " + name;
        }else {
            if (command == MXMidi.COMMAND_NOTEOFF
             || command == MXMidi.COMMAND_NOTEON
             || command == MXMidi.COMMAND_POLYPRESSURE) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + " " + name + MXUtilMidi.nameOfNote(note);
            }
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                return chname + " " + name;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                return chname + " " + name;
            }
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                return chname + " " + name;
            }
            if (command == MXMidi.STATUS_SONGPOSITION) {
                return name;
            }
            if (command == MXMidi.STATUS_SONGSELECT) {
                return name;
            }
        }

        if (command >= 0xf0 && command <= 0xf7) {
            return name;
        }

        String extype = "";
        switch(_extype) {
            case EXTYPE_DUMMY:
                extype = "-";
                break;
            case EXTYPE_SHORTMESSAGE:
                extype = "ShortMessage";
                break;
            case EXTYPE_PROGRAM_INC:
                extype = "ProgINC";
                break;
            case EXTYPE_PROGRAM_DEC:
                extype = "ProgDEC";
                break;
            case EXTYPE_SYSEX:
                extype = "SysEX";
                break;
            case EXTYPE_SYSEX_SPECIAL:
                extype = "SysEXSpecial";
                break;
            case EXTYPE_META:
                extype = "Meta(" + MXUtil.toHexFF(getMetaType()) + ")";
                break;
        }
        return "type[" + extype + ":" + MXUtil.dumpHexFF(getDataBytes()) + "]";
    }

    public boolean hasValueHiField() {
        makeSomeCached();
        return _cachedHasHi;
    }

    public boolean hasValueLowField() {
        makeSomeCached();
        return _cachedHasValue;
    }

    public boolean hasGateLowField() {
        makeSomeCached();
        return _cachedHasGate;
    }

    public boolean hasGateHiField() {
        makeSomeCached();
        return _cachedHasGateHi;
    }
    
    public static String fromD(int dtext) {
        int index = textAlias.indexOfValue(dtext);
        if (index >= 0) {
            return textAlias.get(index).name;
        }
        return MXUtil.toHexFF(dtext) + "h";
    }

    public ArrayList<String> toDArray() {
        ArrayList<String> texts = new ArrayList();

        if (_extype == EXTYPE_PROGRAM_INC) {
            texts.add(EXCOMMAND_PROGRAM_INC);
            return texts;
        }
        if (_extype == EXTYPE_PROGRAM_DEC) {
            texts.add(EXCOMMAND_PROGRAM_DEC);
            return texts;
        }
        //setPort(getPort());//make cache dirty
        makeSomeCached();
        
        if (_template[0] == DTEXT_RPN || _template[0] == DTEXT_NRPN) {
            if (_template[0] == DTEXT_RPN) {
                texts.add("@RPN");
            }else {
                texts.add("@NRPN");
            }
            texts.add(fromD(getDataentryMSB()));
            texts.add(fromD(getDataentryLSB()));
            texts.add(fromD(_template[1]));
            if (_template[2] != DTEXT_NONE) {
                texts.add(fromD(_template[2]));
            }
            return texts;
        }
        
        if (isMessageTypeChannel()) {
            int command = getCommand();
            int channel = getChannel();
            int data1 = getData1FromBytes();
            int data2 = getData2FromBytes();
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                texts.add("@PB");
                texts.add(fromD(_template[1]));
                texts.add(fromD(_template[2]));
                return texts;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                texts.add("@CP");
                texts.add(fromD(_template[1]));
                return texts;
            }
            if (command == MXMidi.COMMAND_POLYPRESSURE) {
                texts.add("@PKP");
                texts.add(fromD(_template[1]));
                texts.add(fromD(_template[2]));
                return texts;
            }
            if (command == MXMidi.COMMAND_CONTROLCHANGE) {
                texts.add("@CC");
                texts.add(fromD(_template[1]));
                texts.add(fromD(_template[2]));
                return texts;
            }
            /*
                @RPN [RPN MSB] [RPN LSB] [Data MSB] [Data LSB] 	RPNを送信します。
                @NRPN [NRPN MSB] [NRPN LSB] [Data MSB] [Data LSB] 	NRPNを送信します。 
            */
        }
        
        int csumFrom = -1;
        if (_checksumLength >= 0) {
            for (int i = 0; i < _template.length; ++ i) {
                if (_template[i] == MXMessage.DTEXT_CHECKSUM) {
                    csumFrom = i -_checksumLength;
                    break;
                 }
            }
        }
        
        for (int i = 0; i < _template.length; ++ i) {
            if (i == csumFrom) {
                texts.add("[");
            }
            int code = _template[i];
            if (i == 0 && isMessageTypeChannel()) {
                code &= 0xf0;
            }
            if (code == MXMessage.DTEXT_CHECKSUM) {
                texts.add("]");
                continue;
            }
            texts.add(fromD(code));
        }
        return texts;
    }
    
    public String toShortString() {
        makeSomeCached();
        
        String chname;
        if (isMessageTypeChannel()) {
            int channel = getChannel();
            chname = "" + (channel+1);
        }else {
            chname = "";
        }

        switch(_extype) {
            case EXTYPE_DUMMY:
                return "-";
            case EXTYPE_PROGRAM_INC:
                return chname + "Prog+";
            case EXTYPE_PROGRAM_DEC:
                return chname + "Prog-";
            case EXTYPE_SYSEX:
                return "Sys";
            case EXTYPE_SYSEX_SPECIAL:
                return "Sys2";
            case EXTYPE_META:
                return "Meta";
            case EXTYPE_SHORTMESSAGE:
                //OK;
                break;
            default:
                return "???";
        }
        
        switch (getDataentryType()) {
            case MXMidi.DATAENTRY_TYPE_RPN:
                return "RPN ";
            case MXMidi.DATAENTRY_TYPE_NRPN:
                return "NRPN ";
        }
    
        int command = getCommand();
        
        if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            int data1 = getGate();
            return chname + MXUtilMidi.nameOfControlChange(data1);
        }else {
            if (command == MXMidi.COMMAND_NOTEOFF) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + MXUtilMidi.nameOfNote(note) + "-";
            }
            if (command == MXMidi.COMMAND_NOTEON) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + MXUtilMidi.nameOfNote(note);
            }
            if (command == MXMidi.COMMAND_POLYPRESSURE) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + "PPrs";
            }
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                int program = getGate();
                return  chname + "PG" + program;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                return  chname + "ChPrs";
            }
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                return  chname + "Pitch";
            }
            if (command == MXMidi.STATUS_SONGPOSITION) {
                return  chname + "Pos";
            }
            if (command == MXMidi.STATUS_SONGSELECT) {
                return  chname + "Song";
            }
        }

        if (isMessageTypeChannel()) {
            return  chname + MXUtilMidi.nameOfMessage(getStatus(), getData1FromBytes(), getData2FromBytes());
        }else {
            return  MXUtilMidi.nameOfMessage(getStatus(), getData1FromBytes(), getData2FromBytes());
        }
    }
    
    public static void main(String[] args) {
        MXMessage message = MXMessageFactory.fromShortMessage(0, MXMidi.COMMAND_NOTEON + 0, 64, 127);
        String dtext = MXMessageFactory.toDText(message);
        MXMessage msg = MXMessageFactory.fromDtext(dtext, message.getPort(), message.getChannel(), message.getGate(), message.getValue());
        PrintStream console = System.out;
        
        console.println(message);
        console.println(dtext);
        console.println(msg);
        
        msg.setPort(6);
        msg.setChannel(2);
        msg.setValue(100);

        console.println(msg);
        console.println("----------------");

        MXMessage message2 = MXMessageFactory.fromShortMessage(0, MXMidi.COMMAND_CONTROLCHANGE + 1, MXMidi.DATA1_CC_CHANNEL_VOLUME, 127);
        console.println(message2);
        String dtext2 = MXMessageFactory.toDText(message2);
        MXMessage msg2 = MXMessageFactory.fromDtext(dtext2, message2.getPort(), message2.getChannel(), message2.getGate(), message2.getValue());

        console.println(dtext2);
        console.println(msg2);
        
        msg2.setPort(6);
        msg2.setChannel(2);
        msg2.setValue(100);

        console.println(msg2);
        console.println("----------------");
    }
    
    public void debugDump(String func) {
        PrintStream console = System.out;
        StringBuffer buf = new StringBuffer();
        buf.append(func + " debugDump [template = ");
        for (int i = 0; i < _template.length; ++ i) {
            buf.append(Integer.toHexString(_template[i]) + " ");
        }
        buf.append("] bytes = [ ");
        byte[] b = getDataBytes();
        for (int i = 0; i < b.length; ++ i) {
            buf.append(MXUtil.toHexFF(b[i]) + " ");
        }
        buf.append("]");
        buf.append(" gate = " + getGate());
        buf.append(" value = " + getValue());
        console.println(buf);
    }
    
    public String toDumpString() {
        makeSomeCached();
        
        String chname;
        if (isMessageTypeChannel()) {
            int channel = getChannel();
            chname = "" + (channel+1);
        }else {
            chname = "";
        }

        switch(_extype) {
            case EXTYPE_DUMMY:
                return "-";
            case EXTYPE_PROGRAM_INC:
                return chname + "Prog+";
            case EXTYPE_PROGRAM_DEC:
                return chname + "Prog-";
            case EXTYPE_SYSEX:
            case EXTYPE_SYSEX_SPECIAL:
            case EXTYPE_SHORTMESSAGE:
                return MXUtil.dumpHexFF(_dataBytes);
            case EXTYPE_META:
                char[] ch = new char[_template.length];
                for (int i = 0; i < ch.length; ++ i) {
                    ch[i] = (char)_template[i];
                }
                return new String(ch);
            default:
                return "???";
        }
    }
    
    public void recalcGate() {
        if (getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            if (_template[1] != DTEXT_VL) {
                int gate = _template[1];
                _template[1] = DTEXT_VL;
                setGate(gate);
            }
        }
    }

    public static int readAliasText(String str) {
        int code = -1;
        if (str.startsWith("#")) {
            int find = MXMessage.textAlias.indexOfName(str);
            if (find >= 0) {
                code = MXMessage.textAlias.get(find).value.intValue();
                return code;
            }
        }
        return MXUtil.parseTextForNumber(str);
    }
}

