/*
 * Copyright (C) 2022 YOSHIDA Shintarou
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.console;

import java.time.format.DateTimeFormatter;
import javax.swing.event.ListDataListener;
import java.util.ArrayList;
import javax.swing.JList;
import javax.swing.event.ListDataEvent;
import jp.synthtarou.midimixer.libs.MultiThreadQueue;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.libs.MXTimer;

/**
 *
 * @author YOSHIDA Shintarou
 */

public class MXConsoleModel<T> implements javax.swing.ListModel {
    private T[] listLogging = null;
    private int indexFrom = 0;
    private int indexTo = 0;
    private JList bind = null;
    private boolean _pause = false;
    private MXConsoleMidiRenderer _render;

    private ArrayList<ListDataListener> listListner = new ArrayList<>();
    private MultiThreadQueue<T> _queue;
    private String _name;

    public void setPause(boolean pause) {
        _pause = pause;
    }
    
    public synchronized void clearText() {
        indexFrom = 0;
        indexTo = 0;
    }
    
    public MXConsoleModel(String name, int maxRowCount) {
        listLogging = (T[])new Object[maxRowCount];
        _name = name;

        _queue = new MultiThreadQueue<T>();
        new Thread(new Runnable() {
            public  void run() {
                while(true) {
                    T text = _queue.pop();
                    if (text == null) {
                       _queue.quit();
                       break;
                    }
                    addTextInThread(text);
                }
            }
        }).start();
    }
    
    public MXConsoleMidiRenderer getRenderer() {
        if (bind == null) {
            throw new IllegalStateException("try installConsole before getRender");
        }
        return _render;
    }
    
    public void installConsole(JList list) {
        bind = list;
        bind.setModel(this);
        _render = new MXConsoleMidiRenderer(_name);
        list.setCellRenderer(_render);
    }

    @Override
    public int getSize() {
        int count = indexTo - indexFrom;
        return (count >= 0) ? count : (count + listLogging.length);
    }

    @Override
    public Object getElementAt(int index) {
        int pos = indexFrom + index;
        while (pos >= listLogging.length) {
            pos -= listLogging.length;
        }
        return listLogging[pos];
    }
    
    static byte[] _hex = "0123456789ABCDEF".getBytes();

    public void addConsoleMessage(T msg)   {
        if (_pause) {
            return;
        }
        _queue.push(msg);
    }

    DateTimeFormatter simpleFormat = DateTimeFormatter.ofPattern("HH:mm:ss.SSS");
    //DateTimeFormatter simpleFormat = DateTimeFormatter.ofPattern("HH:mm");

    private void addTextInThread(final T message) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    addTextInThread(message);
                }
            });
            return;
        }
        boolean eventRemoved = false;
        boolean eventAdded = false;
        
        listLogging[indexTo++] = message;
        if (indexTo >= listLogging.length) {
            indexTo -= listLogging.length;
        }
        eventAdded = true;

        if (indexTo == indexFrom) {
            listLogging[indexFrom++] = null;
            if (indexFrom >= listLogging.length) {
                indexFrom -= listLogging.length;
            }
            eventRemoved = true;
        }

        synchronized(this) {
            if (eventRemoved) {
                final ListDataEvent e = new ListDataEvent(this, ListDataEvent.INTERVAL_REMOVED, 0, 0);
                for (ListDataListener listener : listListner) {
                    listener.intervalRemoved(e);
                }
            }
            if (eventAdded) {
                int size = MXConsoleModel.this.getSize();
                ListDataEvent e = new ListDataEvent(this, ListDataEvent.INTERVAL_ADDED, size - 1, size - 1);
                for (ListDataListener listener : listListner) {
                    listener.intervalAdded(e);
                }
            }

            if (bind != null && eventRemoved == false) {
                bind.ensureIndexIsVisible(bind.getModel().getSize() -1);
            }
        }
    }

    @Override
    public synchronized void addListDataListener(ListDataListener l) {
        listListner.add(l);
    }

    @Override
    public synchronized void removeListDataListener(ListDataListener l) {
        listListner.remove(l);
    }

    boolean reserved = false;
    long lastTick = 0;

    public synchronized  void fireRepaint() {
        long tickNow = System.currentTimeMillis();
        if (tickNow - lastTick < 1000) {
            if (reserved) {
                return;
            }
            reserved = true;
            MXTimer.letsCountdown(1000 -tickNow - lastTick, new Runnable() {
                @Override
                public void run() {
                    fireImpl();
                }
            });
        }else {
            reserved = true;
            fireImpl();
        }
    }

    protected synchronized  void fireImpl() {
        if (reserved) {
            lastTick = System.currentTimeMillis();
            reserved = false;
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    final ListDataEvent e = new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, getSize());
                    for (ListDataListener listener : listListner) {
                        listener.contentsChanged(e);
                    }
                }
            });
        }
    }
}
