/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the FFree Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.LinkedList;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.capture.MXMessageCapture;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.port.FinalMIDIOut;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIIn;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIInManager;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIOutManager;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.swing.themes.ThemeManager;
import jp.synthtarou.midimixer.mx00playlist.MX00Process;
import jp.synthtarou.midimixer.mx10input.MX10Process;
import jp.synthtarou.midimixer.mx12masterkeys.MX12Process;
import jp.synthtarou.midimixer.mx30controller.MX30Process;
import jp.synthtarou.midimixer.mx40layer.MX40Process;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfoReceiver;
import jp.synthtarou.midimixer.mx60output.MX60Process;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMain {    private static MXMain _main = new MXMain(); 

    public static MXMain getMain() {
        return _main;
    }
    
    private MXMainWindow _mainWindow;

    public MXMainWindow getMainWindow() {
        return _mainWindow;
    }
    
    public MX12Process getMasterKeys() {
        return _mx12masterkeyProcess;
    }

    private MX10Process _mx10inputProcess;
    private MX12Process _mx12masterkeyProcess;
    private MX00Process _mx00playlistProcess;
    private MX30Process _mx30kontrolProcess;
    private MX40Process _mx40layerProcess;
    private MX60Process _mx60outputProcess;
    
    public static MXMessageCapture _capture = null;
    
    public static void setCapture(MXMessageCapture capture) {
        _capture = capture;
    }
    
    public void startMainPath(MXMIDIIn input, MXMessage message) {
        synchronized(this) {
            if (_capture != null) {
                _capture.process(message);
            }
            if (_mx12masterkeyProcess == null) {
                return;
            }
            if (_mx12masterkeyProcess.isAcceptInputPanelSignal()) {
                if (message.isMessageTypeChannel()) {
                    if (_mx12masterkeyProcess.isAcceptInputPanelSignal()) {
                        if (input.isToMaster(message.getChannel())) {
                            _mx12masterkeyProcess.processMXMessage(message);
                            return;
                        }
                    }
                    if (_mx12masterkeyProcess.isAcceptThisPageSignal()) {
                        if (message.getPort() == _mx12masterkeyProcess.getMousePort()
                         && message.getChannel() == _mx12masterkeyProcess.getMouseChannel()) {
                            _mx12masterkeyProcess.processMXMessage(message);
                            return;
                        }
                    }
                }
            }
            if (_mx10inputProcess != null) {
                _mx10inputProcess.processMXMessage(message);
                return;
            }
        }
    }

    /**
     * アプリを起動する
     * @param args 引数
     * @throws Exception エラー通知
     */
    public static void main(String[] args) throws Exception {
        try {
            MXDebugConsole.initDebugConsole(args);

            //フォント描写でアンチエイリアスを有効にする
            System.setProperty("awt.useSystemAAFontSettings", "on");
            
            ThemeManager inst = ThemeManager.getInstance();
            /*        
            try {
                LaunchLoopMidi midiout = new LaunchLoopMidi();
            }catch( Throwable e ) {
                e.printStackTrace();
            }
            LookAndFeelPanel.setLookAndFeel(LookAndFeelConfig.THEME_OS_STANDARD);
            LookAndFeelPanel.updateLookAndFeelRecursive();
            */
            Runtime.getRuntime().addShutdownHook(new Thread(){
                @Override
                public void run() {
                }
            });
        
        } catch (Throwable e) {
            e.printStackTrace();
        }

        getMain().startUI();
    }
    
    public void startUI()  {
        _mainWindow = new MXMainWindow(this);

        MXOpening winLogo = MXOpening.showAsStartup(_mainWindow);        
        winLogo.setVisible(true);

        MXMIDIInManager.getManager().initWithSetting();
        MXMIDIOutManager.getManager().initWithSetting();

        winLogo.showProgress(0, 10);

        _mx00playlistProcess = new MX00Process();
        _mx10inputProcess = new MX10Process();
        _mx12masterkeyProcess = new MX12Process();
        winLogo.showProgress(1, 10);
        _mx30kontrolProcess =  new MX30Process();
        _mx40layerProcess = new MX40Process();
        winLogo.showProgress(2, 10);

        _mx60outputProcess = new MX60Process();
        _masterToList = new MXWrapList();

        winLogo.showProgress(3, 10);

        _masterToList.addNameAndValue("PushBack to " + _mx10inputProcess.getReceiverName(), _mx10inputProcess);
        _masterToList.addNameAndValue(_mx30kontrolProcess.getReceiverName(), _mx30kontrolProcess);
        _masterToList.addNameAndValue(_mx40layerProcess.getReceiverName(), _mx40layerProcess);
        _masterToList.addNameAndValue(_mx60outputProcess.getReceiverName(), _mx60outputProcess);
        _masterToList.addNameAndValue("Direct Output", FinalMIDIOut.getInstance());

        _mx10inputProcess.setNextReceiver(_mx30kontrolProcess);
        _mx10inputProcess.setMasterKeyProcess(_mx12masterkeyProcess);
        winLogo.showProgress(4, 10);

        _mx12masterkeyProcess.setNextReceiver(_mx10inputProcess);

        _mx30kontrolProcess.setNextReceiver(_mx40layerProcess);
        _mx40layerProcess.setNextReceiver(_mx60outputProcess);
        _mx60outputProcess.setNextReceiver(FinalMIDIOut.getInstance());

        winLogo.showProgress(5, 10);

        _mx30kontrolProcess.readSettings();
        _mx12masterkeyProcess.readSettings();
        _mx00playlistProcess.readSettings();
        winLogo.showProgress(6, 10);

        _mx10inputProcess.readSettings();
        _mx60outputProcess.readSettings();
        _mx40layerProcess.readSettings();

        winLogo.showProgress(7, 10);
        _mainWindow.setEnabled(false);
        _mainWindow.addWindowListener(new WindowAdapter(){
            public void windowClosing(WindowEvent e) {
                MXUtil.closeOwnerWindow(getMasterKeys().getReceiverView());
            }
            public void windowClosed(WindowEvent e){
                MXSetting.saveEverySettingToFile();
                MXMIDIInManager.getManager().closeAll();
                MXMIDIOutManager.getManager().closeAll();
                System.exit(0);
            }
        });

        winLogo.showProgress(8, 10);
        ArrayList<MXReceiver> reList = new ArrayList();
        reList.add(_mx00playlistProcess);
        reList.add(_mx10inputProcess);
        //reList.add(_velocityProcess);
        reList.add(_mx30kontrolProcess);
        reList.add(_mx40layerProcess);
        reList.add(_mx60outputProcess);

        _mainWindow.initLatebind(reList);
        _mainWindow.setVisible(true);


        _mainWindow.setEnabled(true);
        winLogo.showProgress(9, 10);

        _mx12masterkeyProcess.createView().showAsWindow();

        winLogo.showProgress(10, 10);
        winLogo.setVisible(false);
        
        Runnable run;
        while((run = getNextLaunchSequence()) != null) {
            run.run();
        }
    }
    
    LinkedList<Runnable> _startQueue = new LinkedList();
    
    public void addLaunchSequence(Runnable run) {
        synchronized(this) {
            _startQueue.add(run);
        }
    }
    
    public Runnable getNextLaunchSequence() {
        synchronized(this) {
            if (_startQueue.isEmpty()) {
                return null;
            }
            return _startQueue.removeFirst();
        }
    }

    private MXWrapList<MXReceiver> _masterToList = null;
    
    public MXWrapList<MXReceiver> getReceiverList() {
        MXWrapList<MXReceiver> list = new MXWrapList();
        list.addAll(_masterToList);
        return list;
    }

    public void saveEverySettingToFile() {
        MXSetting.saveEverySettingToFile();
    }
    
    public void exitApp() {
        // exit with event
        _mainWindow.dispatchEvent(new WindowEvent(_mainWindow, WindowEvent.WINDOW_CLOSING));
        /* following codes are unsafety
            _mainWindow.dispose();
            _mainWindow.setVisible(false);
            System.exit(0);
        */
    }
    
    public MXReceiver getInputProcess() {
        return _mx10inputProcess;
    }
    
    public void sequencerStart() {
        _mx40layerProcess.resendProgramChange();
    }
    
    public MXChannelInfoReceiver get40InputInfo() {
        return _mx40layerProcess.getInputInfo();
    }
}
