/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx40layer;

import java.util.ArrayList;
import java.util.List;
import javax.swing.JComponent;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.port.FinalMIDIOut;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingNode;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX40Process extends MXReceiver implements MXSettingTarget {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX40Process.class);

    MX40View _view;
    MXChannelInfoReceiver _inputInfo;
    MXChannelInfoReceiver _outputInfo;
    MXNoteOffWatcher _noteOff;
    MXSetting _setting;
    ArrayList<MX40Group> _groupList;

    public MX40Process() {
        _groupList = new ArrayList();
        _inputInfo = new MXChannelInfoReceiver();
        _outputInfo = FinalMIDIOut.getInstance().getOutputInfo();
        _noteOff = new MXNoteOffWatcher();
        _view = new MX40View(this);
        _setting = new MXSetting("SoundLayer");
        _setting.setTarget(this);
    }
    
    public void readSettings() {
        _setting.readFile();
        //_view.reload
    }

    public void processMXMessage(MXMessage message) {
        if (usingThis() == false) { 
            _inputInfo.processMXMessage(message);
            sendToNext(message); 
            return; 
        }

        if (message._extype == MXMessage.EXTYPE_PROGRAM_INC) {
            message = _inputInfo.incProg(message.getPort(), message.getChannel());
            sendToNext(message);
            return;
        }else if (message._extype == MXMessage.EXTYPE_PROGRAM_DEC) {
            message = _inputInfo.decProg(message.getPort(), message.getChannel());
            sendToNext(message);
            return;
        }else if (message._extype != MXMessage.EXTYPE_SHORTMESSAGE) {
            sendToNext(message);
            return;
        }
        _inputInfo.processMXMessage(message);

        int port = message.getPort();
        int command = message.getCommand();
        int channel = message.getChannel();

        if (command == MXMidi.COMMAND_NOTEOFF) {
            _noteOff.notifyNoteOffEvent(port, channel, message.getNoteNumberFromBytes(), "@0");
            return;
        }
        
        boolean dispatched = false;

        if (message.isMessageTypeChannel()) {
            _inputInfo.tryPickup(message);
            for (int i = 0; i < _groupList.size(); ++ i) {
                final MX40Group col = _groupList.get(i);
                if (col.processByGroup(message)) {
                    if (command == MXMidi.COMMAND_NOTEON) {
                        _noteOff.addListener(message, message, new MXNoteOffWatcher.Handler() {
                            public void onNoteOffEvent(MXMessage target) {
                                MXMessage message = MXMessageFactory.fromShortMessage(
                                        target.getPort(), 
                                        MXMidi.COMMAND_NOTEOFF + target.getChannel(), 
                                        target.getNoteNumberFromBytes(), 0);
                                col.processByGroup(message);
                            }
                        }, "@0");
                    }
                    dispatched = true;
                    break;
                }
            }
        }
        if (!dispatched) {
            if (command == MXMidi.COMMAND_NOTEON) {
                _noteOff.addListener(message, message, new MXNoteOffWatcher.Handler() {
                    public void onNoteOffEvent(MXMessage target) {
                        MXMessage message = MXMessageFactory.fromShortMessage(
                                target.getPort(), 
                                MXMidi.COMMAND_NOTEOFF + target.getChannel(), 
                                target.getNoteNumberFromBytes(), 0);
                        sendToNext(message);
                    }
                }, "@0");
            }
            sendToNext(message);
        }
    }

    @Override
    public String getReceiverName() {
        return "Sound Layer";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }

    public synchronized void resendProgramChange(MX40Group group) {
        for(int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            for(int channel = 0; channel < 16; ++ channel) {
                MXChannelInfo info = _inputInfo.getChannelInfo(port, channel);
                MX40Group hit = null;
                if (info._havingProgram) {
                    MXMessage programChange = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, info._program, 0);
                    processMXMessage(programChange);
                }else if (info._havingBank) {
                    MXMessage programChange = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, info._program, 0);
                    processMXMessage(programChange);
                }
            }
        }
    }
    
    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("Group[].title");
        setting.register("Group[].isWatchPort");
        setting.register("Group[].watchingPort");
        setting.register("Group[].isWatchChannel");
        setting.register("Group[].watchingCahnnel");
        setting.register("Group[].isWatchBank");
        setting.register("Group[].watchingBankMSB");
        setting.register("Group[].watchingBankLSB");
        setting.register("Group[].isWatchProgram");
        setting.register("Group[].watchingProgram");

        setting.register("Group[].Layer");
        setting.register("Group[].Layer[]");
        setting.register("Group[].Layer[].title");
        setting.register("Group[].Layer[].disabled");
        setting.register("Group[].Layer[].modPort");
        setting.register("Group[].Layer[].fixedPort");
        setting.register("Group[].Layer[].modChannel");
        setting.register("Group[].Layer[].fixedChannel");
        setting.register("Group[].Layer[].modBank");
        setting.register("Group[].Layer[].fixedBankMSB");
        setting.register("Group[].Layer[].fixedBankLSB");
        setting.register("Group[].Layer[].modProgram");
        setting.register("Group[].Layer[].fixedProgram");
        setting.register("Group[].Layer[].adjustPan");
        setting.register("Group[].Layer[].adjustExpression");
        setting.register("Group[].Layer[].adjustTranspose");
        setting.register("Group[].Layer[].adjustVelocity");
        setting.register("Group[].Layer[].sendKeyLowest");
        setting.register("Group[].Layer[].sendKeyHighest");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        ArrayList<MX40Group> newGroupList = new ArrayList();
        this._groupList = newGroupList;
        List<MXSettingNode> readingGroups = setting.findByPath("Group");   
        if (readingGroups.size() > 0) {
            readingGroups = readingGroups.get(0).findNumbers();
        }
        for (MXSettingNode node : readingGroups) {
            MX40Group group = new MX40Group(this);
            group._title = node.findNode("title")._value;

            group._isWatchPort = node.getSettingAsBoolean("isWatchPort", false);
            group._watchingPort = node.getSettingAsInt("watchingPort", 0);
            group._isWatchChannel = node.getSettingAsBoolean("isWatchChannel", false);
            group._watchingChannel = node.getSettingAsInt("watchingCahnnel", 0);

            group._isWatchBank = node.getSettingAsBoolean("isWatchBank", false);
            group._watchingBankMSB = node.getSettingAsInt("watchingBankMSB",0 );
            group._watchingBankLSB = node.getSettingAsInt("watchingBankLSB",0 );

            group._isWatchProgram = node.getSettingAsBoolean("isWatchProgram", false);
            group._watchingProgram = node.getSettingAsInt("watchingProgram", 0);
            newGroupList.add(group);

            MXSettingNode layerNode = node.findNode("Layer");
            //_debug.println("layerNode = " + layerNode);
            if (layerNode != null) {
                List<MXSettingNode> numbers = layerNode.findNumbers();
                for (MXSettingNode node2 : numbers) {
                    //_debug.println("node2 = " + node2);
                    MX40Layer layer = new MX40Layer(this, group);
                    layer._title = node2.getSetting("title");
                    layer._disabled  = node2.getSettingAsBoolean("disabled", false);
                    layer._modPort = node2.getSettingAsInt("modPort", MX40Layer.MOD_ASFROM);
                    layer._fixedPort = node2.getSettingAsInt("fixedPort", 0);

                    layer._modChannel = node2.getSettingAsInt("modChannel", MX40Layer.MOD_ASFROM);
                    layer._fixedChannel = node2.getSettingAsInt("fixedChannel", 0);

                    layer._modBank = node2.getSettingAsInt("modBank", MX40Layer.MOD_ASFROM);
                    layer._fixedBankMSB = node2.getSettingAsInt("fixedBankMSB", 0);
                    layer._fixedBankLSB = node2.getSettingAsInt("fixedBankLSB", 0);

                    layer._modProgram = node2.getSettingAsInt("modProgram", MX40Layer.MOD_ASFROM);
                    layer._fixedProgram = node2.getSettingAsInt("fixedProgram", 0);

                    layer._adjustPan = node2.getSettingAsInt("adjustPan", 0);
                    layer._adjustExpression = node2.getSettingAsInt("adjustExpression", 0);
                    layer._adjustTranspose = node2.getSettingAsInt("adjustTranspose", 0);
                    layer._adjustVelocity = node2.getSettingAsInt("adjustVelocity", 0);
                    layer._acceptKeyLowest = node2.getSettingAsInt("sendKeyLowest", 0);
                    layer._acceptKeyHighest = node2.getSettingAsInt("sendKeyHighest", 0);

                    group._listLayer.add(layer);
                }
            }
        }
        _view.listUpGroups();
        if (newGroupList.size() > 0) {
            _view.startEditGroup(newGroupList.get(0));
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        for (int i = 0; i < _groupList.size(); i ++){
            String prefixG = "Group[" + i + "]";
            MX40Group group = _groupList.get(i);
            setting.setSetting(prefixG + ".title", group._title);
            setting.setSetting(prefixG + ".isWatchPort", group._isWatchPort);
            setting.setSetting(prefixG + ".watchingPort", group._watchingPort);
            setting.setSetting(prefixG + ".isWatchChannel", group._isWatchChannel);
            setting.setSetting(prefixG + ".watchingCahnnel", group._watchingChannel);
            setting.setSetting(prefixG + ".isWatchBank", group._isWatchBank);
            setting.setSetting(prefixG + ".watchingBankMSB", group._watchingBankMSB);
            setting.setSetting(prefixG + ".watchingBankLSB", group._watchingBankLSB);
            setting.setSetting(prefixG + ".isWatchProgram", group._isWatchProgram);
            setting.setSetting(prefixG + ".watchingProgram", group._watchingProgram);
            for (int j = 0; j < group._listLayer.size(); ++ j) {
                String prefixL = "Group[" + i + "].Layer[" + j + "]";
                MX40Layer layer = group._listLayer.get(j);
                setting.setSetting(prefixL + ".title", layer._title);
                setting.setSetting(prefixL + ".disabled", layer._disabled);
                setting.setSetting(prefixL + ".modPort", layer._modPort);
                setting.setSetting(prefixL + ".fixedPort", layer._fixedPort);
                setting.setSetting(prefixL + ".modChannel", layer._modChannel);
                setting.setSetting(prefixL + ".fixedChannel", layer._fixedChannel);
                setting.setSetting(prefixL + ".modBank", layer._modBank);
                setting.setSetting(prefixL + ".fixedBankMSB", layer._fixedBankMSB);
                setting.setSetting(prefixL + ".fixedBankLSB", layer._fixedBankLSB);
                setting.setSetting(prefixL + ".modProgram", layer._modProgram);
                setting.setSetting(prefixL + ".fixedProgram",layer._fixedProgram);
                setting.setSetting(prefixL + ".adjustPan", layer._adjustPan);
                setting.setSetting(prefixL + ".adjustExpression", layer._adjustExpression);
                setting.setSetting(prefixL + ".adjustTranspose", layer._adjustTranspose);
                setting.setSetting(prefixL + ".adjustVelocity", layer._adjustVelocity);
                setting.setSetting(prefixL + ".sendKeyLowest", layer._acceptKeyLowest);
                setting.setSetting(prefixL + ".sendKeyHighest", layer._acceptKeyHighest);
            }
        }
    }
}
