/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller.ui;

import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXUIValue {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXUIValue.class);

    public static final int TYPE_CIRCLE = 1;
    public static final int TYPE_DRUM = 3;
    public static final int TYPE_SLIDER = 2;

    public final int _controllerType;
    public final int _controllerRow;
    public final int _controllerColumn;

    private MXMessage _monitor;
    public String _name;
    public String _note;
    
    public int _uiValue = 0;
    public int _uiValueMinimum = 0;
    public int _uiValueMaximum = 0;
    public boolean _uiValueInvert = false;
    public int _uiValueDefault = 0;    
    public int _uiValueLastSent = -1;

    private int _monitorSP = SP_NONE;

    public static final int SP_NONE = 0xff;
    public static final int SP_NOTE_ON = 0x00;
    public static final int SP_MESSAGE_VALUE = 0x10;
    public static final int SP_SYSTEM = 0x70;
    public static final int SP_PROG_UP = 0x80;
    public static final int SP_PROG_DOWN = 0x90;

    public boolean _drumCurrentPushing;
    public boolean _drumIsToggleSwitch;
    public boolean _drumLastOfToggle;

    public int _drumHitVelocity;

    public MXUIValue(int uiType, int row, int column) {
        _controllerType = uiType;
        _controllerRow = row;
        _controllerColumn = column;

        _drumCurrentPushing = false;
        _drumIsToggleSwitch = false;
        _drumLastOfToggle = false;

        _drumHitVelocity = 100;
    }
    
    public void setMonitoring(MXMessage familly) {
        _monitor = MXMessageFactory.fromClone(familly);
        int min = 0;
        int max = 0;
        if (familly.hasValueHiField()) {
            max = 127 + 127 * 128;
            _monitorSP = MXUIValue.SP_MESSAGE_VALUE;
        }else if (familly.hasValueLowField()) {
            max = 127;
            _monitorSP = MXUIValue.SP_MESSAGE_VALUE;
        }else {
            max = 127;
            _monitorSP = MXUIValue.SP_NONE;
        }
        if (_controllerType == TYPE_DRUM) {
            if (_monitor.getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
                max = _monitor.getValue();
                min = _monitor.getValue();
            }
        }
        int def = max;
        _uiValueDefault = def;
        _uiValueMinimum = min;
        _uiValueMaximum = max;
        _name = _monitor.toShortString();
    }

    public boolean watchingChannelMessage() {
        return _monitor.isMessageTypeChannel();
    }

    public String toString() {
        MXMessage message;
        String text;
        switch (_controllerType) {
            case TYPE_CIRCLE:
                message = toMXMessage();
                text = "Circle";
                break;
            case TYPE_SLIDER:
                message = toMXMessage();
                text = "Slider";
                break;
            case TYPE_DRUM:
                boolean x = _drumCurrentPushing;
                _drumCurrentPushing = true;
                message = toMXMessage();
                _drumCurrentPushing = x;
                text = "Pad";
                break;
            default:
                message = null;
                text = "Error";
                break;
        }
        return text + "[" + (_controllerRow+1) + "," + (_controllerColumn+1) +"]\n" + (message == null ? "NONE" : message);
    }

    public boolean catchMessageValue(MXMessage message) {
        if (_monitor._extype != message._extype) {
            return false;
        }

        int port1 = _monitor.getPort();
        int command1 = _monitor.getCommand();
        int channel1 = -1;
        if (_monitor.isMessageTypeChannel()) {
            channel1 = _monitor.getChannel();
        }

        int port2 = message.getPort();
        int command2 = message.getCommand();
        int channel2 = -1;
        if (message.isMessageTypeChannel()) {
            channel2 = message.getChannel();
        }
        
        if (port1 != port2) {
            return false;
        }

        if (channel1 != channel2) {
            return false;
        }
        
        if (command1 != command2) {
            return false;
        }
        
        if (command1 == MXMidi.COMMAND_CONTROLCHANGE) {
            if (_monitor.getCCCodeFromBytes() == MXMidi.DATA1_CC_DATAENTRY) {
                int cc = message.getCCCodeFromBytes();
                if (cc == MXMidi.DATA1_CC_DATAENTRY || cc == MXMidi.DATA1_CC_DATAINC  || cc == MXMidi.DATA1_CC_DATADEC) {
                    int original = _monitor.getValue();
                    int value = message.getValue();
                    switch(cc) {
                        case MXMidi.DATA1_CC_DATAENTRY:
                            value = original;
                        case MXMidi.DATA1_CC_DATAINC:
                            value = original + 1;
                            break;
                        case MXMidi.DATA1_CC_DATADEC:
                            value = original - 1;
                            break;
                        default:
                            _debug.println("bad message");
                            return false;
                    }
                    if (value >= _uiValueMinimum && value <= _uiValueMaximum) {
                        setValue(value);
                        return true;
                    }
                }
                return false;
            }
        }

        if (_monitor.hasSameTemplateExcludeGateAndValue(message)) {
            if (_monitor.hasGateField()) {
                if (message.hasGateField()) {
                    if (_monitor.getGate() != message.getGate()) {
                        return false;
                    }
                }else {
                    return false;
                }
            }
            if (_monitor.getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
                if (_monitor.getValue() == message.getValue()) {
                    return true;
                }
                return false;
            }
            int value = message.getValue();
            if (value >= _uiValueMinimum && value <= _uiValueMaximum) {
                setValue(value);
                return true;
            }
        }
        
        return false;
    }

    public void catchDrumValue(boolean value, boolean secondLink) {
        boolean proced = false;
        if (value) {
            if (_drumIsToggleSwitch) {
                if (_drumLastOfToggle) {

                }else {
                    _drumLastOfToggle = true;
                    _drumCurrentPushing = !_drumCurrentPushing;
                }
            }else {
                if (!_drumCurrentPushing) {
                    _drumCurrentPushing = true;
                }
            }
        }else {
            if (_drumIsToggleSwitch) {
                _drumLastOfToggle = false;
            }else {
                if (_drumCurrentPushing) {
                    _drumCurrentPushing = false;
                }
            }
        }
    }

    public MXMessage toMXMessage() {
        if (_monitorSP == MXUIValue.SP_NONE) {
            return MXMessageFactory.makeDummy();
        }

        MXMessage message = MXMessageFactory.fromClone(_monitor);
        
        if (_controllerType == TYPE_DRUM) {
            if (_drumCurrentPushing) {
                if (message.getCommand() == MXMidi.COMMAND_NOTEON) {
                    setValue(_drumHitVelocity);
                }else {
                    setValue(_uiValueMaximum);
                }
            }else {
                if (message.getCommand() == MXMidi.COMMAND_NOTEON) {
                    message.setStatus(MXMidi.COMMAND_NOTEOFF + message.getChannel());
                    setValue(0);
                }else if (message.getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
                    return MXMessageFactory.makeDummy();
                }else {
                    setValue(_uiValueMinimum);
                }
            }
        }
        
        message.setValue(getValue());
        return message;
    }

    public void setPortChannelGate(int port, int channel, int gate) {
        _monitor.setPort(port);
        _monitor.setChannel(channel);
        _monitor.setGate(gate);
    }
    
    public void setValue(int value) {
        _uiValue = value;
    }
    
    public int getValue() {
        return _uiValue;
    }
}
