/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx12masterkeys;

import java.awt.Dimension;
import javax.swing.JComponent;
import javax.swing.JFrame;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX12Process extends MXReceiver implements MXSettingTarget {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX12Process.class);

    private MXSetting _setting;
    MX12MasterkeysPanel _view;
    MXNoteOffWatcher _noteOff;
    
    int _mousePort = 0;
    int _mouseChannel = 0;
    int _mouseVelocity = 100;

    boolean _overwriteControllerChannel;
    boolean _construction;
    
    public MX12Process() {
        _setting = new MXSetting("MasterKey");
        _setting.setTarget(this);
        _construction = true;
        _view = new MX12MasterkeysPanel(this);
        _noteOff = new MXNoteOffWatcher();
        _construction = false;
        _overwriteControllerChannel = false;
    }

    public void showAsWindow(JFrame parent, boolean modal) {
        JFrame dialog = new JFrame();
        dialog.setTitle("Master Keys (" + MXStatic.MX_APPNAME + ")");
        //dialog.setAlwaysOnTop(modal ? true : false);
        dialog.pack();
        dialog.getContentPane().add(this._view);
        this._view.setPreferredSize(new Dimension(1200, 170));
        dialog.pack();
        MXUtil.centerWindow(dialog);
        dialog.setVisible(true);
    }
    
    public void readSettings() {
        _setting.readFile();
        _view.updateViewForSettingChange();
    }
    
    public void setTargetStrongChange(boolean overwrite) {
        _overwriteControllerChannel = overwrite;
    }

    @Override
    public String getReceiverName() {
        return "Master Keyboard";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("outputReceiver");
        setting.register("outputPort");
        setting.register("outputChannel");
        setting.register("overwriteControllerChannel");
        setting.register("outputVelocity");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        String receiverName = setting.getSetting("outputReceiver");
        if (receiverName != null) {
            int x = MXMain.getMain().getMasterToList().indexOfName(receiverName);
            if (x >= 0) {
                setNextReceiver(MXMain.getMain().getMasterToList().get(x).value);
            }
        }
        _overwriteControllerChannel = setting.getSettingAsInt("overwriteControllerChannel", 0) != 0;        
        _mousePort = setting.getSettingAsInt("outputPort", 0);
        _mouseChannel = setting.getSettingAsInt("outputChannel", 0);
        _mouseVelocity = setting.getSettingAsInt("outputVelocity", 100);
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        setting.setSetting("outputReceiver", getNextReceiver().getReceiverName());
        setting.setSetting("outputPort", _mousePort);
        setting.setSetting("outputChannel", _mouseChannel);
        setting.setSetting("overwriteControllerChannel", _overwriteControllerChannel);
        setting.setSetting("outputVelocity", _mouseVelocity);
    }
   
    public class MyNoteOffHandler implements MXNoteOffWatcher.Handler {
        MXReceiver _receiver;
        
        public MyNoteOffHandler(MXReceiver receiver) {
            _receiver = receiver;
        }

        @Override
        public void onNoteOffEvent(MXMessage target) {
            MXMessage makeOff = MXMessageFactory.fromShortMessage(
                    target.getPort(), 
                    MXMidi.COMMAND_NOTEOFF | target.getChannel(), 
                    target.getNoteNumberFromBytes(),
                    0);
            _receiver.processMXMessage(makeOff);
            _view._piano.noteOff(target.getNoteNumberFromBytes());
        }
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (usingThis() == false) { sendToNext(message); return; }
        
        if (message.isMessageTypeChannel()) {
            int port = message.getPort();
            int ch = message.getChannel();
            int command = message.getCommand();
            int data1 = message.getData1FromBytes();
            int data2 = message.getData2FromBytes();

            if (command == MXMidi.COMMAND_NOTEON && data2 == 0) {
                command = MXMidi.COMMAND_NOTEOFF;
            }

            if (command == MXMidi.COMMAND_NOTEOFF) {
               if (_noteOff.notifyNoteOffEvent(port, ch, data1, "@3")) {
                    //done
                    return;
                }
            }

            MXMessage newMessage = null;
            if (_overwriteControllerChannel) {
                newMessage = MXMessageFactory.fromClone(message);
                newMessage.setPort(_mousePort);
                if (newMessage.isMessageTypeChannel()) {
                    newMessage.setChannel(_mouseChannel);
                }
            }

            if (command == MXMidi.COMMAND_NOTEON) {
                _view._piano.noteOn(data1);
                if (newMessage != null) {
                    _noteOff.addListener(message, newMessage, new MyNoteOffHandler(getNextReceiver()));
                }else {
                    _noteOff.addListener(message, message, new MyNoteOffHandler(getNextReceiver()));
                }
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && data1 == MXMidi.DATA1_CCDUMPERPEDAL) {
                _view._piano.sustain(data2);
            }else if (command == MXMidi.COMMAND_PITCHWHEEL) {
                _view.setPitchBend(message.getValue());
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && data1 == MXMidi.DATA1_CCMODULATION) {
                _view.setModulatoinWheel(message.getValue());
            }
            if (newMessage != null) {
                sendToNext(newMessage);
            }else {
                sendToNext(message);
            }
            return;
        }
        sendToNext(message);
    }

    public void mouseMessage(MXMessage message) {
        if (_construction) {
            return;
        }
        sendToNext(message);
    }
}
