/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import java.util.ArrayList;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.ShortMessage;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXLineReader;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageCapture;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIIn {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMIDIIn.class);

    public static final MXMIDIIn INTERNAL_SEQUENCER = new MXMIDIIn("<PlayList>", null);
    
    public static MXMessageCapture _doCapture = null;
    
    private boolean _isopen;
    private String _name;
    private MidiDevice _device;
    private MXNoteOffWatcher _noteOff = new MXNoteOffWatcher();
    private int _assigned;
    private boolean[] _isToMaster = new boolean[16];
    
    public static void setCaptureSwitch(MXMessageCapture capture) {
        _doCapture = capture;
    }
    
    public boolean isOpen() {
        return _isopen;
    }
    
    public void pushInput(MXMessage message) {
        if (message.getCommand() == ShortMessage.TIMING_CLOCK) {
            return;
        }

        MXMessageCapture track = _doCapture;
        if (track != null) {
            track.process(message);
        }
        
        Thread t = Thread.currentThread();
        if (t != _prioritySeat) {
            _prioritySeat = t;
            if (t.getPriority() < 9) {
                try {
                    t.setPriority(9);
                }catch(Exception e) {
                    e.printStackTrace();
                }
            }
        }

        if (_assigned >= 0 && _isopen) {
            MXMain.startMainPath(this, message);
        }
    }
    
    static Thread _prioritySeat = null;

    public MXMIDIIn(String name, MidiDevice device) {
        _isopen = false;
        _name = name;
        _assigned = -1;
        _device = device;
    }
    
    public int assigned() {
        return _assigned;
    }
    
    public boolean isToMaster(int channel) {
        return _isToMaster[channel];
    }
    
    public void setToMaster(int channel, boolean toMaster) {
        _isToMaster[channel] = toMaster;
    }

    public String getMasterList() {
        StringBuffer str = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (str.length() == 0) {
                    str.append(Integer.toString(ch));
                }else {
                    str.append(",");
                    str.append(Integer.toString(ch));
                }
            }
        }
        return str.toString();
    }
    
    public void setMasterList(String text) {
        if (text == null) {
            return;
        }
        ArrayList<String> list = new ArrayList();
        MXUtil.split(text, list, ',');
        _isToMaster = new boolean[16];
        for (String x : list) {
            int ch = MXUtil.parseTextForNumber(x);
            if (ch >= 0) {
                _isToMaster[ch] = true;
            }
        }
    }
    
    /**
     * 
     * @return 
     */
    public String getName() {
        return _name;
    }

    /**
     * 
     * @return 
     */
    public String toString() {
        return _name;
    }

    public void processMidiIn(MXMessage message) {
        if (message.isMessageTypeChannel()) {
            final int channel = message.getChannel();
            if (message.getCommand() == MXMidi.COMMAND_NOTEOFF) {
                _noteOff.notifyNoteOffEvent(_assigned, channel,  message.getNoteNumberFromBytes(), "@7");
                return;
            }
            else if (message.getCommand() == MXMidi.COMMAND_NOTEON && message.getVelocityFromBytes() == 0) {
                _noteOff.notifyNoteOffEvent(_assigned, channel,  message.getNoteNumberFromBytes(), "@6");
                return;
            }
            else if (message.getCommand() == MXMidi.COMMAND_NOTEON) {
                int note = message.getNoteNumberFromBytes();
                MXMessage msg2 = MXMessageFactory.fromShortMessage(_assigned, MXMidi.COMMAND_NOTEOFF + channel, note, 0);
                _noteOff.addListener(message, msg2, new MXNoteOffWatcher.Handler() {
                    @Override
                    public void onNoteOffEvent(MXMessage target) {
                        MXMIDIIn.this.pushInput(target);
                    }
                });
            }
        }
        try {
            pushInput(message);
        }catch(Exception e) {
            e.printStackTrace();
        }
    }
    
    public void close()  {
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            allNoteOff();
            if (_isopen) {
                MXMIDIInManager.getManager().onClose(this);
                if (_device != null) {
                    _device.close();
                }
                _isopen = false;
            }
        }
    }

    boolean assign(int assign) {
        int oldassign = _assigned;
        int newassign = assign;

        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            manager.clearMIDIInCache();
            MXMIDIIn input = manager.findMIDIInput(newassign);
            if (input != this && input != null) {
                input.assign(-1);
            }
             if (newassign < 0) {
                close();
            }
            _assigned = newassign;
           return true;
        }
    }

    public boolean open() {
        if (_assigned < 0) {
            //_debug.println("Not assigned ");
            return false;
        }
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            if (_isopen) {
                _debug.println("Already open");
                return true;
            }
            MXMIDIIn input = manager.findMIDIInput(getName());
            if (input == null) {
                _debug.println("no device named  " + getName());
                return false;
            }
            MidiDevice device = input._device;
            if (_device != null) {
                try {
                    _device.open();
                    _isopen = true;
                    _device.getTransmitter().setReceiver(new Receiver() {
                        @Override
                        public void send(MidiMessage midi, long timeStamp) {
                            try {
                                MXMessage message = MXMessageFactory.fromJavaMessage(_assigned, midi);
                                MXMIDIIn.this.processMidiIn(message);
                            }catch(Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void close() {
                            MXMIDIIn.this.close();
                        }
                    });
                 }catch(MidiUnavailableException e) {
                     e.printStackTrace();
                     close();
                     return false;
                 }
            }
            _isopen = true;
            return true;
        }
    }

    public int hashCode() {
        return _name.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        MXMIDIIn in = (MXMIDIIn) o;
        if (in._name == this._name) {
            return true;
        }
        return false;
    }
    
    public void allNoteOff() {
        _noteOff.allNoteOff();
    }

    public String textForMasterChannel() {
        StringBuffer masterMark = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (masterMark.length() != 0) {
                    masterMark.append(", ");
                }
                masterMark.append(Integer.valueOf(ch + 1));
            }
        }
        return masterMark.toString();
    }
    
    public int parseMasteredText(String text){ 
        if (text == null) {
            return 0;
        }
        String[] array = text.split("[ ,]+");
        int hit = 0;
        
        for (int i = 0; i < array.length; ++ i) {
            String parts = array[i];
            if (parts.length() >= 1) {
                int ch1 = parts.charAt(0) - '0';
                if (ch1 < 0 || ch1 > 9) {
                    continue;
                }
                if (parts.length() >= 2) {
                    int ch2 = parts.charAt(2) - '0';
                    if (ch2 < 0 || ch2 > 9) {
                        continue;
                    }
                    ch1 = ch1 * 10 + ch2;
                }
                if (ch1 >= 1 && ch1 <= 16) {
                    setToMaster(ch1 - 1, true);
                }
                hit ++;
            }
        }
        return hit;
    }
}
