/*
 * Copyright (C) 2022 user0001
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi;

import java.util.ArrayList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MetaMessage;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.SysexMessage;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMessageFactory {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMessageFactory.class);
    
    public static MXMessage makeDummy() {
        return new MXMessage();
    }

    public static MXMessage fromClone(MXMessage old) {
        MXMessage message = new MXMessage();
        message._extype = old._extype;
        message._port = old._port;
        message._template = old._template.clone();
        message._cached = false;
        
        message._value = old._value;
        message._gate = old._gate;
        message._channel = old._channel;
        return message;
    }

    public static MXMessage fromJavaMessage(int port, MidiMessage sm) {
        if (sm instanceof ShortMessage) {
            ShortMessage msg = (ShortMessage) sm;
            return MXMessageFactory.fromShortMessage(port, msg.getStatus(), msg.getData1(), msg.getData2());
        }
        if (sm instanceof SysexMessage) {
            MXMessage m = new MXMessage();
            m._port = port;
            byte[] data = ((SysexMessage) sm).getData();
            int status = ((SysexMessage) sm).getStatus();
            int[] template = new int[data.length + 1];
            template[0] = status;
            for (int i = 0; i < data.length; ++i) {
                template[i + 1] = data[i] & 0xff;
            }
            m._template = template;
            if (status == 240) {
                m._extype = MXMessage.EXTYPE_SYSEX;
            } else if (status == 247) {
                m._extype = MXMessage.EXTYPE_SYSEX_SPECIAL;
            } else {
                _debug.println("Can't inport message " + sm.getClass());
            }
            return m;
        }
        if (sm instanceof MetaMessage) {
            MXMessage message = new MXMessage();
            message._port = port;

            byte[] data = ((SysexMessage) sm).getData();
            message._extype = MXMessage.EXTYPE_META;
            int[] template = new int[data.length];
            for (int i = 0; i < data.length; ++i) {
                template[i] = data[i] & 0xff;
            }
            message._template = template;
            message._dataBytes = new byte[data.length];
            return message;
        }
        _debug.println("Unknown " + sm.getClass());
        return null;
    }

    public static MXMessage fromProgramUp(int port, int channel) {
        MXMessage message = new MXMessage();
        message._port = port;
        message._extype = MXMessage.EXTYPE_PROGRAM_INC;
        return message;
    }

    public static MXMessage fromProgramDown(int port, int channel) {
        MXMessage message = new MXMessage();
        message._port = port;
        message._extype = MXMessage.EXTYPE_PROGRAM_DEC;
        return message;
    }

    public static MXMessage fromSysexMessage(int port, int status, byte[] data) {
        MXMessage message = new MXMessage();
        message._port = port;
        int[] template = new int[data.length + 1];
        template[0] = status & 0x0ff;
        for (int i = 0; i < data.length; ++i) {
            template[i + 1] = data[i] & 0x0ff;
        }
        if (status == 240) {
            message._extype = MXMessage.EXTYPE_SYSEX;
        } else {
            message._extype = MXMessage.EXTYPE_SYSEX_SPECIAL;
        }
        message._template = template;
        return message;
    }

    public static MidiMessage toJavaMessage(MXMessage msg) throws InvalidMidiDataException {
        byte[] data = msg.getDataBytes();
        int extype = msg._extype;
        if (extype == MXMessage.EXTYPE_SHORTMESSAGE) {
            return new ShortMessage(data[0] & 0x0ff, data[1] & 0x0ff, data[2] & 0x0ff);
        }

        byte[] newData = new byte[data.length - 1];
        for (int i = 0; i < data.length - 1; ++ i) {
            newData[i] = data[i + 1];
        }

        if (extype == MXMessage.EXTYPE_SYSEX) {
            return new SysexMessage(data[0] & 0xff, newData, newData.length);
        }
        if (extype == MXMessage.EXTYPE_META) {
            return new MetaMessage(data[0] & 0xff, newData, newData.length);
        }
        if (extype == MXMessage.EXTYPE_SYSEX_SPECIAL) {
            return new SysexMessage(data[0] & 0xff, newData, newData.length);
        }
        throw new InvalidMidiDataException("Not supported.");
    }

    public static MXMessage fromShortMessage(int port, int status, int data1, int data2) {
        MXMessage message = new MXMessage();
        message._port = port;
        message._cached = false;
        int command = status & 0xf0;
        int channel = status & 0x0f;
        message._extype = MXMessage.EXTYPE_SHORTMESSAGE;
        message._channel = channel;

        if (command < 0 || command > 255) {
            _debug.println("command = " + command);
            _debug.printStackTrace();
            return null;
        }
        if (channel < 0 || channel > 15) {
            _debug.println("channel= " + channel);
            _debug.printStackTrace();
            return null;
        }
        if (data1 < 0 || data1 > 127) {
            _debug.println("data1 = " + data1);
            _debug.printStackTrace();
            return null;
        }
        if (data2 < 0 || data2 > 127) {
            _debug.println("data2 = " + data2);
            _debug.printStackTrace();
            return null;
        }

        message._template[0] = status & 0xff;
        message._template[1] = data1 & 0xff;
        message._template[2] = data2 & 0xff;

        int defValue = 0;

        switch (command) {
            case MXMidi.COMMAND_PROGRAMCHANGE:
                message._channel = channel;
                message._template[1] = MXMessage.DTEXT_VL;
                message._template[2] = 0;
                message._extype = MXMessage.EXTYPE_SHORTMESSAGE;
                message.setValue(data1);
                break;
            case MXMidi.COMMAND_CONTROLCHANGE:
                message.setGate(data1);
                message.setValue(data2);
                message._template[1] = MXMessage.DTEXT_GL;
                message._template[2] = MXMessage.DTEXT_VL;
                switch(message.getCCCodeFromBytes()) {
                    case MXMidi.DATA1_CCPANPOT:
                        defValue = 64;
                        break;
                    case MXMidi.DATA1_CC_EXPRESSION:
                        defValue = 127;
                        break;
                }
                break;
            case MXMidi.COMMAND_NOTEON:
            case MXMidi.COMMAND_NOTEOFF:
            case MXMidi.COMMAND_POLYPRESSURE:
                message.setGate(data1);
                message.setValue(data2);
                message._template[1] = MXMessage.DTEXT_GL;
                message._template[2] = MXMessage.DTEXT_VL;
                break;
            case MXMidi.COMMAND_PITCHWHEEL:
                message.setValue((data1 & 127) | (data2 << 7));
                message._template[1] = MXMessage.DTEXT_VL;
                message._template[2] = MXMessage.DTEXT_VH;
                break;
            case MXMidi.COMMAND_CHANNELPRESSURE:
                message.setValue(data1);
                message._template[1] = MXMessage.DTEXT_VL;
                break;
            default:
                if (command >= 240 && command <= 247) {
                    if (command == MXMidi.STATUS_SONGPOSITION) {
                        message.setValue((data1 & 127) | (data2 << 7));
                        message._template[1] = MXMessage.DTEXT_VL;
                        message._template[2] = MXMessage.DTEXT_VH;
                    }
                    if (command == MXMidi.STATUS_SONGSELECT) {
                        message.setValue(data1);
                        message._template[1] = MXMessage.DTEXT_VL;
                    }
                }
                break;
        }
        return message;
    }

    public static MXMessage fromDtext(String text) {
        if (text.length() == 0) {
            return makeDummy();
        }
        int checksumLength;
        int rpn_msb;
        int rpn_lsb;
        int nrpn_msb;
        int nrpn_lsb;

        char[] line = text.toCharArray();

        char[] word = new char[line.length];
        int wx = 0;

        int readX = 0;
        ArrayList<String> separated = new ArrayList();
        
        checksumLength = -1;

        while(readX < line.length) {
            char ch = line[readX ++];
            if (ch == '[') {
                if (checksumLength < 0) {
                    checksumLength = 0;
                }else {
                    new Exception("Checksum should be only one").printStackTrace();
                }
                continue;
            }
            if (checksumLength >= 0)  {
                checksumLength ++;
            }
            if (ch == ']') {
                if (checksumLength >= 0) {
                    if (wx != 0) {
                        separated.add(new String(word, 0, wx));
                    }
                    separated.add("#CHECKSUM");
                    wx = 0;
                }else {
                    _debug.println("Checksum have not opened");
                    _debug.printStackTrace();
                }
                continue;
            }
            if (ch == ' '|| ch == '\t' || ch == ',') {
                if (wx != 0) {
                    separated.add(new String(word, 0, wx));
                }
                wx = 0;
                continue;
            }
            word[wx ++] = ch;
        }
        if (wx != 0) {
            separated.add(new String(word, 0, wx));
            wx = 0;
        }
        
        int gate = -1;
        if (text.contains("@")) {
            ArrayList<String> sepa2 = new ArrayList();
            for (int sx = 0; sx < separated.size(); ++ sx) {
                String str = separated.get(sx);
                if (str.startsWith("@")) {
                    if (str.equalsIgnoreCase("@PB")) {
                        sepa2.add("#ECH");
                        sepa2.add(separated.get(++ sx));
                        sepa2.add(separated.get(++ sx));
                    }
                    else if (str.equalsIgnoreCase("@CP")) {
                        sepa2.add("#DCH");
                        sepa2.add(separated.get(++ sx));
                        sepa2.add("#NONE");
                    }
                    else if (str.equalsIgnoreCase("@PKP")) {
                        sepa2.add("#ACH");
                        String t = separated.get(++sx);
                        if(t.startsWith("#")) {
                            sepa2.add(t);
                        }else {
                            gate = MXUtil.parseTextForNumber(t);
                            sepa2.add("#GL");
                        }
                        sepa2.add(separated.get(++ sx));
                    }
                    else if (str.equalsIgnoreCase("@CC")) {
                        sepa2.add("#BCH");
                        String t = separated.get(++sx);
                        if(t.startsWith("#")) {
                            sepa2.add(t);
                        }else {
                            gate = MXUtil.parseTextForNumber(t);
                            sepa2.add("#GL");
                        }
                        sepa2.add(separated.get(++ sx));
                    }
                    else if (str.equalsIgnoreCase("@SYSEX")) {
                        //THRU (no need recompile)
                    }
                    else if (str.equalsIgnoreCase("@RPN")) {
                        int lsb = MXUtil.parseTextForNumber(separated.get(++ sx));
                        int msb = MXUtil.parseTextForNumber(separated.get(++ sx));
                        int value = MXUtil.parseTextForNumber(separated.get(++ sx));
                        sepa2.add("#BCH");
                        sepa2.add("06H"); // DATA_ENTRY
                        sepa2.add(MXUtil.toHexFF(value) + "H");
                        rpn_lsb = lsb;
                        rpn_msb = msb;
                    }
                    else if (str.equalsIgnoreCase("@NRPN")) {
                        int lsb = MXUtil.parseTextForNumber(separated.get(++ sx));
                        int msb = MXUtil.parseTextForNumber(separated.get(++ sx));
                        int value = MXUtil.parseTextForNumber(separated.get(++ sx));
                        sepa2.add("#BCH");
                        sepa2.add("06H");
                        sepa2.add(MXUtil.toHexFF(value) + "H");
                        nrpn_lsb = lsb;
                        nrpn_msb = msb;
                    }
                }else {
                    sepa2.add(str);
                }
            }
            separated = sepa2;
        }
        
        // cleanup
        int[] compiled = new int[line.length];
        int cx = 0;
        int px = 0;

        for (int sx = 0; sx < separated.size(); ++ sx) {
            String str = separated.get(sx);
            int code = -1;
            if (str.startsWith("#")) {
                int find = MXMessage.textAlias.indexOfName(str);
                if (find >= 0) {
                    code = MXMessage.textAlias.get(find).value.intValue();
                }
                if (code == -1) {
                    new IllegalArgumentException("Syntax Error [" + str + "] in " + "[" + text + "]").printStackTrace();
                }
            }
            if (code != -1) {
                compiled[px++] = code;
                continue;
            }

            int value = MXUtil.parseTextForNumber(str);
            if (value < 0) {
                new IllegalArgumentException("Syntax Error [" + str + "] in " + "[" + text + "]").printStackTrace();
                value = 0x100; // NONE
            }
            compiled[px ++] = value;
        }
        int[] template = new int[cx];
        template = new int[px];
        for (int i = 0; i < px; ++ i) {
            int x = compiled[i];
            template[i] = compiled[i];
        }
        MXMessage ret = new MXMessage(template, checksumLength);
        if (gate > 0) {
            ret.setGate(gate);
        }
        return ret;
    }
    
    public static String toDText(MXMessage message) {
        ArrayList<String> array = message.toDArray();
        StringBuffer text = new StringBuffer();
        String last = "";
        for (String seg : array) {
            if (text.length() >= 0) {
                if (seg.equals("[") || seg.equals("]") || last.equals("[") || last.equals("]")) {
                    // nothing
                }else {
                    text.append(" ");
                }
            }
            text.append(seg);
        }
        return text.toString();
    }
}
