/*
 * Copyright (C) 2022 user0001
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public Lficense for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.synthtarou.midimixer.libs.midi;

import java.util.ArrayList;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXWrapList;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_DUMMY;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_META;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_PROGRAM_DEC;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_PROGRAM_INC;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_SHORTMESSAGE;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_SYSEX;
import static jp.synthtarou.midimixer.libs.midi.MXMessage.EXTYPE_SYSEX_SPECIAL;

/**
 *
 * @author YOSHIDA Shintarou
 * infomation from g200kg Music & Software https://www.g200kg.com/
 */
public final class MXMessage {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMessage.class);

    public static final int EXTYPE_DUMMY = 0xff00;
    public static final int EXTYPE_SHORTMESSAGE = 0x0100;
    public static final int EXTYPE_PROGRAM_INC = 0x0200;
    public static final int EXTYPE_PROGRAM_DEC = 0x0300;
    public static final int EXTYPE_SYSEX = 0x0400;
    public static final int EXTYPE_SYSEX_SPECIAL = 0x0500;
    public static final int EXTYPE_META = 0x0600;

    public int _extype = EXTYPE_DUMMY;

    int _value = -1;
    int _gate = -1;
    int _channel = 0;
    int _port = -1;

    protected byte[] _dataBytes = new byte[3];
    boolean _cached = false;
    protected boolean _cachedHasHi, _cachedHasLo, _cachedHasGate;
    protected int[] _template = new int[3];
    protected int _checksumLength = -1;
    
    static final int DTEXT_NONE = 0x100;
    static final int DTEXT_VL = 0x200;
    static final int DTEXT_VH = 0x300;
    static final int DTEXT_GL = 0x400;
    static final int DTEXT_GH = 0x500;
    static final int DTEXT_CH = 0x600;
    static final int DTEXT_1CH = 0x700;
    static final int DTEXT_2CH = 0x800;
    static final int DTEXT_3CH = 0x900;
    static final int DTEXT_PCH = 0xA00;
    static final int DTEXT_1RCH = 0xB00;
    static final int DTEXT_2RCH = 0xC00;
    static final int DTEXT_3RCH = 0xD00;
    static final int DTEXT_4RCH = 0xE00;
    static final int DTEXT_VF1 = 0xF00;
    static final int DTEXT_VF2 = 0x1000;
    static final int DTEXT_VF3 = 0x1100;
    static final int DTEXT_VF4 = 0x1200;
    static final int DTEXT_VPGL = 0x1300;
    static final int DTEXT_VPGH = 0x1400;
    static final int DTEXT_CCNUM = 0x1500;
    static final int DTEXT_RPNMSB = 0x1600;
    static final int DTEXT_RPNLSB = 0x1700;
    static final int DTEXT_NRPNMSB = 0x1800;
    static final int DTEXT_NRPNLSB = 0x1900;
    static final int DTEXT_RSCTRT1 = 0x1A00;
    static final int DTEXT_RSCTRT2 = 0x1B00;
    static final int DTEXT_RSCTRT3 = 0x1C00;
    static final int DTEXT_RSCTRT1P = 0x1D00;
    static final int DTEXT_RSCTRT2P = 0x1E00;
    static final int DTEXT_RSCTRT3P = 0x1F00;
    static final int DTEXT_RSCTPT1 = 0x2000;
    static final int DTEXT_RSCTPT2 = 0x2100;
    static final int DTEXT_RSCTPT3 = 0x2200;
    static final int DTEXT_RSCTPT1P = 0x2300;
    static final int DTEXT_RSCTPT2P = 0x2400;
    static final int DTEXT_RSCTPT3P = 0x2500;
    static final int DTEXT_CHECKSUM = 0x2600;

    static final int DTEXT_4CH = 0x2700;
    static final int DTEXT_5CH = 0x2800;
    static final int DTEXT_6CH = 0x2900;
    static final int DTEXT_7CH = 0x2a00;
    static final int DTEXT_8CH = 0x2b00;
    static final int DTEXT_9CH = 0x2c00;
    static final int DTEXT_ACH = 0x2d00;
    static final int DTEXT_BCH = 0x2e00;
    static final int DTEXT_CCH = 0x2f00;
    static final int DTEXT_DCH = 0x3000;
    static final int DTEXT_ECH = 0x3100;
    static final int DTEXT_FCH = 0x3200;

    static MXWrapList<Integer> textAlias = new MXWrapList();
   
    static {
        textAlias.addNameAndValue("#NONE", DTEXT_NONE);
        textAlias.addNameAndValue("#VL", DTEXT_VL);
        textAlias.addNameAndValue("#VH", DTEXT_VH);
        textAlias.addNameAndValue("#GL", DTEXT_GL);
        textAlias.addNameAndValue("#GH", DTEXT_GH);
        textAlias.addNameAndValue("#CH", DTEXT_CH);
        textAlias.addNameAndValue("#1CH", DTEXT_1CH);
        textAlias.addNameAndValue("#2CH", DTEXT_2CH);
        textAlias.addNameAndValue("#3CH", DTEXT_3CH);
        textAlias.addNameAndValue("#PCH", DTEXT_PCH);
        textAlias.addNameAndValue("#1RCH", DTEXT_1RCH);
        textAlias.addNameAndValue("#2RCH", DTEXT_2RCH);
        textAlias.addNameAndValue("#3RCH", DTEXT_3RCH);
        textAlias.addNameAndValue("#4RCH", DTEXT_4RCH);
        textAlias.addNameAndValue("#VF1", DTEXT_VF1);
        textAlias.addNameAndValue("#VF2", DTEXT_VF2);
        textAlias.addNameAndValue("#VF3", DTEXT_VF3);
        textAlias.addNameAndValue("#VF4", DTEXT_VF4);
        textAlias.addNameAndValue("#VPGL", DTEXT_VPGL);
        textAlias.addNameAndValue("#VPGH", DTEXT_VPGH);
        textAlias.addNameAndValue("#RSCTRT1", DTEXT_RSCTRT1);
        textAlias.addNameAndValue("#RSCTRT2", DTEXT_RSCTRT2);
        textAlias.addNameAndValue("#RSCTRT3", DTEXT_RSCTRT3);
        textAlias.addNameAndValue("#RSCTRT1P", DTEXT_RSCTRT1P);
        textAlias.addNameAndValue("#RSCTRT2P", DTEXT_RSCTRT2P);
        textAlias.addNameAndValue("#RSCTRT3P", DTEXT_RSCTRT3P);
        textAlias.addNameAndValue("#RSCTPT1", DTEXT_RSCTPT1);
        textAlias.addNameAndValue("#RSCTPT2", DTEXT_RSCTPT2);
        textAlias.addNameAndValue("#RSCTPT3", DTEXT_RSCTPT3);
        textAlias.addNameAndValue("#RSCTPT1P", DTEXT_RSCTPT1P);
        textAlias.addNameAndValue("#RSCTPT2P", DTEXT_RSCTPT2P);
        textAlias.addNameAndValue("#RSCTPT3P", DTEXT_RSCTPT3P);
        textAlias.addNameAndValue("#CHECKSUM", DTEXT_CHECKSUM);

        textAlias.addNameAndValue("#4CH", DTEXT_4CH);
        textAlias.addNameAndValue("#5CH", DTEXT_5CH);
        textAlias.addNameAndValue("#6CH", DTEXT_6CH);
        textAlias.addNameAndValue("#7CH", DTEXT_7CH);
        textAlias.addNameAndValue("#8CH", DTEXT_8CH);
        textAlias.addNameAndValue("#9CH", DTEXT_9CH);
        textAlias.addNameAndValue("#ACH", DTEXT_ACH);
        textAlias.addNameAndValue("#BCH", DTEXT_BCH);
        textAlias.addNameAndValue("#CCH", DTEXT_CCH);
        textAlias.addNameAndValue("#DCH", DTEXT_DCH);
        textAlias.addNameAndValue("#ECH", DTEXT_ECH);
        textAlias.addNameAndValue("#FCH", DTEXT_FCH);
    }
    
    public void makeSomeCached() {
        if (_cached) { 
            return; 
        }
        synchronized(this) {
            if (_template.length != _dataBytes.length) {
                _dataBytes = new byte[_template.length];
            }
            int checksumTo = -1;
            int newValue = _value;
            int newGate = _gate;
            
            for (int i = 0; i < _template.length; ++ i) {
                int x = _template[i];
                if ((x & 0xff00) != 0) {
                    switch(x & 0xff00) {
                        case DTEXT_NONE:
                            x = 0;
                            break;
                        case DTEXT_VL:
                            x = newValue & 0x7f;
                            break;
                        case DTEXT_VH:
                            x = (newValue >> 7) & 0x7f;
                            break;
                        case DTEXT_GL:
                            x = newGate & 0x7f;
                            break;
                        case DTEXT_GH:
                            x = (newGate >> 7) & 0x7f;
                            break;
                        case DTEXT_CH:
                            x = _channel;
                            break;
                        case DTEXT_1CH:
                            x = 0x10 + _channel;
                            break;
                        case DTEXT_2CH:
                            x = 0x20 + _channel;
                            break;
                        case DTEXT_3CH:
                            x = 0x30 + _channel;
                            break;
                        case DTEXT_4CH:
                            x = 0x40 + _channel;
                            break;
                        case DTEXT_5CH:
                            x = 0x50 + _channel;
                            break;
                        case DTEXT_6CH:
                            x = 0x60 + _channel;
                            break;
                        case DTEXT_7CH:
                            x = 0x70 + _channel;
                            break;
                        case DTEXT_8CH:
                            x = 0x80 + _channel;
                            break;
                        case DTEXT_9CH:
                            x = 0x90 + _channel;
                            break;
                        case DTEXT_ACH:
                            x = 0xA0 + _channel;
                            break;
                        case DTEXT_BCH:
                            x = 0xB0 + _channel;
                            break;
                        case DTEXT_CCH:
                            x = 0xC0 + _channel;
                            break;
                        case DTEXT_DCH:
                            x = 0xD0 + _channel;
                            break;
                        case DTEXT_ECH:
                            x = 0xE0 + _channel;
                            break;
                        case DTEXT_FCH:
                            x = 0xF0 + _channel;
                            break;
                        case DTEXT_PCH:
                            if (_port >= 0 && _port <= 3) {
                                x = _port * 0x10 + _channel;
                            }else {
                                x = 0x30 + _channel;
                            }
                            break;
                        case DTEXT_1RCH:
                        case DTEXT_2RCH:
                        case DTEXT_3RCH:
                        case DTEXT_4RCH:
                            throw new IllegalArgumentException("1RCH, 2RCH, 3RCH, 4RCH not supported.");
                            //break;
                        case DTEXT_VF1:
                            x = (_value) & 0x0f;
                            break;
                        case DTEXT_VF2:
                            x = (_value >> 4) & 0x0f;
                            break;
                        case DTEXT_VF3:
                            x = (_value >> 8) & 0x0f;
                            break;
                        case DTEXT_VF4:
                            x = (_value >> 12) & 0x0f;
                            break;
                        case DTEXT_VPGL:
                            x = (_value + _gate) & 0x7f;
                            break;
                        case DTEXT_VPGH:
                            x = ((_value + _gate) >> 7) & 0x7f;
                            break;
                        case DTEXT_RSCTRT1:
                        case DTEXT_RSCTRT2:
                        case DTEXT_RSCTRT3:
                            throw new IllegalArgumentException("RSCTRT1, RSCTRT2, RSCTRT3 not supported.");
                            //break;
                        case DTEXT_RSCTRT1P:
                        case DTEXT_RSCTRT2P:
                        case DTEXT_RSCTRT3P:
                            throw new IllegalArgumentException("RSCTRT1P, RSCTRT2P, RSCTRT3P not supported.");
                            //break;
                        case DTEXT_RSCTPT1:
                        case DTEXT_RSCTPT2:
                        case DTEXT_RSCTPT3:
                            throw new IllegalArgumentException("RSCTPT1, RSCTPT2, RSCTPT3 not supported.");
                             //break;
                        case DTEXT_RSCTPT1P:
                        case DTEXT_RSCTPT2P:
                        case DTEXT_RSCTPT3P:
                            throw new IllegalArgumentException("RSCTPT1P, RSCTPT2P, RSCTPT3P not supported.");
                            //break;
/*
    static final int DTEXT_CCNUM = 0x1500;
    static final int DTEXT_RPNMSB = 0x1600;
    static final int DTEXT_RPNLSB = 0x1700;
    static final int DTEXT_NRPNMSB = 0x1800;
    static final int DTEXT_NRPNLSB = 0x1900;
*/
                        case DTEXT_CHECKSUM:
                            checksumTo = i;
                            break;

                        default:
                            x = 0;
                            boolean haveEx = false;
                            StringBuffer str = new StringBuffer();
                            for (int n = 0; n < _template.length; ++ n) {
                                int x1 = (_template[n] >> 16) & 0xff;
                                int x2 = _template[n] & 0xff;
                                if (x1 != 0) {
                                    haveEx = true;
                                    str.append(MXUtil.toHexFF(x1));
                                    str.append(MXUtil.toHexFF(x2));
                                }else {
                                    str.append(MXUtil.toHexFF(x2));
                                }
                                str.append(",");
                            }
                            if (haveEx) {
                                System.out.println(str.toString());
                            }
                            throw new IllegalArgumentException("something wrong ");
                            //break;
                    }
                }
                _dataBytes[i] = (byte)(x & 0xff);
            }
            int command = _dataBytes[0] & 0xf0;
            if (command >= 0x80 && command <= 0xe0) {
                _dataBytes[0] = (byte)(command + _channel);
            }
            if (_checksumLength >= 0 && checksumTo >= 0) {
                int x128 = 0;
                for (int x = checksumTo - _checksumLength; x < checksumTo; ++ x) {
                    x128 += _dataBytes[x];
                }
                x128 = x128 & 0x7f;
                int r = 128 - x128;
                _dataBytes[checksumTo] = (byte)(r & 0x7f);
            }
            
            _cachedHasHi = false;
            _cachedHasLo = false;
            _cachedHasGate = false;
            for (int i = 0; i < _template.length; ++ i) {
                if (_template[i] == DTEXT_VL) {
                    _cachedHasLo = true;
                }
                if (_template[i] == DTEXT_VH) {
                    _cachedHasHi = true;
                }
                if (_template[i] == DTEXT_GL || _template[i] == DTEXT_GH) {
                    _cachedHasGate = true;
                }
            }
            _cached = true;
        }
    }

    public int getPort() {
        return _port;
    }
    
    public synchronized void setPort(int port) {
        _cached = false;
        _port = port;
    }
 
    public synchronized int getStatus() {
        makeSomeCached();
        int status = _dataBytes[0] & 0xff;
        return status;
    }
    
    public synchronized int getCommand() {
        makeSomeCached();
        int status = _dataBytes[0] & 0xff;
        if (status >= 0x80 && status <= 0xe0) {
            return status & 0xf0;
        }else {
            return status;
        }
    }
    
    public synchronized void setStatus(int status) {
        _cached = false;
        if (_extype == EXTYPE_SHORTMESSAGE) {
            _template[0] = status & 0xff;
        }else {
            throw new IllegalStateException("setStatus on none ShortMessage");
        }
    }
    
    public synchronized void setChannel(int channel) {
        _cached = false;
        if (channel  < 0 || channel > 15) {
            _debug.println("setChannel " + channel);
        }
        _channel = channel;
    }
    
    public int getData1FromBytes() {
        makeSomeCached();
        return _dataBytes[1] & 0xff;
    }

    public int getData2FromBytes() {
        makeSomeCached();
        return _dataBytes[2] & 0xff;
    }
    
    public synchronized int getCCCodeFromBytes() {
        if (getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            if (getData1FromBytes() >= 128) {
                _debug.println("getData1 " + getData1FromBytes());
            }
            return getData1FromBytes();
        }
        _debug.println("Its not cc.");
        return 0;
    }
    
    public synchronized int getNoteNumberFromBytes() {
        switch(getCommand()) {
        case MXMidi.COMMAND_NOTEON:
        case MXMidi.COMMAND_NOTEOFF:
        case MXMidi.COMMAND_POLYPRESSURE:
            return getData1FromBytes();
        }
        _debug.println("Its not note message.");
        return 0;
    }

    public synchronized int getVelocityFromBytes() {
        switch(getCommand()) {
        case MXMidi.COMMAND_NOTEON:
        case MXMidi.COMMAND_NOTEOFF:
        case MXMidi.COMMAND_POLYPRESSURE:
            return getData2FromBytes();
        }
        _debug.println("Its not note message.");
        return 0;
    }

    public int getChannel() {
        if (_channel < 0 || _channel > 15) {
            _debug.println("getAsChannel " + _channel);
            return 0;
        }
        return _channel;
    }
    
    public int getValue() {
        return _value;
    }

    public synchronized void setValue(int value) {
        _value = value;
        _cached = false;
    }

    public int getGate() {
        return _gate;
    }

    public synchronized void setGate(int gate) {
        _gate = gate;
        _cached = false;
    }

    protected MXMessage() {
    }

    protected MXMessage(int[] template, int checksumLength) {
        _template = template;
        _checksumLength = checksumLength;
        if (template[0] == MXMidi.STATUS_SYSEXSTART) {
            _extype = EXTYPE_SYSEX;
        }else if (template[0] == MXMidi.STATUS_SYSEXFIN) {
            _extype = EXTYPE_SYSEX_SPECIAL;
        }else {
            _extype = EXTYPE_SHORTMESSAGE;
        }
    }

    public synchronized byte[] getDataBytes() {
        makeSomeCached();
        return _dataBytes;        //SYSEXの場合１バイト目は、STATUSに入る
    }

    public boolean isMessageTypeChannel() {
        makeSomeCached();

        if (_extype != EXTYPE_SHORTMESSAGE) {
            return false;
        }
        if (getCommand() >= 0x80 && getCommand() <= 0xe0) {
            return true;
        }
        return false;
    }

    public boolean isMessageTypeSystemKnown() {
        makeSomeCached();

        if (_extype != EXTYPE_SHORTMESSAGE) {
            return false;
        }
        if (getCommand() >= 0x80 && getCommand() <= 0xe0) {
            return true;
        }
        if (getCommand() >= 0xf0 && getCommand() <= 0xf7) {
            return true;
        }
        
        return false;
    }

    public boolean isMessageTypeDatas() {
        makeSomeCached();

        if (getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            switch (getCCCodeFromBytes()) {
                case MXMidi.DATA1_CC_DATAENTRY:
                case MXMidi.DATA1_CC_DATAINC:
                case MXMidi.DATA1_CC_DATADEC:
                    return true;
            }
        }
        return false;
    }

    public String toString() {
        makeSomeCached();
        
        int port = getPort();
        
        switch (_extype) {
            case MXMessage.EXTYPE_META:
                return "Meta";
            case MXMessage.EXTYPE_SYSEX:
                return "Sysex [" + MXUtil.dumpHexFF(getDataBytes()) + "]";
            case MXMessage.EXTYPE_SYSEX_SPECIAL:
                return "SysexSpecial [" + MXUtil.dumpHexFF(getDataBytes()) + "]";
            case MXMessage.EXTYPE_SHORTMESSAGE:
                break;
            case MXMessage.EXTYPE_PROGRAM_INC:
                return "Program up";
            case MXMessage.EXTYPE_PROGRAM_DEC:
                return "Program dec";
        }
    
        String chname;
        if (isMessageTypeChannel()) {
            int channel = getChannel();
            chname = "[" + MXMidiUI.nameOfPort(port) + (channel+1) +"]";
        }else {
            chname = "[" + MXMidiUI.nameOfPort(port) + "]";
        }

        int command = getCommand();
        
        String name = MXMidiUI.nameOfMessage(getStatus(), getData1FromBytes(), getData2FromBytes());

        if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            int data1 = getCCCodeFromBytes();
            int data2 = getValue();
            if (data1 == MXMidi.DATA1_CCBANKSELECT) {
                return chname + " " + name + " = " + data2 + " <MSB LSB>";// + _addInfoBankLSB;
            }
            if (data1 == MXMidi.DATA1_CC_DATAENTRY) {
                return chname + " " + name + " = " + getValue();
            }
            return chname + " " + name + " = " + data2;
        }else {
            if (command == MXMidi.COMMAND_NOTEOFF) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + " " + name + ":" + note + "(" +MXMidiUI.nameOfNote(note) + ")" + velocity;
            }
            if (command == MXMidi.COMMAND_NOTEON) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + " " + name + ":" + note + "(" +MXMidiUI.nameOfNote(note) + ")" + velocity;
            }
            if (command == MXMidi.COMMAND_POLYPRESSURE) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + " " + name + ":" + note + "(" +MXMidiUI.nameOfNote(note) + ")" + velocity;
            }
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                int program = getValue();
                return chname + " " + name + " = " + program;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                return chname + " " + name + " press " + getValue();
            }
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                return chname + " " + name + " = " + getValue();
            }
            if (command == MXMidi.STATUS_SONGPOSITION) {
                return name + " @" + getValue();
            }
            if (command == MXMidi.STATUS_SONGSELECT) {
                return name + " @" + getValue();
            }
        }

        if (command >= 0xf0 && command <= 0xf7) {
            return name;
        }

        String extype = "";
        switch(_extype) {
            case EXTYPE_DUMMY:
                extype = "Unknown";
                break;
            case EXTYPE_SHORTMESSAGE:
                extype = "ShortMessage";
                break;
            case EXTYPE_PROGRAM_INC:
                extype = "ProgINC";
                break;
            case EXTYPE_PROGRAM_DEC:
                extype = "ProgDEC";
                break;
            case EXTYPE_SYSEX:
                extype = "SysEX";
                break;
            case EXTYPE_SYSEX_SPECIAL:
                extype = "SysEXSpecial";
                break;
            case EXTYPE_META:
                extype = "Meta";
                break;
        }
        return "type[" + extype + ":" + MXUtil.dumpHexFF(getDataBytes()) + "]";
    }
    
    public boolean hasValueHiField() {
        makeSomeCached();
        return _cachedHasHi;
    }

    public boolean hasValueLowField() {
        makeSomeCached();
        return _cachedHasLo;
    }

    public boolean hasGateField() {
        makeSomeCached();
        return _cachedHasGate;
    }

    public boolean hasValueSetted() {
        return _value >= 0;
    }

    public boolean hasSameTemplateExcludeGateAndValue(MXMessage target) {
        if (this._extype != target._extype) {
            return false;
        }
        
        int[] t1 = _template;
        int[] t2 = target._template;

        if (t1.length != t2.length) {
            return false;
        }
        
        for (int i = 0; i < _template.length; ++ i) {
            if ((t1[i] & 0xff00) !=  0 || (t2[i] & 0xff00) != 0) {
                if ((t1[i] & 0xff00) != (t2[i] & 0xff00)) {
                    return false;
                }
                if ((t1[i] & 0x00ff) != (t2[i] & 0x00ff)) {
                    throw new IllegalStateException("(template[x] & 0x00ff) should be 0 when (template[x] & 0xff00) != 0");
                }
            }else {
                if ((t1[i] & 0x00ff) != (t2[i] & 0x00ff)) {
                    return false;
                }
            }
        }
        /*
        if (this.getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
            System.out.println("same " + target + " == " + this);
            new Throwable().printStackTrace();
        }*/
        return true;
    }

    public ArrayList<String> toDArray() {
        ArrayList<String> texts = new ArrayList();
        makeSomeCached();
        
        if (isMessageTypeChannel()) {
            int command = getCommand();
            int channel = getChannel();
            int data1 = getData1FromBytes();
            int data2 = getData2FromBytes();
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                texts.add("@PB");
                texts.add("#VH");
                texts.add("#VL");
                return texts;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                texts.add("@CP");
                texts.add("#VL");
                return texts;
            }
            if (command == MXMidi.COMMAND_POLYPRESSURE) {
                texts.add("@PKP");
                texts.add("#GL");
                texts.add("#VL");
                return texts;
            }
            if (command == MXMidi.COMMAND_CONTROLCHANGE) {
                texts.add("@CC");
                texts.add(String.valueOf(data1));
                texts.add("#VL");
                return texts;
            }
            /*
                @RPN [RPN MSB] [RPN LSB] [Data MSB] [Data LSB] 	RPNを送信します。
                @NRPN [NRPN MSB] [NRPN LSB] [Data MSB] [Data LSB] 	NRPNを送信します。 
            */
        }
        
        int csumTo = -1;
        for (int i = 0; i < _template.length; ++ i) {
            if (_template[i] == MXMessage.DTEXT_CHECKSUM) {
                csumTo = i;
                break;
            }
        }
        int csumFrom = -1;
        if (csumTo >= 0) {
            csumFrom = csumTo - _checksumLength;
        }
        
        for (int i = 0; i < _template.length; ++ i) {
            if (i == csumFrom) {
                texts.add("[");
            }
            int code = _template[i];
            if (code == MXMessage.DTEXT_CHECKSUM) {
                texts.add("]");
                continue;
            }
            if ((code >> 8) != 0) {
                int index = textAlias.indexOfValue(code);
                texts.add(textAlias.nameOfIndex(index));
            }else {
                String seg = MXUtil.toHexFF(code);
                texts.add(seg + 'h');
            }
        }
        return texts;
    }
    
    public String toShortString() {
        makeSomeCached();
        
        int port = getPort();
        
        switch(_extype) {
            case EXTYPE_DUMMY:
                return "-";
            case EXTYPE_PROGRAM_INC:
                return "++";
            case EXTYPE_PROGRAM_DEC:
                return "--";
            case EXTYPE_SYSEX:
                return "Sys";
            case EXTYPE_SYSEX_SPECIAL:
                return "Sys2";
            case EXTYPE_META:
                return "Meta";
            case EXTYPE_SHORTMESSAGE:
                //OK;
                break;
            default:
                return "???";
        }
    
        String chname;
        if (isMessageTypeChannel()) {
            int channel = getChannel();
            chname = "" + (channel+1);
        }else {
            chname = "";
        }

        int command = getCommand();
        
        if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            int data1 = getCCCodeFromBytes();
            if (data1 == MXMidi.DATA1_CCBANKSELECT) {
                return chname + "bank";
            }
            if (data1 == MXMidi.DATA1_CC_DATAENTRY) {
                return chname + "data";
            }
            return chname + MXMidiUI.nameOfControlChange(data1);
        }else {
            if (command == MXMidi.COMMAND_NOTEOFF) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + MXMidiUI.nameOfNote(note) + "-";
            }
            if (command == MXMidi.COMMAND_NOTEON) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + MXMidiUI.nameOfNote(note);
            }
            if (command == MXMidi.COMMAND_POLYPRESSURE) {
                int note = getNoteNumberFromBytes();
                int velocity = getVelocityFromBytes();
                return  chname + "PPrs";
            }
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                int program = getValue();
                return  chname + "PG" + program;
            }
            if (command == MXMidi.COMMAND_CHANNELPRESSURE) {
                return  chname + "ChPrs";
            }
            if (command == MXMidi.COMMAND_PITCHWHEEL) {
                return  chname + "Pitch";
            }
            if (command == MXMidi.STATUS_SONGPOSITION) {
                return  chname + "Pos";
            }
            if (command == MXMidi.STATUS_SONGSELECT) {
                return  chname + "Song";
            }
        }

        return  MXMidiUI.nameOfMessage(getStatus(), getData1FromBytes(), getData2FromBytes());
    }
}
