/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the FFree Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer;

import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.port.FinalMIDIOut;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIInManager;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIIn;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIOutManager;
import jp.synthtarou.midimixer.mx00playlist.MX00Process;
import jp.synthtarou.midimixer.mx10input.MX10Process;
import jp.synthtarou.midimixer.mx12masterkeys.MX12Process;
import jp.synthtarou.midimixer.mx30controller.MX30Process;
import jp.synthtarou.midimixer.mx40layer.MX40Process;
import jp.synthtarou.midimixer.mx60output.MX60Process;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMain {
    private static MXMain _main = new MXMain(); 

    public static MXMain getMain() {
        return _main;
    }
    
    private MXMainWindow _mainWindow;

    public MXMainWindow getMainWindow() {
        return _mainWindow;
    }
    
    public MX12Process getMasterKeys() {
        return _masterkeyProcess;
    }

    private MX10Process _inputProcess;
    private MX12Process _masterkeyProcess;
    private MX00Process _playlistProcess;
    private MX30Process _kontrolProcess;
    private MX40Process _layerProcess;
    private MX60Process _outputProcess;
    
    public boolean _logInput = true;
    public boolean _logOutput = true;
    
    public static void startMainPath(MXMIDIIn input, MXMessage message) {
        if (message.isMessageTypeChannel() && input.isToMaster(message.getChannel())) {
            _main._masterkeyProcess.processMXMessage(message);
        }else {
            _main._inputProcess.processMXMessage(message);
        }
    }

    /**
     * アプリを起動する
     * @param args 引数
     * @throws Exception エラー通知
     */
    public static void main(String[] args) throws Exception {
        try {
            MXDebugConsole.initDebugConsole(args);

            //フォント描写でアンチエイリアスを有効にする
            System.setProperty("awt.useSystemAAFontSettings", "on");

            /*        
            try {
                LaunchLoopMidi midiout = new LaunchLoopMidi();
            }catch( Throwable e ) {
                e.printStackTrace();
            }
            LookAndFeelPanel.setLookAndFeel(LookAndFeelConfig.THEME_OS_STANDARD);
            LookAndFeelPanel.updateLookAndFeelRecursive();
            */
            
        } catch (Throwable e) {
            e.printStackTrace();
        }

        getMain().startUI();
    }
    
    public void startUI()  {
        //スレッドとして起動したほうが見た目がよくなるジンクス
        if (SwingUtilities.isEventDispatchThread() == false) {
            try {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        startUI();
                    }
                });
            } catch (Throwable ex) {
                ex.printStackTrace();
                JOptionPane.showMessageDialog(null, "起動に失敗しました");
            }            
            return;
        }

        MXMIDIInManager.getManager().initWithSetting();
        MXMIDIOutManager.getManager().initWithSetting();

        _mainWindow = new MXMainWindow(this);
        final MXOpening winLogo = MXOpening.showAsStartup(_mainWindow);
        winLogo.setVisible(true);

        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                _playlistProcess = new MX00Process();
                _inputProcess = new MX10Process();
                _masterkeyProcess = new MX12Process();
                _kontrolProcess =  new MX30Process();
                _layerProcess = new MX40Process();
                _outputProcess = new MX60Process();

                _masterToList = new MXWrapList();
                _masterToList.addNameAndValue("PushBack to " + _inputProcess.getReceiverName(), _inputProcess);
                _masterToList.addNameAndValue( _kontrolProcess.getReceiverName(), _kontrolProcess);
                _masterToList.addNameAndValue(_layerProcess.getReceiverName(), _layerProcess);
                _masterToList.addNameAndValue(_outputProcess.getReceiverName(), _outputProcess);
                _masterToList.addNameAndValue("Direct Output", FinalMIDIOut.getInstance());

                _inputProcess.setNextReceiver(_kontrolProcess);
                _inputProcess.setMasterKeyProcess(_masterkeyProcess);
                
                _masterkeyProcess.setNextReceiver(_inputProcess);

                _kontrolProcess.setNextReceiver(_layerProcess);
                _layerProcess.setNextReceiver(_outputProcess);
                _outputProcess.setNextReceiver(FinalMIDIOut.getInstance());

                _masterkeyProcess.readSettings();
                _playlistProcess.readSettings();
                _inputProcess.readSettings();
                _outputProcess.readSettings();
                _layerProcess.readSettings();
                
                _mainWindow.setEnabled(false);
                _mainWindow.addWindowListener(new WindowListener() {
                    @Override
                    public void windowOpened(WindowEvent e) {
                    }

                    @Override
                    public void windowClosing(WindowEvent e) {
                        MXMIDIInManager.getManager().closeAll();
                        MXMIDIOutManager.getManager().closeAll();
                    }

                    @Override
                    public void windowClosed(WindowEvent e) {
                    }

                    @Override
                    public void windowIconified(WindowEvent e) {
                    }

                    @Override
                    public void windowDeiconified(WindowEvent e) {
                    }

                    @Override
                    public void windowActivated(WindowEvent e) {
                    }

                    @Override
                    public void windowDeactivated(WindowEvent e) {
                    }
                });

                ArrayList<MXReceiver> reList = new ArrayList();
                reList.add(_playlistProcess);
                reList.add(_inputProcess);
                //reList.add(_velocityProcess);
                reList.add(_kontrolProcess);
                reList.add(_layerProcess);
                reList.add(_outputProcess);

                _mainWindow.initLatebind(reList);
                _mainWindow.setVisible(true);

                _mainWindow.setEnabled(true);
                winLogo.setVisible(false);

                _masterkeyProcess.showAsWindow(MXMain._main.getMainWindow(), false);
            }
         });                    
    }
    
    private MXWrapList<MXReceiver> _masterToList = null;
    
    public MXWrapList<MXReceiver> getMasterToList() {
        return _masterToList;
    }

    public void saveEverySettingToFile() {
        MXSetting.saveEverySettingToFile();
    }
    
    public void exitApp() {
        // exit with event
        _mainWindow.dispatchEvent(new WindowEvent(_mainWindow, WindowEvent.WINDOW_CLOSING));
        /* following codes are unsafety
            _mainWindow.dispose();
            _mainWindow.setVisible(false);
            System.exit(0);
        */
    }
}
