import javax.sound.midi.*;
//
// MIDI specification data
//
public class MIDISpec {
  ///////////////////////////////////////////////////////////////////
  //
  // MIDI Channel
  public static final int	MAX_CHANNELS = 16;
  //
  ///////////////////////////////////////////////////////////////////
  //
  // Pitch Bend
  public static final int	PITCH_BEND_NONE = 8192;
  //
  ///////////////////////////////////////////////////////////////////
  //
  // Meta Message
  //
  public static String getMetaName( int message_type ) {
    if( message_type < 0x10 ) {
      return meta_message_type_names[message_type];
    }
    switch( message_type ) {
      case 0x20: return "MIDI Ch.Prefix";
      case 0x21: return "MIDI Output Port";
      case 0x2F: return "End Of Track";
      case 0x51: return "Tempo";
      case 0x54: return "SMPTE Offset";
      case 0x58: return "Time Signature";
      case 0x59: return "Key Signature";
      case 0x7F: return "Sequencer Specific";
    }
    return null;
  }
  public static boolean hasMetaText( int message_type ) {
    return (message_type > 0 && message_type < 10);
  }
  public static boolean isEOT( int meta_type ) {
    return meta_type == 0x2F;
  }
  public static boolean isEOT( MidiMessage midi_msg ) {
    return (midi_msg instanceof MetaMessage) ?
      isEOT( ((MetaMessage)midi_msg).getType() ) : false;
  }
  public static boolean isEOT( MidiEvent midi_evt ) {
    return isEOT(midi_evt.getMessage());
  }
  public static final int MICROSECOND_PER_MINUTE = (60 * 1000 * 1000);
  public static int byteArrayToQpmTempo( byte[] b ) {
      int tempo_in_us_per_quarter
        = ((b[0] & 0xFF) << 16) | ((b[1] & 0xFF) << 8) | (b[2] & 0xFF);
      return MICROSECOND_PER_MINUTE / tempo_in_us_per_quarter;
  }
  public static byte[] qpmTempoToByteArray( int qpm ) {
      int tempo_in_us_per_quarter = MICROSECOND_PER_MINUTE / qpm;
      byte[] b = new byte[3];
      b[0] = (byte)((tempo_in_us_per_quarter >> 16) & 0xFF);
      b[1] = (byte)((tempo_in_us_per_quarter >> 8) & 0xFF);
      b[2] = (byte)(tempo_in_us_per_quarter & 0xFF);
      return b;
  }
  public static String getNameOf( Track track ) {
    MidiEvent midi_event;
    MidiMessage msg;
    MetaMessage meta_msg;
    for( int i=0; i<track.size(); i++ ) {
      midi_event = track.get(i);
      if( midi_event.getTick() > 0 ) { // No more event at top, try next track
        break;
      }
      msg = midi_event.getMessage();
      if( ! (msg instanceof MetaMessage) ) { // Not meta message
        continue;
      }
      meta_msg = (MetaMessage)msg;
      if( meta_msg.getType() != 0x03 ) { // Not sequence name
        continue;
      }
      return new String(meta_msg.getData());
    }
    return null;
  }
  public static boolean setNameOf( Track track, String name ) {
    MidiEvent midi_event = null;
    MidiMessage msg = null;
    MetaMessage meta_msg = null;
    for( int i=0; i<track.size(); i++ ) {
      if(
        (midi_event = track.get(i)).getTick() > 0
        ||
        (msg = midi_event.getMessage()) instanceof MetaMessage
        &&
        (meta_msg = (MetaMessage)msg).getType() == 0x03
      ) {
        break;
      }
      meta_msg = null;
    }
    if( meta_msg == null ) {
      if( name.isEmpty() ) return false;
      track.add(new MidiEvent(
        (MidiMessage)(meta_msg = new MetaMessage()), 0
      ));
    }
    byte ub[] = name.getBytes();
    try {
      meta_msg.setMessage( 0x03, ub, ub.length );
    }
    catch( InvalidMidiDataException e ) {
      e.printStackTrace();
      return false;
    }
    return true;
  }
  public static String getNameOf( Sequence seq ) {
    // Returns name of the MIDI sequence.
    // A sequence name is placed at top of first track of the sequence.
    //
    Track tracks[] = seq.getTracks();
    String s;
    for( Track track : tracks )
      if( (s = getNameOf(track)) != null )
        return s;
    return null;
  }
  public static boolean setNameOf( Sequence seq, String name ) {
    Track tracks[] = seq.getTracks();
    for( Track track : tracks )
      if( setNameOf(track,name) ) return true;
    return false;
  }
  private static final String meta_message_type_names[] = {
    "Seq Number", "Text", "Copyright", "Seq/Track Name",
    "Instrument Name", "Lyric", "Marker","Cue Point",
    "Program Name", "Device Name", null, null,
    null, null, null, null
  };
  ///////////////////////////////////////////////////////////////////
  //
  // Channel Message / System Message
  //
  public static String getStatusName( int status ) {
    if( status < 0x80 ) {
      // No such status
      return null;
    }
    else if ( status < 0xF0 ) {
      // Channel Message
      return ch_msg_status_names[ (status >> 4) - 0x08 ];
    }
    else if ( status <= 0xFF ) {
      // System Message
      return sys_msg_names[ status - 0xF0 ];
    }
    return null;
  }
  public static boolean isChannelMessage( ShortMessage msg ) {
    return isChannelMessage( msg.getStatus() );
  }
  public static boolean isChannelMessage( int status ) {
    return ( status < 0xF0 && status >= 0x80 );
  }
  private static final String ch_msg_status_names[] = {
    // 0x80 - 0xE0 : Channel Voice Message
    // 0xB0 : Channel Mode Message
    "NoteOFF", "NoteON",
    "Polyphonic Key Pressure", "Ctrl/Mode",
    "Program", "Ch.Pressure", "Pitch Bend"
  };
  private static final String sys_msg_names[] = {
    // 0xF0 : System Exclusive
    "SysEx", 
    //
    // 0xF1 - 0xF7 : System Common Message
    "MIDI Time Code Quarter Frame",
    "Song Position Pointer", "Song Select",
    null, null, "Tune Request", "Special SysEx",
    //
    // 0xF8 - 0xFF : System Realtime Message
    // 0xFF : Meta Message (SMF only, Not for wired MIDI message)
    "Timing Clock", null, "Start", "Continue",
    "Stop", null, "Active Sensing", "Meta / Sys.Reset",
  };
  ///////////////////////////////////////////////////////////////////
  //
  // Control Change / Channel Mode Message
  //
  public static String getControllerName( int controller_number ) {
    if( controller_number < 0x00 ) {
      return null;
    }
    else if( controller_number < 0x20 ) {
      String s = controller_names_0[controller_number];
      if( s != null ) s += " (MSB)";
      return s;
    }
    else if( controller_number < 0x40 ) {
      String s = controller_names_0[controller_number - 0x20];
      if( s != null ) s += " (LSB)";
      return s;
    }
    else if( controller_number < 0x78 ) {
      return controller_momentary_switch_names[controller_number - 0x40];
    }
    else if( controller_number < 0x80 ) {
      return controller_mode_message_names[controller_number - 0x78];
    }
    else {
      return null;
    }
  }
  private static final String controller_names_0[] = {
    //
    // 0x00-0x0F (MSB)
    "Bank Select", "Modulation Depth", "Breath Controller", null,
    "Foot Controller", "Portamento Time", "Data Entry", "Volume",
    "Balance", null, "Pan", "Expression",
    "Effect Control 1", "Effect Control 2", null, null,
    //
    // 0x10-0x1F (MSB)
    "General Purpose 1", "General Purpose 2",
    "General Purpose 3", "General Purpose 4",
    null, null, null, null,
    null, null, null, null,
    null, null, null, null,
    //
    // 0x20-0x3F (LSB)
  };
  private static final String controller_momentary_switch_names[] = {
    //
    // 0x40-0x4F
    "Damper Pedal (Sustain)", "Portamento",
    "Sustenuto", "Soft Pedal",
    "Legato Footswitch", "Hold 2",
    "Sound Controller 1 (Sound Variation)",
    "Sound Controller 2 (Timbre/Harmonic Intens)",
    "Sound Controller 3 (Release Time)",
    "Sound Controller 4 (Attack Time)",
    "Sound Controller 5 (Brightness)",
    "Sound Controller 6 (Decay Time)",
    "Sound Controller 7 (Vibrato Rate)",
    "Sound Controller 8 (Vibrato Depth)", 
    "Sound Controller 9 (Vibrato Delay)",
    "Sound Controller 10 (Undefined)",
    //
    // 0x50-0x5F
    "General Purpose 5", "General Purpose 6 (Temp Change)",
    "General Purpose 7", "General Purpose 8",
    "Portamento Control", null, null, null,
    null, null, null, "Reverb (Ext.Effects Depth)",
    "Tremelo Depth", "Chorus Depth",
    "Celeste (Detune) Depth", "Phaser Depth",
    //
    // 0x60-0x6F
    "Data Increment", "Data Decrement",
    "NRPN (LSB)", "NRPN (MSB)",
    "RPN (LSB)", "RPN (MSB)", null, null,
    null, null, null, null,
    null, null, null, null,
    //
    // 0x70-0x77
    null, null, null, null,
    null, null, null, null
  };
  private static final String controller_mode_message_names[] = {
    // 0x78-0x7F
    "All Sound OFF", "Reset All Controllers",
    "Local Control", "All Notes OFF",
    "Omni Mode OFF", "Omni Mode ON",
    "Mono Mode ON", "Poly Mode ON"
  };
  ///////////////////////////////////////////////////////////////////
  //
  // System Exclusive
  //
  public static String getSysExManufacturerName( int id ) {
    switch( id ) {
    //
    // Manufacturer
    case 0x40: return "KAWAI";
    case 0x41: return "Roland";
    case 0x42: return "KORG";
    case 0x43: return "YAMAHA";
    case 0x44: return "CASIO";
    //
    // Special
    case 0x7D: return "Non-Commercial";
    case 0x7E: return "Universal: Non-RealTime";
    case 0x7F: return "Universal: RealTime";
    //
    default: return null;
    }
  }
  ///////////////////////////////////////////////////////////////////
  //
  // Note Number
  //
  public static final int	MAX_NOTE_NO = 127;
  //
  // GM Instrument family names
  //
  public static final String instrument_family_names[] = {

    "Piano",
    "Chrom.Percussion",
    "Organ",
    "Guitar",
    "Bass",
    "Strings",
    "Ensemble",
    "Brass",

    "Reed",
    "Pipe",
    "Synth Lead",
    "Synth Pad",
    "Synth Effects",
    "Ethnic",
    "Percussive",
    "Sound Effects",
  };
  //
  // GM Instrument names
  //
  public static final String instrument_names[] = {
"Acoustic Grand Piano",
"Bright Acoustic Piano",
"Electric Grand Piano",
"Honky-tonk Piano",
"Electric Piano 1",
"Electric Piano 2",
"Harpsichord",
"Clavi",
"Celesta",
"Glockenspiel",
"Music Box",
"Vibraphone",
"Marimba",
"Xylophone",
"Tubular Bells",
"Dulcimer",
"Drawbar Organ",
"Percussive Organ",
"Rock Organ",
"Church Organ",
"Reed Organ",
"Accordion",
"Harmonica",
"Tango Accordion",
"Acoustic Guitar (nylon)",
"Acoustic Guitar (steel)",
"Electric Guitar (jazz)",
"Electric Guitar (clean)",
"Electric Guitar (muted)",
"Overdriven Guitar",
"Distortion Guitar",
"Guitar harmonics",
"Acoustic Bass",
"Electric Bass (finger)",
"Electric Bass (pick)",
"Fretless Bass",
"Slap Bass 1",
"Slap Bass 2",
"Synth Bass 1",
"Synth Bass 2",
"Violin",
"Viola",
"Cello",
"Contrabass",
"Tremolo Strings",
"Pizzicato Strings",
"Orchestral Harp",
"Timpani",
"String Ensemble 1",
"String Ensemble 2",
"SynthStrings 1",
"SynthStrings 2",
"Choir Aahs",
"Voice Oohs",
"Synth Voice",
"Orchestra Hit",
"Trumpet",
"Trombone",
"Tuba",
"Muted Trumpet",
"French Horn",
"Brass Section",
"SynthBrass 1",
"SynthBrass 2",
"Soprano Sax",
"Alto Sax",
"Tenor Sax",
"Baritone Sax",
"Oboe",
"English Horn",
"Bassoon",
"Clarinet",
"Piccolo",
"Flute",
"Recorder",
"Pan Flute",
"Blown Bottle",
"Shakuhachi",
"Whistle",
"Ocarina",
"Lead 1 (square)",
"Lead 2 (sawtooth)",
"Lead 3 (calliope)",
"Lead 4 (chiff)",
"Lead 5 (charang)",
"Lead 6 (voice)",
"Lead 7 (fifths)",
"Lead 8 (bass + lead)",
"Pad 1 (new age)",
"Pad 2 (warm)",
"Pad 3 (polysynth)",
"Pad 4 (choir)",
"Pad 5 (bowed)",
"Pad 6 (metallic)",
"Pad 7 (halo)",
"Pad 8 (sweep)",
"FX 1 (rain)",
"FX 2 (soundtrack)",
"FX 3 (crystal)",
"FX 4 (atmosphere)",
"FX 5 (brightness)",
"FX 6 (goblins)",
"FX 7 (echoes)",
"FX 8 (sci-fi)",
"Sitar",
"Banjo",
"Shamisen",
"Koto",
"Kalimba",
"Bag pipe",
"Fiddle",
"Shanai",
"Tinkle Bell",
"Agogo",
"Steel Drums",
"Woodblock",
"Taiko Drum",
"Melodic Tom",
"Synth Drum",
"Reverse Cymbal",
"Guitar Fret Noise",
"Breath Noise",
"Seashore",
"Bird Tweet",
"Telephone Ring",
"Helicopter",
"Applause",
"Gunshot",
  };
  //
  // GM percussion names
  //
  public static final int	MIN_PERCUSSION_NO = 35;
  public static String getPercussionName(int note_no) {
      int i = note_no - MIN_PERCUSSION_NO ;
      return i>=0 && i < percussion_names.length ? percussion_names[i] : "(Unknown)" ;
  }
  public static final String	percussion_names[] = {
"Acoustic Bass Drum",
"Bass Drum 1",
"Side Stick",
"Acoustic Snare",
"Hand Clap",
"Electric Snare",
"Low Floor Tom",
"Closed Hi Hat",
"High Floor Tom",
"Pedal Hi-Hat",
"Low Tom",
"Open Hi-Hat",
"Low-Mid Tom",
"Hi Mid Tom",
"Crash Cymbal 1",
"High Tom",
"Ride Cymbal 1",
"Chinese Cymbal",
"Ride Bell",
"Tambourine",
"Splash Cymbal",
"Cowbell",
"Crash Cymbal 2",
"Vibraslap",
"Ride Cymbal 2",
"Hi Bongo",
"Low Bongo",
"Mute Hi Conga",
"Open Hi Conga",
"Low Conga",
"High Timbale",
"Low Timbale",
"High Agogo",
"Low Agogo",
"Cabasa",
"Maracas",
"Short Whistle",
"Long Whistle",
"Short Guiro",
"Long Guiro",
"Claves",
"Hi Wood Block",
"Low Wood Block",
"Mute Cuica",
"Open Cuica",
"Mute Triangle",
"Open Triangle",
  };
}
