/*
 *	Chord Button Matrix class for MIDI Chord Helper
 *
 *	Copyright (C) 2004-2009 Akiyoshi Kamide
 *	http://www.yk.rim.or.jp/~kamide/music/chordhelper/
 */
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

interface ChordMatrixListener extends EventListener {
  void chordChanged();
  void keySignatureChanged();
}
public class ChordMatrix extends JPanel
  implements
    MouseListener, KeyListener,
    MouseMotionListener, MouseWheelListener
{
  public static final int	N_COLUMNS = 25;
  public static final int	ROW_SUS4 = 0;
  public static final int	ROW_MAJOR = 1;
  public static final int	ROW_MINOR = 2;
  public static final int	CHORD_BUTTON_ROWS = 3;
  public Co5Label keysig_labels[] = new Co5Label[ N_COLUMNS ];
  public ChordLabel
    chord_labels[] = new ChordLabel[
      N_COLUMNS * CHORD_BUTTON_ROWS
    ];
  public ChordGuide	chord_guide;
  //
  ChordDisplay
    chord_display = new ChordDisplay("Chord Pad", this, null);
  //
  // Capo
  CapoComboBoxModel capo_value_model = new CapoComboBoxModel();
  CapoSelecter	capo_selecter = new CapoSelecter(capo_value_model);
  //
  private Music.Key	key = null;
  private Music.Key	capo_key = null;
  private boolean	is_dark = false;
  private EventListenerList listenerList = new EventListenerList();

  private Music.Chord	selected_chord = null;
  private Music.Chord	selected_chord_capo = null;
  private ChordLabel	selected_chord_label = null;
  private ChordLabel	destination_chord_label = null;
  private int		selected_note_index = -1;

  private byte		timesig_upper = 4;
  private byte		current_beat = 0;
  private boolean	is_playing = false;
  //
  // For PC keyboard operation
  //
  private int pc_key_next_shift7 = Music.Chord.ROOT;

  class ColorSet {
    boolean is_dark = false;
    Color[] focus = new Color[2];	// 0:lost 1:gained
    Color[] foregrounds = new Color[2];	// 0:unselected 1:selected
    Color[] backgrounds = new Color[4]; // 0:remote 1:left 2:local 3:right
    Color[] indicators = new Color[3];	// 0:natural 1:sharp 2:flat
    public ColorSet() { this(false); }
    public ColorSet(boolean is_dark) {
      this.is_dark = is_dark;
    }
  }
  ColorSet
    normal_mode_colorset, dark_mode_colorset, current_colorset;

  class ChordLabelSelection {
    ChordLabel chord_label;
    int bit_index;
    boolean is_sus4;
    public ChordLabelSelection( ChordLabel chord_label, int bit_index ) {
      this.chord_label = chord_label;
      this.bit_index = bit_index;
      this.is_sus4 = chord_label.is_sus4;
    }
    public void setCheckBit( boolean is_on ) {
      chord_label.setCheckBit( is_on, bit_index );
    }
    public boolean setBassCheckBit( boolean is_on ) {
      if( bit_index == 0 && ! is_sus4 ) {
        chord_label.setCheckBit( is_on, 6 );
        return true;
      }
      else {
        return false;
      }
    }
  }
  class ChordLabelSelections {
    int note_no;
    int weight = 0;
    int bass_weight = 0;
    boolean is_active = false;
    boolean is_bass_active = false;
    private ChordLabelSelection acls[];
    public ChordLabelSelections(int note_no, ArrayList<ChordLabelSelection> al) {
      this.note_no = note_no;
      acls = al.toArray(new ChordLabelSelection[al.size()]);
    }
    void addWeight(int weight_diff) {
      if( (weight += weight_diff) < 0 ) weight = 0;
      if( (weight > 0) != is_active ) {
        is_active = !is_active;
        for( ChordLabelSelection cls : acls ) {
          cls.setCheckBit(is_active);
        }
      }
    }
    void addBassWeight(int weight_diff) {
      if( (bass_weight += weight_diff) < 0 ) bass_weight = 0;
      if( (bass_weight > 0) != is_bass_active ) {
        is_bass_active = !is_bass_active;
        for( ChordLabelSelection cls : acls ) {
          if( ! cls.setBassCheckBit(is_bass_active) ) {
            // No more root major/minor
            break;
          }
        }
      }
      addWeight(weight_diff);
    }
    void clearWeight() {
      weight = bass_weight = 0;
      is_active = is_bass_active = false;
      for( ChordLabelSelection cls : acls ) {
        cls.setCheckBit(false);
        cls.setBassCheckBit(false);
      }
    }
  }
  ChordLabelSelections
    chord_label_selections[] = new ChordLabelSelections[12];
  //
  // Key signature button
  //
  class Co5Label extends JLabel {
    public boolean is_selected = false;
    public int co5_value = 0;
    private Color indicator_color;
    public Co5Label(int v) {
      Music.Key key = new Music.Key(co5_value = v);
      setOpaque(true);
      setBackground(false);
      setForeground( current_colorset.foregrounds[0] );
      setHorizontalAlignment( JLabel.CENTER );
      String tip = "Key signature: ";
      if( v != key.toCo5() ) {
        tip += "out of range" ;
      }
      else {
        tip += key.signatureDescription() + " " +
          key.toStringIn(Music.NoteSymbol.NOTE_IN_JAPANESE);
        if( v == 0 ) {
          setIcon(new ButtonIcon(ButtonIcon.NATURAL_ICON));
        }
        else {
          setFont( getFont().deriveFont(Font.PLAIN) );
          setText( key.signature() );
        }
      }
      setToolTipText(tip);
    }
    public void paint(Graphics g) {
      super.paint(g);
      Dimension d = getSize();
      if( ChordMatrix.this.isFocusOwner() && is_selected ) {
        g.setColor( current_colorset.focus[1] );
        g.drawRect( 0, 0, d.width-1, d.height-1 );
      }
      if( !is_selected || !is_playing || current_beat+1 == timesig_upper ) {
        return;
      }
      if( current_beat == 0 ) {
        g.setColor( indicator_color );
        g.drawRect( 2, 2, d.width-5, d.height-5 );
        g.setColor( is_dark ? indicator_color.darker() : indicator_color.brighter() );
        g.drawRect( 0, 0, d.width-1, d.height-1 );
        return;
      }
      Color color = current_colorset.indicators[0];
      g.setColor( color );
      if( current_beat == 1 ) {
        //
        // ||__ii
        g.drawLine( 2, d.height-3, d.width-3, d.height-3 );
        g.drawLine( d.width-3, d.height*3/4, d.width-3, d.height-3 );
        g.drawLine( 2, 2, 2, d.height-3 );
        g.setColor( is_dark ? color.darker() : color.brighter() );
        g.drawLine( 0, d.height-1, d.width-1, d.height-1 );
        g.drawLine( d.width-1, d.height*3/4, d.width-1, d.height-1 );
        g.drawLine( 0, 0, 0, d.height-1 );
      }
      else {
        //
        // ii__
        //
        int vertical_top = (d.height-1) * (current_beat-1) / (timesig_upper-2) ;
        g.drawLine( 2, vertical_top == 0 ? 2 : vertical_top, 2, d.height-3 );
        g.setColor( is_dark ? color.darker() : color.brighter() );
        g.drawLine( 0, vertical_top, 0, d.height-1 );
      }
    }
    public void setBackground(boolean is_active) {
      super.setBackground(
        current_colorset.backgrounds[ is_active ? 2 : 0 ]
      );
      setIndicatorColor();
      setOpaque(true);
    }
    public void setSelection(boolean is_selected) {
      this.is_selected = is_selected;
      setSelection();
    }
    public void setSelection() {
      setForeground(
        current_colorset.foregrounds[ this.is_selected ? 1 : 0 ]
      );
    }
    public void setIndicatorColor() {
      if( co5_value < 0 ) {
        indicator_color = current_colorset.indicators[2];
      }
      else if( co5_value > 0 ) {
        indicator_color = current_colorset.indicators[1];
      }
      else {
        indicator_color = current_colorset.foregrounds[1];
      }
    }
  }
  //
  // Chord button
  //
  class ChordLabel extends JLabel {
    public byte check_bits = 0;
    public int co5_value;
    public boolean is_minor;
    public boolean is_sus4;
    public boolean is_selected = false;
    public Music.Chord chord;

    private boolean in_active_zone = true;
    private Font bold_font, plain_font;
    private int indicator_color_indices[] = new int[5];
    private byte indicator_bits = 0;

    public ChordLabel( Music.Chord chord ) {
      this.chord = chord;
      is_minor = chord.isMinor();
      is_sus4 = chord.isSus4();
      co5_value = chord.rootNoteSymbol().toCo5();
      if( is_minor ) co5_value -= 3;
      String label_string = ( is_sus4 ? chord.symbolSuffix() : chord.toString() );
      if( is_minor && label_string.length() > 3 ) {
        float small_point_size = getFont().getSize2D() - 2;
        bold_font = getFont().deriveFont(Font.BOLD, small_point_size);
        plain_font = getFont().deriveFont(Font.PLAIN, small_point_size);
      }
      else {
        bold_font = getFont().deriveFont(Font.BOLD);
        plain_font = getFont().deriveFont(Font.PLAIN);
      }
      setOpaque(true);
      setBackground(0);
      setForeground( current_colorset.foregrounds[0] );
      setBold(false);
      setHorizontalAlignment( JLabel.CENTER );
      setText( label_string );
      setToolTipText( "Chord: " + chord.toName() );
    }
    public void paint(Graphics g) {
      super.paint(g);
      Dimension d = getSize();
      Graphics2D g2 = (Graphics2D) g;
      Color color = null;

      if( ! in_active_zone ) {
        g2.setColor( Color.gray );
      }

      if( (indicator_bits & 32) != 0 ) {
        //
        // Draw square  []  with 3rd/sus4th note color
        //
        if( in_active_zone ) {
          color = current_colorset.indicators[indicator_color_indices[1]];
          g2.setColor( color );
        }
        g2.drawRect( 0, 0, d.width-1, d.height-1 );
        g2.drawRect( 2, 2, d.width-5, d.height-5 );
      }
      if( (indicator_bits & 1) != 0 ) {
        //
        // Draw  ||__  with root note color
        //
        if( in_active_zone ) {
          color = current_colorset.indicators[indicator_color_indices[0]];
          g2.setColor( color );
        }
        g2.drawLine( 0, 0, 0, d.height-1 );
        g2.drawLine( 2, 2, 2, d.height-3 );
      }
      if( (indicator_bits & 64) != 0 ) {
        // Draw bass mark with root note color
        //
        if( in_active_zone ) {
          color = current_colorset.indicators[indicator_color_indices[0]];
          g2.setColor( color );
        }
        g2.fillRect( 6, d.height-7, d.width-12, 2 );
      }
      if( (indicator_bits & 4) != 0 ) {
        //
        // Draw short  __ii  with parfect 5th color
        //
        if( in_active_zone ) {
          color = current_colorset.indicators[indicator_color_indices[2]];
          g2.setColor( color );
        }
        g2.drawLine( d.width-1, d.height*3/4, d.width-1, d.height-1 );
        g2.drawLine( d.width-3, d.height*3/4, d.width-3, d.height-3 );
      }
      if( (indicator_bits & 2) != 0 ) {
        //
        // Draw  __  with 3rd note color
        //
        if( in_active_zone ) {
          color = current_colorset.indicators[indicator_color_indices[1]];
          g2.setColor( color );
        }
        g2.drawLine( 0, d.height-1, d.width-1, d.height-1 );
        g2.drawLine( 2, d.height-3, d.width-3, d.height-3 );
      }
      if( (indicator_bits & 8) != 0 ) {
        //
        // Draw circle with diminished 5th color
        //
        if( in_active_zone ) {
          g2.setColor( current_colorset.indicators[indicator_color_indices[3]] );
        }
        g2.drawOval( 1, 1, d.width-2, d.height-2 );
      }
      if( (indicator_bits & 16) != 0 ) {
        //
        // Draw + with augument 5th color
        //
        if( in_active_zone ) {
          g2.setColor( current_colorset.indicators[indicator_color_indices[4]] );
        }
        g2.drawLine( 1, 3, d.width-3, 3 );
        g2.drawLine( 1, 4, d.width-3, 4 );
        g2.drawLine( d.width/2-1, 0, d.width/2-1, 7 );
        g2.drawLine( d.width/2, 0, d.width/2, 7 );
      }
    }
    public void clearCheckBit() {
      check_bits = indicator_bits = 0;
    }
    public void setCheckBit( boolean is_on, int bit_index ) {
      //
      // Check bits: x6x43210
      //   6:BassRoot
      //   4:Augumented5th, 3:Diminished5th, 2:Parfect5th,
      //   1:Major3rd/minor3rd/sus4th, 0:Root
      //
      byte mask = ((byte)(1<<bit_index));
      byte old_check_bits = check_bits;
      if( is_on ) {
        check_bits |= mask;
      }
      else {
        check_bits &= ~mask;
      }
      if( old_check_bits == check_bits ) {
        // No bits changed
        return;
      }
      // Indicator bits: x6543210	6:Bass||_  5:[]  4:+  3:O  2:_ii  1:__  0:||_
      //
      byte indicator_bits = 0;
      if( (check_bits & 1) != 0 ) {
        if( (check_bits & 7) == 7 ) { // All triad notes appared
          //
          // Draw square
          indicator_bits |= 0x20;
          //
          // Draw different-colored vertical lines
          if( indicator_color_indices[0] != indicator_color_indices[1] ) {
            indicator_bits |= 1;
          }
          if( indicator_color_indices[2] != indicator_color_indices[1] ) {
            indicator_bits |= 4;
          }
        }
        else if( !is_sus4 ) {
          //
          // Draw vertical lines  || ii
          indicator_bits |= 5;
          //
          if( (check_bits & 2) != 0 && (!is_minor || (check_bits & 0x18) != 0) ) {
            //
            // Draw horizontal bottom lines __
            indicator_bits |= 2;
          }
        }
        if( !is_sus4 ) {
          if( is_minor || (check_bits & 2) != 0 ) {
            indicator_bits |= (byte)(check_bits & 0x18);  // Copy bit 3 and bit 4
          }
          if( (check_bits & 0x40) != 0 ) {
            indicator_bits |= 0x40; // Bass
          }
        }
      }
      if( this.indicator_bits == indicator_bits ) {
        // No shapes changed
        return;
      }
      this.indicator_bits = indicator_bits;
      repaint();
    }
    public void setBackground(int i) {
      switch( i ) {
        case  0: 
        case  1:  
        case  2:  
        case  3: 
          super.setBackground( current_colorset.backgrounds[i] );
          setOpaque(true);
          break; 
        default: return;
      }
    }
    public void setSelection(boolean is_selected) {
      this.is_selected = is_selected;
      setSelection();
    }
    public void setSelection() {
      setForeground( current_colorset.foregrounds[ this.is_selected ? 1 : 0 ] );
    }
    public void setBold(boolean is_bold) {
      setFont( is_bold ? bold_font : plain_font );
    }
    public void keyChanged() {
      int co5_key = capo_key.toCo5();
      int co5_offset = co5_value - co5_key;
      in_active_zone = (co5_offset <= 6 && co5_offset >= -6) ;
      int root_note = chord.rootNote();
      //
      // Reconstruct color index
      //
      // Root
      indicator_color_indices[0] = Music.isOnScale(
        root_note, co5_key
      ) ? 0 : co5_offset > 0 ? 1 : 2;
      //
      // 3rd / sus4
      indicator_color_indices[1] = Music.isOnScale(
        root_note+(is_minor?3:is_sus4?5:4), co5_key
      ) ? 0 : co5_offset > 0 ? 1 : 2;
      //
      // P5th
      indicator_color_indices[2] = Music.isOnScale(
        root_note+7, co5_key
      ) ? 0 : co5_offset > 0 ? 1 : 2;
      //
      // dim5th
      indicator_color_indices[3] = Music.isOnScale(
        root_note+6, co5_key
      ) ? 0 : co5_offset > 4 ? 1 : 2;
      //
      // aug5th
      indicator_color_indices[4] = Music.isOnScale(
        root_note+8, co5_key
      ) ? 0 : co5_offset > -3 ? 1 : 2;
    }
  }

  public ChordMatrix() {
    //
    // Define colors
    //
    normal_mode_colorset = new ColorSet();
    normal_mode_colorset.foregrounds[0] = null;
    normal_mode_colorset.foregrounds[1] = new Color(0xFF,0x3F,0x3F);
    normal_mode_colorset.backgrounds[0] = new Color(0xCF,0xFF,0xCF);
    normal_mode_colorset.backgrounds[1] = new Color(0x9F,0xFF,0xFF);
    normal_mode_colorset.backgrounds[2] = new Color(0xFF,0xCF,0xCF);
    normal_mode_colorset.backgrounds[3] = new Color(0xFF,0xFF,0x9F);
    normal_mode_colorset.indicators[0] = new Color(0xFF,0x3F,0x3F);
    normal_mode_colorset.indicators[1] = new Color(0xCF,0x6F,0x00);
    normal_mode_colorset.indicators[2] = new Color(0x3F,0x3F,0xFF);
    normal_mode_colorset.focus[0] = null;
    normal_mode_colorset.focus[1] = getBackground().darker();
    //
    dark_mode_colorset = new ColorSet(true);
    dark_mode_colorset.foregrounds[0] = Color.gray.darker();
    dark_mode_colorset.foregrounds[1] = Color.pink.brighter();
    dark_mode_colorset.backgrounds[0] = Color.black;
    dark_mode_colorset.backgrounds[1] = new Color(0x00,0x18,0x18);
    dark_mode_colorset.backgrounds[2] = new Color(0x20,0x00,0x00);
    dark_mode_colorset.backgrounds[3] = new Color(0x18,0x18,0x00);
    dark_mode_colorset.indicators[0] = Color.pink;
    dark_mode_colorset.indicators[1] = Color.yellow;
    dark_mode_colorset.indicators[2] = Color.cyan;
    dark_mode_colorset.focus[0] = Color.black;
    dark_mode_colorset.focus[1] = getForeground().brighter();
    //
    current_colorset = normal_mode_colorset;
    //
    chord_guide = new ChordGuide(this);
    //
    int i, v;
    Dimension button_size = new Dimension(28,26);
    //
    // Make key-signature labels and chord labels
    Co5Label l;
    for (i=0, v= -12; i<N_COLUMNS; i++, v++) {
      l = new Co5Label(v);
      l.addMouseListener(this);
      l.addMouseMotionListener(this);
      add( keysig_labels[i] = l );
      l.setPreferredSize(button_size);
    }
    int row_no;
    for (i=0; i < N_COLUMNS * CHORD_BUTTON_ROWS; i++) {
      row_no = i / N_COLUMNS;
      v = i - (N_COLUMNS * row_no) - 12;
      Music.Chord chord = new Music.Chord(
        row_no==2 ? v+3 : v
      );
      if( row_no==0 ) chord.setSus4();
      else if( row_no==2 ) chord.setMinorThird();
      ChordLabel cl = new ChordLabel(chord);
      cl.addMouseListener(this);
      cl.addMouseMotionListener(this);
      cl.addMouseWheelListener(this);
      add( chord_labels[i] = cl );
      cl.setPreferredSize(button_size);
    }
    setFocusable(true);
    setOpaque(true);
    addKeyListener(this);
    addFocusListener(new FocusListener() {
      public void focusGained(FocusEvent e) {
        repaint();
      }
      public void focusLost(FocusEvent e) {
        selected_chord = selected_chord_capo = null;
        fireChordChanged();
        repaint();
      }
    });
    setLayout(new GridLayout( 4, N_COLUMNS, 2, 2 ));
    setKeySignature( new Music.Key() );
    //
    // Make chord label selections index
    //
    int note_index;
    ArrayList<ChordLabelSelection> al;
    Music.Chord chord;
    for( int note_no=0; note_no<chord_label_selections.length; note_no++ ) {
      al = new ArrayList<ChordLabelSelection>();
      //
      // Root major/minor chords
      for( ChordLabel cl : chord_labels ) {
        if( ! cl.is_sus4 && cl.chord.indexOf(note_no) == 0 ) {
          al.add(new ChordLabelSelection( cl, 0 )); // Root
        }
      }
      // Root sus4 chords
      for( ChordLabel cl : chord_labels ) {
        if( cl.is_sus4 && cl.chord.indexOf(note_no) == 0 ) {
          al.add(new ChordLabelSelection( cl, 0 )); // Root
        }
      }
      // 3rd,sus4th,5th included chords
      for( ChordLabel cl : chord_labels ) {
        note_index = cl.chord.indexOf(note_no);
        if( note_index == 1 || note_index == 2 ) {
          al.add(new ChordLabelSelection( cl, note_index )); // 3rd,sus4,P5
        }
      }
      // Diminished chords (major/minor chord button only)
      for( ChordLabel cl : chord_labels ) {
        if( cl.is_sus4 ) continue;
        (chord = cl.chord.clone()).setFlattedFifth();
        if( chord.indexOf(note_no) == 2 ) {
          al.add(new ChordLabelSelection( cl, 3 ));
        }
      }
      // Augumented chords (major chord button only)
      for( ChordLabel cl : chord_labels ) {
        if( cl.is_sus4 || cl.is_minor ) continue;
        (chord = cl.chord.clone()).setSharpedFifth();
        if( chord.indexOf(note_no) == 2 ) {
          al.add(new ChordLabelSelection( cl, 4 ));
        }
      }
      chord_label_selections[note_no] = new ChordLabelSelections( note_no, al );
    }
    // Capo
    //
    capo_selecter.checkbox.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) { capoChanged(); }
    });
    capo_selecter.value_selecter.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { capoChanged(); }
    });
  }
  //
  // MouseListener
  public void mousePressed(MouseEvent e) {
    Component obj = e.getComponent();
    if( obj instanceof ChordLabel ) {
      ChordLabel cl = (ChordLabel)obj;
      Music.Chord chord = cl.chord.clone();
      if( (e.getModifiersEx() & e.BUTTON3_DOWN_MASK) != 0 ) {
        if( e.isShiftDown() ) chord.setMajorSeventh();
        else chord.setSeventh();
      }
      else if( e.isShiftDown() ) chord.setSixth();

      if( e.isControlDown() ) chord.setNinth();
      else chord.clearNinth();

      if( e.isAltDown() ) {
        if( cl.is_sus4 ) {
          chord.setMajorThird(); // To cancel sus4
          chord.setSharpedFifth();
        }
        else chord.setFlattedFifth();
      }
      if( selected_chord_label != null ) {
        selected_chord_label.setSelection(false);
      }
      (selected_chord_label = cl).setSelection(true);
      setSelectedChord(chord);
    }
    else if( obj instanceof Co5Label ) {
      int v = ((Co5Label)obj).co5_value;
      if( (e.getModifiersEx() & e.BUTTON3_DOWN_MASK) != 0 ) {
        setKeySignature( new Music.Key(Music.oppositeCo5(v)) );
      }
      else if ( v == key.toCo5() ) {
        //
        // Cancel selected chord
        //
        setSelectedChord( (Music.Chord)null );
      }
      else {
        // Change key
        setKeySignature( new Music.Key(v) );
      }
    }
    requestFocusInWindow();
    repaint();
  }
  public void mouseReleased(MouseEvent e) {
    destination_chord_label = null;
  }
  public void mouseEntered(MouseEvent e) { }
  public void mouseExited(MouseEvent e) { }
  public void mouseClicked(MouseEvent e) { }
  //
  // MouseMotionListener
  public void mouseDragged(MouseEvent e) {
    Component obj = e.getComponent();
    if( obj instanceof ChordLabel ) {
      ChordLabel l_src = (ChordLabel)obj;
      Component obj2 = this.getComponentAt(
        l_src.getX() + e.getX(),
        l_src.getY() + e.getY()
      );
      if( obj2 == this ) {
        //
        // Entered gap between chord buttons - do nothing
        //
        return;
      }
      ChordLabel l_dst = 
        ( obj2 == null ? null :
          (obj2 instanceof ChordLabel ) ? (ChordLabel)obj2 :
         null
        );
      if( l_dst == l_src ) {
        //
        // Returned to original chord button
        //
        destination_chord_label = null;
        return;
      }
      if( destination_chord_label != null ) {
        //
        // Already touched another chord button
        // 
        return;
      }
      Music.Chord chord = l_src.chord.clone();
      if( l_src.is_minor ) {
        if( l_dst == null ) { // Out of chord buttons
          // mM7
          chord.setMajorSeventh();
        }
        else if( l_src.co5_value < l_dst.co5_value ) { // Right
          // m6
          chord.setSixth();
        }
        else { // Left or up from minor to major
          // m7
          chord.setSeventh();
        }
      }
      else if( l_src.is_sus4 ) {
        if( l_dst == null ) { // Out of chord buttons
          return;
        }
        else if( ! l_dst.is_sus4 ) { // Down from sus4 to major
          chord.setMajorThird();
        }
        else if( l_src.co5_value < l_dst.co5_value ) { // Right
          chord.setNinth();
        }
        else { // Left
          // 7sus4
          chord.setSeventh();
        }
      }
      else {
        if( l_dst == null ) { // Out of chord buttons
          return;
        }
        else if( l_dst.is_sus4 ) { // Up from major to sus4
          chord.setNinth();
        }
        else if( l_src.co5_value < l_dst.co5_value ) { // Right
          // M7
          chord.setMajorSeventh();
        }
        else if( l_dst.is_minor ) { // Down from major to minor
          // 6
          chord.setSixth();
        }
        else { // Left
          // 7
          chord.setSeventh();
        }
      }
      if( chord.hasNinth() || (l_src.is_sus4 && (l_dst == null || ! l_dst.is_sus4) ) ) {
        if( (e.getModifiersEx() & e.BUTTON3_DOWN_MASK) != 0 ) {
          if( e.isShiftDown() ) {
            chord.setMajorSeventh();
          }
          else {
            chord.setSeventh();
          }
        }
        else if( e.isShiftDown() ) {
          chord.setSixth();
        }
      }
      else {
        if( e.isControlDown() ) chord.setNinth(); else chord.clearNinth();
      }
      if( e.isAltDown() ) {
        if( l_src.is_sus4 ) {
          chord.setMajorThird();
          chord.setSharpedFifth();
        }
        else {
          chord.setFlattedFifth();
        }
      }
      setSelectedChord(chord);
      destination_chord_label = (l_dst == null ? l_src : l_dst ) ;
    }
    else if( obj instanceof Co5Label ) {
      Co5Label l_src = (Co5Label)obj;
      Component obj2 = this.getComponentAt(
        l_src.getX() + e.getX(),
        l_src.getY() + e.getY()
      );
      if( obj2 == null || !(obj2 instanceof Co5Label) ) {
        return;
      }
      Co5Label l_dst = (Co5Label)obj2;
      int v = l_dst.co5_value;
      if( (e.getModifiersEx() & e.BUTTON3_DOWN_MASK) != 0 ) {
        setKeySignature( new Music.Key(Music.oppositeCo5(v)) );
      }
      else {
        setKeySignature( new Music.Key(v) );
      }
      repaint();
    }
  }
  public void mouseMoved(MouseEvent e) { }
  //
  // MouseWheelListener
  //
  public void mouseWheelMoved(MouseWheelEvent e) {
    if( selected_chord != null ) {
      if( e.getWheelRotation() > 0 ) { // Wheel moved down
        if( --selected_note_index < 0 ) {
          selected_note_index = selected_chord.numberOfNotes() - 1;
        }
      }
      else { // Wheel moved up
        if( ++selected_note_index >= selected_chord.numberOfNotes() ) {
          selected_note_index = 0;
        }
      }
      fireChordChanged();
    }
  }
  //
  // KeyListener
  public void keyPressed(KeyEvent e) {
    int i = -1, i_col = -1, i_row = 1;
    boolean shift_pressed = false; // True if Shift-key pressed or CapsLocked
    char keyChar = e.getKeyChar();
    int keyCode = e.getKeyCode();
    ChordLabel cl = null;
    Music.Chord chord = null;
    int key_co5 = key.toCo5();
    // System.out.println( keyChar + " Pressed on chord matrix" );
    //
    if( (i = "6 ".indexOf(keyChar)) >= 0 ) {
      selected_chord = selected_chord_capo = null;
      fireChordChanged();
      pc_key_next_shift7 = Music.Chord.ROOT;
      return;
    }
    else if( (i = "asdfghjkl;:]".indexOf(keyChar)) >= 0 ) {
      i_col = i + key_co5 + 7;
    }
    else if( (i = "ASDFGHJKL+*}".indexOf(keyChar)) >= 0 ) {
      i_col = i + key_co5 + 7;
      shift_pressed = true;
    }
    else if( (i = "zxcvbnm,./\\".indexOf(keyChar)) >=0 ) {
      i_col = i + key_co5 + 7;
      i_row = 2;
    }
    else if( (i = "ZXCVBNM<>?_".indexOf(keyChar)) >=0 ) {
      i_col = i + key_co5 + 7;
      i_row = 2;
      shift_pressed = true;
    }
    else if( (i = "qwertyuiop@[".indexOf(keyChar)) >= 0 ) {
      i_col = i + key_co5 + 7;
      i_row = 0;
    }
    else if( (i = "QWERTYUIOP`{".indexOf(keyChar)) >= 0 ) {
      i_col = i + key_co5 + 7;
      i_row = 0;
      shift_pressed = true;
    }
    else if( keyChar == '5' ) {
      pc_key_next_shift7 = Music.Chord.MAJOR_SEVENTH; return;
    }
    else if( keyChar == '7' ) {
      pc_key_next_shift7 = Music.Chord.SEVENTH; return;
    }
    // Shift current key-signature
    else if( keyCode == e.VK_LEFT || keyCode == e.VK_KP_LEFT ) {
      // Add a flat
      setKeySignature( new Music.Key(key_co5-1) );
      return;
    }
    else if( keyCode == e.VK_RIGHT || keyCode == e.VK_KP_RIGHT ) {
      // Add a sharp
      setKeySignature( new Music.Key(key_co5+1) );
      return;
    }
    else if( keyCode == e.VK_DOWN || keyCode == e.VK_KP_DOWN ) {
      // Semitone down
      Music.Key key = new Music.Key(key_co5);
      key.transpose(-1);
      setKeySignature(key);
      return;
    }
    else if( keyCode == e.VK_UP || keyCode == e.VK_KP_UP ) {
      // Semitone up
      Music.Key key = new Music.Key(key_co5);
      key.transpose(1);
      setKeySignature(key);
      return;
    }
    if( i < 0 ) // No key char found
      return;
    if( i_col < 0 ) i_col += 12; else if( i_col > N_COLUMNS ) i_col -= 12; 
    cl = chord_labels[i_col + N_COLUMNS * i_row];
    chord = cl.chord.clone();
    if( shift_pressed )
      chord.setSeventh();
    else
      chord.offsets[Music.Chord.SEVENTH_OFFSET] = pc_key_next_shift7; // specify by previous key
    if( e.isAltDown() ) {
      if( cl.is_sus4 ) {
        chord.setMajorThird(); // To cancel sus4
        chord.setSharpedFifth();
      }
      else chord.setFlattedFifth();
    }
    if( e.isControlDown() ) { // Cannot use for ninth ?
      chord.setNinth();
    }
    if( selected_chord_label != null ) clear();
    (selected_chord_label = cl).setSelection(true);
    setSelectedChord(chord);
    pc_key_next_shift7 = Music.Chord.ROOT;
    return;
  }
  public void keyReleased(KeyEvent e) { }
  public void keyTyped(KeyEvent e) { }

  protected void fireChordChanged() {
    Object[] listeners = listenerList.getListenerList();
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i]==ChordMatrixListener.class) {
        ((ChordMatrixListener)listeners[i+1]).chordChanged();
      }
    }
    if( selected_chord == null ) clearIndicators();
  }
  protected void fireKeySignatureChanged() {
    Object[] listeners = listenerList.getListenerList();
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i]==ChordMatrixListener.class) {
        ((ChordMatrixListener)listeners[i+1]).keySignatureChanged();
      }
    }
  }
  private int capo = 0;
  protected void capoChanged() {
    int new_capo = capo_selecter.getCapo();
    if( this.capo == new_capo ) return;
    (this.capo_key = this.key.clone()).transpose(this.capo = new_capo);
    selected_chord_capo = (
      selected_chord == null ? null : selected_chord.clone().transpose(new_capo)
    );
    for( ChordLabel cl : chord_labels ) cl.keyChanged();
    fireKeySignatureChanged();
  }

  // Methods
  //
  public boolean isDragged() {
    return destination_chord_label != null ;
  }
  public void setDarkMode(boolean is_dark) {
    this.is_dark = is_dark;
    current_colorset = (is_dark ? dark_mode_colorset : normal_mode_colorset);
    setBackground( current_colorset.focus[0] );
    Music.Key prev_key = key;
    key = null;
    setKeySignature(prev_key);
    for( int i=0; i < keysig_labels.length; i++ ) keysig_labels[i].setSelection();
    for( int i=0; i <  chord_labels.length; i++ ) chord_labels[i].setSelection();
    chord_guide.setDarkMode( is_dark );
    chord_display.setDarkMode( is_dark );
    Color col = is_dark ? Color.black : null;
    capo_selecter.setBackground( col );
    capo_selecter.value_selecter.setBackground( col );
  }
  public void setPlaying(boolean is_playing) {
    this.is_playing = is_playing;
    repaint();
  }
  public boolean isPlaying() { return is_playing; }
  public void setBeat(byte beat, byte tsu) {
    if( current_beat == beat && timesig_upper == tsu )
      return;
    timesig_upper = tsu;
    current_beat = beat;
    keysig_labels[ key.toCo5() + 12 ].repaint();
  }
  public void note( int channel, boolean is_note_on, int note_no ) {
    int weight_diff = (is_note_on ? 1 : -1);
    ChordLabelSelections cls = chord_label_selections[
      Music.mod12(note_no)
    ];
    if( note_no < 49 ) cls.addBassWeight( weight_diff );
    else cls.addWeight( weight_diff );
  }
  public void clearIndicators() {
    for( int i=0; i<chord_label_selections.length; i++ ) {
      chord_label_selections[i].clearWeight();
    }
    repaint();
  }
  public void addChordMatrixListener(ChordMatrixListener l) {
    listenerList.add(ChordMatrixListener.class, l);
  }
  public void removeChordMatrixListener(ChordMatrixListener l) {
    listenerList.remove(ChordMatrixListener.class, l);
  }
  public Music.Key getKeySignature() { return key; }
  public Music.Key getKeySignatureCapo() { return capo_key; }
  public void setKeySignature( Music.Key key ) {
    if(
      key == null ||
      this.key != null && key.equals(this.key)
    ) return;
    int i;
    // Clear old value
    if( this.key == null ) {
      for( i = 0; i < keysig_labels.length; i++ ) {
        keysig_labels[i].setBackground(false);
      }
    }
    else {
      keysig_labels[this.key.toCo5() + 12].setSelection(false);
      for( i = Music.mod12(this.key.toCo5()); i < N_COLUMNS; i+=12 ) {
        keysig_labels[i].setBackground(false);
      }
    }
    // Set new value
    keysig_labels[i = key.toCo5() + 12].setSelection(true);
    for( i = Music.mod12(key.toCo5()); i < N_COLUMNS; i+=12 ) {
      keysig_labels[i].setBackground(true);
    }
    // Change chord-label's color & font
    int i_color, old_i_color;
    for( ChordLabel cl : chord_labels ) {
      i_color = ((cl.co5_value - key.toCo5() + 31)/3) & 3;
      if( this.key != null ) {
        old_i_color = ((cl.co5_value - this.key.toCo5() + 31)/3) & 3;
        if( i_color != old_i_color ) {
          cl.setBackground(i_color);
        }
      }
      else cl.setBackground(i_color);
      if( !(cl.is_sus4) ) {
        if(
          this.key != null &&
          Music.mod12( cl.co5_value - this.key.toCo5() ) == 0
        )
          cl.setBold(false);
        if( Music.mod12( cl.co5_value - key.toCo5() ) == 0 )
          cl.setBold(true);
      }
    }
    this.capo_key = (this.key = key).clone().transpose(
      capo_selecter.getCapo()
    );
    for( ChordLabel cl : chord_labels ) cl.keyChanged();
    fireKeySignatureChanged();
  }
  public JComponent getSelectedButton() {
    return selected_chord_label;
  }
  public Music.Chord getSelectedChord() {
    return selected_chord;
  }
  public Music.Chord getSelectedChordCapo() {
    return selected_chord_capo;
  }
  public void setSelectedChordCapo( Music.Chord chord ) {
    setNoteIndex(-1); // Cancel arpeggio mode
    selected_chord = (chord == null ? null : chord.clone().transpose(-capo,capo_key));
    selected_chord_capo = chord;
    fireChordChanged();
  }
  public void setSelectedChord( Music.Chord chord ) {
    setNoteIndex(-1); // Cancel arpeggio mode
    selected_chord = chord;
    selected_chord_capo = (chord == null ? null : chord.clone().transpose(capo,key));
    fireChordChanged();
  }
  public void setSelectedChord( String chord_symbol ) {
    Music.Chord chord = null;
    if( chord_symbol != null && ! chord_symbol.isEmpty() ) {
      try {
        chord = new Music.Chord(chord_symbol);
      } catch( IllegalArgumentException ex ) {
        // Ignored
      }
    }
    setSelectedChord(chord);
  }
  public int getNoteIndex() {
    return
      selected_chord == null || selected_note_index < 0 ?
      -1 : selected_note_index;
  }
  public void setNoteIndex(int note_index) {
    selected_note_index = note_index;
  }
  public void clear() {
    if( selected_chord_label != null ) {
      selected_chord_label.setSelection(false);
      selected_chord_label = null;
    }
    selected_chord = null; selected_note_index = -1;
  }
}

///////////////////////////////////////////////////////////////////
//
// Chord guide label
//
///////////////////////////////////////////////////////////////////

class ChordButtonLabel extends JLabel {
  private ChordMatrix cm;
  public ChordButtonLabel(String txt, ChordMatrix cm) {
    super(txt,CENTER);
    this.cm = cm;
    setOpaque(true);
    setFont(getFont().deriveFont(Font.PLAIN));
    setDarkMode(false);
  }
  public void setDarkMode(boolean is_dark) {
    setBackground( is_dark ?
      cm.dark_mode_colorset.backgrounds[2] :
      cm.normal_mode_colorset.backgrounds[2]
    );
    setForeground( is_dark ?
      cm.dark_mode_colorset.foregrounds[0] :
      cm.normal_mode_colorset.foregrounds[0]
    );
  }
}

class ChordGuideLabel extends ChordButtonLabel {
  JPopupMenu popup_menu = new JPopupMenu();
  public ChordGuideLabel(String txt, ChordMatrix cm) {
    super(txt,cm);
    addMouseListener(new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        popup_menu.show( e.getComponent(), 0, getHeight() );
      }
    });
  }
  public void addMenu(JMenuItem menu_item) {
    popup_menu.add(menu_item);
  }
  public void addSeparator() {
    popup_menu.addSeparator();
  }
}

class ChordGuide extends JPanel {
  ChordGuideLabel
	chord_guide_5, chord_guide_76, chord_guide_9;
  public ChordGuide(ChordMatrix cm) {
    //
    // Chord suffix help
    chord_guide_76 = new ChordGuideLabel(" 6  7  M7 ",cm);
    chord_guide_76.setToolTipText("How to add 7th, major 7th, 6th");
    chord_guide_76.addMenu(new JMenuItem("7        = <RightClick>"));
    chord_guide_76.addMenu(new JMenuItem("M7(maj7) = [Shift] <RightClick>"));
    chord_guide_76.addMenu(new JMenuItem("6        = [Shift]"));
    //
    chord_guide_5 = new ChordGuideLabel(" -5 dim +5 aug ",cm);
    chord_guide_5.setToolTipText("How to add -5, dim, +5, aug");
    chord_guide_5.addMenu(new JMenuItem("-5 (b5)      = [Alt]"));
    chord_guide_5.addMenu(new JMenuItem("+5 (#5/aug)  = [Alt] sus4"));
    chord_guide_5.addSeparator();
    chord_guide_5.addMenu(new JMenuItem("dim  (m-5)  = [Alt] minor"));
    chord_guide_5.addMenu(new JMenuItem("dim7 (m6-5) = [Alt] [Shift] minor"));
    chord_guide_5.addMenu(new JMenuItem("m7-5 = [Alt] minor <RightClick>"));
    chord_guide_5.addMenu(new JMenuItem("aug7 (7+5)  = [Alt] sus4 <RightClick>"));
    //
    chord_guide_9 = new ChordGuideLabel(" add9 ",cm);
    chord_guide_9.setToolTipText("How to add 9th");
    chord_guide_9.addMenu(new JMenuItem("add9  = [Ctrl]"));
    chord_guide_9.addSeparator();
    chord_guide_9.addMenu(new JMenuItem("9     = [Ctrl] <RightClick>"));
    chord_guide_9.addMenu(new JMenuItem("M9    = [Ctrl] [Shift] <RightClick>"));
    chord_guide_9.addMenu(new JMenuItem("69    = [Ctrl] [Shift]"));
    chord_guide_9.addMenu(new JMenuItem("dim9  = [Ctrl] [Shift] [Alt] minor"));
    //
    setLayout(
      new BoxLayout( this, BoxLayout.X_AXIS )
    );
    add(chord_guide_76);
    add( Box.createHorizontalStrut(2) );
    add(chord_guide_5);
    add( Box.createHorizontalStrut(2) );
    add(chord_guide_9);
  }
  public void setDarkMode(boolean is_dark) {
    setBackground( is_dark ? Color.black : null );
    chord_guide_76.setDarkMode( is_dark );
    chord_guide_5.setDarkMode( is_dark );
    chord_guide_9.setDarkMode( is_dark );
  }
}

