/*
 * MIDI Message Form for MIDI Chord Helper
 *
 *	Copyright (C) 2006-2009 悵 - Akiyoshi Kamide
 *	http://www.yk.rim.or.jp/~kamide/music/chordhelper/
 */
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.sound.midi.*;

class MidiEventDialog extends JDialog {
  MidiMessageForm midi_message_form = new MidiMessageForm();
  TickPositionForm tick_position_form = new TickPositionForm();
  JButton ok_button = new JButton("OK");
  JButton cancel_button = new JButton("Cancel");
  public MidiEventDialog() {
    setLayout(new FlowLayout());
    add( tick_position_form );
    add( midi_message_form );
    JPanel ok_cancel_panel = new JPanel();
    ok_cancel_panel.add( ok_button );
    ok_cancel_panel.add( cancel_button );
    add( ok_cancel_panel );
    cancel_button.addActionListener(
      new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          setVisible(false);
        }
      }
    );
  }
  public void openTickForm() {
    tick_position_form.setVisible(true);
    midi_message_form.setVisible(false);
    setBounds( 200, 300, 500, 120 );
    setVisible(true);
  }
  public void openEventForm() {
    tick_position_form.setVisible(true);
    midi_message_form.setVisible(true);
    midi_message_form.setDurationVisible(true);
    setBounds( 200, 300, 630, 320 );
    setVisible(true);
  }
  public void openMessageForm() {
    tick_position_form.setVisible(false);
    midi_message_form.setVisible(true);
    midi_message_form.setDurationVisible(false);
    setBounds( 200, 300, 630, 270 );
    setVisible(true);
  }
}

///////////////////////////////////////////////////
//
// MIDI Message Entry Form - MIDIbZ[W͗
//
///////////////////////////////////////////////////
class MidiMessageForm extends JPanel
  implements ActionListener
{
  // f[^IpR{{bNXf
  DefaultComboBoxModel
    status_combo_box_model = new DefaultComboBoxModel(),
    note_combo_box_model = new DefaultComboBoxModel(),
    percussion_combo_box_model = new DefaultComboBoxModel(),
    control_change_combo_box_model = new DefaultComboBoxModel(),
    instrument_combo_box_model = new DefaultComboBoxModel(),
    meta_type_combo_box_model = new DefaultComboBoxModel(),
    hex_data1_combo_box_model = new DefaultComboBoxModel(),
    hex_data2_combo_box_model = new DefaultComboBoxModel();

  // f[^I𑀍암
  HexSelecter
    status_text = new HexSelecter("Status/Command"),
    data1_text = new HexSelecter("[Data1] "),
    data2_text = new HexSelecter("[Data2] ");
  MidiChannelComboSelecter
    channel_text = new MidiChannelComboSelecter("MIDI Channel");

  JComboBox
    status_combo_box = status_text.getComboBox(),
    data1_combo_box = data1_text.getComboBox(),
    data2_combo_box = data2_text.getComboBox(),
    channel_combo_box = channel_text.getComboBox();

  // lieLXg܂͐lj̓͗
  HexTextForm data_text = new HexTextForm("Data:",3,50);

  // K͗psAmL[{[h
  PianoKeyboardPanel keyboard_panel = new PianoKeyboardPanel();

  // ̒
  DurationForm duration_form = new DurationForm();

  // ^Cxgie|EqEj
  TempoSelecter tempo_selecter = new TempoSelecter();
  TimeSignatureSelecter
    timesig_selecter = new TimeSignatureSelecter();
  KeySignatureSelecter
    keysig_selecter = new KeySignatureSelecter();

  // 炷oMIDI`l
  private MidiChannel[] midi_channels = null;

  // Note on/off ̂Ƃ Duration tH[\邩
  private boolean is_duration_visible = true;

  public MidiMessageForm() {

    int i; String s;

    // MIDIXe[^XI
    for( i = 0x80; i <= 0xE0 ; i += 0x10 ) { // `lbZ[W
      if( (s = MIDISpec.getStatusName(i)) == null ) continue;
      status_combo_box_model.addElement(
        (Object)String.format("0x%02X : %s", i, s)
      );
    }
    for( i = 0xF0; i <= 0xFF ; i++ ) { // `lȂ SysEx ⃁^bZ[WȂ
      if( (s = MIDISpec.getStatusName(i)) == null ) continue;
      status_combo_box_model.addElement(
        (Object)String.format("0x%02X : %s", i, s)
      );
    }

    // m[gԍIf[^f
    for( i = 0; i<=0x7F; i++ ) {
      note_combo_box_model.addElement(
        (Object)String.format(
          "0x%02X : %d : %s", i, i, Music.noteNoToSymbol(i)
        )
      );
    }

    // Ŋy햼If[^f
    for( i = 0; i<=0x7F; i++ ) {
      percussion_combo_box_model.addElement(
        (Object)String.format(
          "0x%02X : %d : %s", i, i, MIDISpec.getPercussionName(i)
        )
      );
    }

    // Rg[`FWIf[^f
    for( i = 0; i<=0x7F; i++ ) {
      if( (s = MIDISpec.getControllerName(i)) == null ) continue;
      control_change_combo_box_model.addElement(
        (Object)String.format("0x%02X : %d : %s", i, i, s)
      );
    }

    // y햼iFjIf[^f
    for( i = 0; i<=0x7F; i++ ) {
      instrument_combo_box_model.addElement(
         (Object)String.format(
           "0x%02X : %s", i, MIDISpec.instrument_names[i]
         )
      );
    }

    // PUil݂̂̑If[^f
    for( i = 0; i<=0x7F; i++ ) {
      hex_data1_combo_box_model.addElement(
        (Object)String.format("0x%02X : %d", i, i )
      );
    }
    // ShortMessagef[^QoCgڂl
    for( i = 0; i<=0x7F; i++ ) {
      hex_data2_combo_box_model.addElement(
        (Object)String.format("0x%02X : %d", i, i )
      );
    }

    // MetaMessage Type
    String initial_type_string = null;
    for( int type = 0; type < 0x80 ; type++ ) {
      if( (s = MIDISpec.getMetaName( type )) == null ) {
        continue;
      }
      s = String.format("0x%02X : %s", type, s);
      meta_type_combo_box_model.addElement( (Object)s );
      if( type == 0x51 ) initial_type_string = s; // Tempo
    }
    meta_type_combo_box_model.setSelectedItem( initial_type_string );

    keyboard_panel.keyboard.setPreferredSize(new Dimension(300,40));

    // Set models
    //
    status_combo_box.setModel( status_combo_box_model );
    data2_combo_box.setModel( hex_data2_combo_box_model );

    // Add listener
    //
    keyboard_panel.keyboard.addPianoKeyboardListener(
      new PianoKeyboardAdapter() {
        public void pianoKeyPressed(int n, InputEvent e) {
          data1_text.setValue(n);
          if( midi_channels != null )
            midi_channels[channel_text.getSelectedChannel()].
              noteOn( n, data2_text.getValue() );
        }
        public void pianoKeyReleased(int n, InputEvent e) {
          if( midi_channels != null ) {
            midi_channels[channel_text.getSelectedChannel()].
              noteOff( n, data2_text.getValue() );
          }
        }
      }
    );
    tempo_selecter.tempo_spinner_model.addChangeListener(
      new ChangeListener() {
        public void stateChanged(ChangeEvent e) {
          data_text.setValue( tempo_selecter.getTempoByteArray() );
        }
      }
    );
    timesig_selecter.upper_timesig_spinner_model.addChangeListener(
      new ChangeListener() {
        public void stateChanged(ChangeEvent e) {
          data_text.setValue( timesig_selecter.getByteArray() );
        }
      }
    );
    timesig_selecter.lower_timesig_combobox.addActionListener(
      new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          data_text.setValue( timesig_selecter.getByteArray() );
        }
      }
    );
    keysig_selecter.keysig_combobox.addActionListener(
      new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          data_text.setValue( keysig_selecter.getKey().getBytes() );
        }
      }
    );
    keysig_selecter.minor_checkbox.addItemListener(
      new ItemListener() {
        public void itemStateChanged(ItemEvent e) {
          data_text.setValue( keysig_selecter.getKey().getBytes() );
        }
      }
    );
    status_combo_box.addActionListener(this);
    channel_combo_box.addActionListener(this);
    data1_combo_box.addActionListener(this);

    // Layout
    //
    JPanel panel1 = new JPanel();
    panel1.add( status_text );
    panel1.add( channel_text );

    JPanel panel2 = new JPanel();
    panel2.add( data1_text );
    panel2.add( keyboard_panel );

    JPanel panel3 = new JPanel();
    panel3.add( data2_text );

    setLayout(new BoxLayout( this, BoxLayout.Y_AXIS ));
    add( panel1 );
    add( duration_form );
    add( panel2 );
    add( panel3 );
    add( tempo_selecter );
    add( timesig_selecter );
    add( keysig_selecter );
    add( data_text );

    status_combo_box.setSelectedIndex(1); // NoteOn
    data2_combo_box.setSelectedIndex(64); // Center
    note_combo_box_model.setSelectedItem(
      note_combo_box_model.getElementAt(60)
    ); // Center note C
    percussion_combo_box_model.setSelectedItem(
      percussion_combo_box_model.getElementAt(35)
    ); // Acoustic Bass Drum

    updateVisible();
  }
  // ActionListener
  //
  public void actionPerformed(ActionEvent e) {
    Object src = e.getSource();
    if( src == data1_combo_box ) {
      int status = status_text.getValue();
      int data1 = data1_text.getValue();
      if( isNote(status) ) { // Data1 -> Note
        if( data1 >= 0 ) keyboard_panel.keyboard.setSelectedNote(data1);
      }
      else if( status == 0xFF ) {
        switch( data1 ) { // Data type -> Selecter
          case 0x51: data_text.setValue( tempo_selecter.getTempoByteArray() ); break;
          case 0x58: data_text.setValue( timesig_selecter.getByteArray() ); break;
          case 0x59: data_text.setValue( keysig_selecter.getKey().getBytes() ); break;
          default: break;
        }
      }
    }
    updateVisible();
  }
  //
  // Methods
  //
  public void setOutputMidiChannels( MidiChannel midi_channels[] ) {
    this.midi_channels = midi_channels;
  }
  public void setDurationVisible(boolean is_visible) {
    is_duration_visible = is_visible;
    updateVisible();
  }
  public boolean isDurationVisible() {
    return is_duration_visible;
  }
  public void updateVisible() {
    int msg_status = status_text.getValue();
    boolean is_ch_msg = MIDISpec.isChannelMessage(msg_status);
    channel_text.setVisible(is_ch_msg);
    status_text.setTitle(
      "[Status] "+(is_ch_msg ? "Command" : "")
    );
    duration_form.setVisible( is_duration_visible && isNote(msg_status) );
    keyboard_panel.setVisible( msg_status <= 0xAF );

    if(
      msg_status <= 0xEF
      ||
      msg_status >= 0xF1 && msg_status <= 0xF3
      ||
      msg_status == 0xFF
    ) {
      data1_text.setVisible( true );
    }
    else {
      data1_text.setVisible( false );
    }

    if(
      (msg_status >= 0xC0 && msg_status <= 0xDF) ||
      msg_status == 0xF0 || msg_status == 0xF1 ||
      msg_status == 0xF3 || msg_status >= 0xF6
    ) {
      data2_text.setVisible( false );
    }
    else {
      data2_text.setVisible( true );
    }
    data2_text.setTitle("[Data2] "+(
      msg_status <= 0x9F ? "Velocity" :
      msg_status <= 0xAF ? "Pressure" :
      msg_status <= 0xBF ? "Value" :
      (msg_status & 0xF0) == 0xE0 ? "High 7bit value" : ""
    ));

    // Show if Sysex or Meta
    data_text.setVisible(
      msg_status == 0xF0 ||
      msg_status == 0xF7 ||
      msg_status == 0xFF
    );

    if( msg_status != 0xFF ) {
      tempo_selecter.setVisible(false);
      timesig_selecter.setVisible(false);
      keysig_selecter.setVisible(false);
    }

    switch( msg_status & 0xF0 ) {
    // Xe[^XɉāAPoCgڂ̃f[^f؂ւB

    case 0x80: // Note Off
    case 0x90: // Note On
    case 0xA0: // Polyphonic Key Pressure
      int ch = channel_text.getSelectedChannel();
      data1_text.setTitle(
        "[Data1] "+( ch == 9 ? "Percussion" : "Note No." )
      );
      data1_combo_box.setModel(
        ch == 9 ? percussion_combo_box_model : note_combo_box_model
      );
      break;

    case 0xB0: // Control Change / Mode Change
      data1_text.setTitle("[Data1] Control/Mode No.");
      data1_combo_box.setModel(control_change_combo_box_model);
      break;

    case 0xC0: // Program Change
      data1_text.setTitle( "[Data1] Program No.");
      data1_combo_box.setModel(instrument_combo_box_model);
      break;

    case 0xD0: // Channel Pressure
      data1_text.setTitle("[Data1] Pressure");
      data1_combo_box.setModel(hex_data1_combo_box_model);
      break;

    case 0xE0: // Pitch Bend
      data1_text.setTitle("[Data1] Low 7bit value");
      data1_combo_box.setModel(hex_data1_combo_box_model);
      break;

    default:
      if( msg_status == 0xFF ) { // MetaMessage
        data1_text.setTitle("[Data1] MetaEvent Type");
        data1_combo_box.setModel(meta_type_combo_box_model);
        int msg_type = data1_text.getValue();
        tempo_selecter.setVisible( msg_type == 0x51 );
        timesig_selecter.setVisible( msg_type == 0x58 );
        keysig_selecter.setVisible( msg_type == 0x59 );
        //
        if( MIDISpec.isEOT(msg_type) ) {
          data_text.clear();
          data_text.setVisible(false);
        }
        else {
          data_text.setTitle(
            MIDISpec.hasMetaText( msg_type ) ? "Text:":"Data:"
          );
        }
      }
      else {
        data1_text.setTitle("[Data1] ");
        data1_combo_box.setModel(hex_data1_combo_box_model);
      }
      break;
    }
  }
  public MidiMessage getMessage() {
    int msg_status = status_text.getValue();
    if( msg_status < 0 ) {
      return null;
    }
    else if( msg_status == 0xFF ) {
      int msg_type = data1_text.getValue();
      if( msg_type < 0 ) return null;
      byte msg_data[];
      if( MIDISpec.hasMetaText( msg_type ) ) {
        msg_data = data_text.getBytesFromString();
      }
      else if( msg_type == 0x2F ) { // EOT
        // To avoid inserting un-removable EOT, ignore the data.
        msg_data = new byte[0];
      }
      else {
        if( (msg_data = data_text.getBytes() ) == null ) {
          return null;
        }
      }
      MetaMessage msg = new MetaMessage();
      try {
        msg.setMessage( msg_type, msg_data, msg_data.length );
      } catch( InvalidMidiDataException e ) {
        e.printStackTrace();
        return null;
      }
      return (MidiMessage)msg;
    }
    else if( msg_status == 0xF0 || msg_status == 0xF7 ) {
      SysexMessage msg = new SysexMessage();
      byte data[] = data_text.getBytes();
      if( data == null ) return null;
      try {
        msg.setMessage(
          (int)(msg_status & 0xFF), data, data.length
        );
      } catch( InvalidMidiDataException e ) {
        e.printStackTrace();
        return null;
      }
      return (MidiMessage)msg;
    }
    ShortMessage msg = new ShortMessage();
    int msg_data1 = data1_text.getValue();
    if( msg_data1 < 0 ) msg_data1 = 0;
    int msg_data2 = data2_text.getValue();
    if( msg_data2 < 0 ) msg_data2 = 0;
    try {
      if( MIDISpec.isChannelMessage( msg_status ) ) {
        msg.setMessage(
          (msg_status & 0xF0),
          channel_text.getSelectedChannel(),
          msg_data1, msg_data2
        );
      }
      else {
        msg.setMessage( msg_status, msg_data1, msg_data2 );
      }
    } catch( InvalidMidiDataException e ) {
      e.printStackTrace();
      return null;
    }
    return (MidiMessage)msg;
  }
  public void setMessage( MidiMessage msg ) {
    if( msg instanceof ShortMessage ) {
      ShortMessage smsg = (ShortMessage)msg;
      int msg_ch = 0;
      int msg_status = smsg.getStatus();
      if( MIDISpec.isChannelMessage(msg_status) ) {
        msg_status = smsg.getCommand();
        msg_ch = smsg.getChannel();
      }
      status_text.setValue( msg_status );
      channel_text.setSelectedChannel( msg_ch );
      data1_text.setValue( smsg.getData1() );
      data2_text.setValue( smsg.getData2() );
    }
    else if( msg instanceof SysexMessage ) {
      SysexMessage sysex_msg = (SysexMessage)msg;
      status_text.setValue( sysex_msg.getStatus() );
      data_text.setValue( sysex_msg.getData() );
    }
    else if( msg instanceof MetaMessage ) {
      MetaMessage meta_msg = (MetaMessage)msg;
      int msg_type = meta_msg.getType();
      byte data[] = meta_msg.getData();
      status_text.setValue( 0xFF );
      data1_text.setValue( msg_type );
      switch( msg_type ) {
      case 0x51: tempo_selecter.setTempo( data ); break;
      case 0x58: timesig_selecter.setValue( data[0], data[1] ); break;
      case 0x59: keysig_selecter.setKey( new Music.Key(data) ); break;
      default: break;
      }
      if( MIDISpec.hasMetaText( msg_type ) ) {
        data_text.setString( new String(data) );
      }
      else {
        data_text.setValue( data );
      }
      updateVisible();
    }
  }
  public boolean setNote( int ch, int note_no, int velocity ) {
    channel_text.setSelectedChannel(ch);
    data1_text.setValue(note_no);
    data2_text.setValue(velocity);
    return true;
  }
  public boolean isNote() {
    return isNote( status_text.getValue() );
  }
  public boolean isNote( int status ) {
    int cmd = status & 0xF0;
    return ( cmd == ShortMessage.NOTE_ON || cmd == ShortMessage.NOTE_OFF );
  }
  public boolean isNote( boolean note_on ) {
    return isNote( note_on, status_text.getValue() );
  }
  public boolean isNote( boolean note_on, int status ) {
    int cmd = status & 0xF0;
    return (
       note_on && cmd == ShortMessage.NOTE_ON && data2_text.getValue() > 0
       ||
       !note_on && (
         cmd == ShortMessage.NOTE_ON && data2_text.getValue() <= 0
         ||
         cmd == ShortMessage.NOTE_OFF
       )
    );
  }
  public ShortMessage getPartnerMessage() {
    ShortMessage sm = (ShortMessage)getMessage();
    if( sm == null ) return null;
    ShortMessage partner_sm;
    if( isNote(true) ) { // NoteOn
      partner_sm = new ShortMessage();
      try{
        partner_sm.setMessage(
          ShortMessage.NOTE_OFF,
          sm.getChannel(),
          sm.getData1(), sm.getData2()
        );
      } catch( InvalidMidiDataException e ) {
        e.printStackTrace();
        return null;
      }
      return partner_sm;
    }
    else if( isNote(false) ) { // NoteOff
      partner_sm = new ShortMessage();
      try{
        partner_sm.setMessage(
          ShortMessage.NOTE_ON,
          sm.getChannel(),
          sm.getData1() == 0 ? 100 : sm.getData1(),
          sm.getData2()
        );
      } catch( InvalidMidiDataException e ) {
        e.printStackTrace();
        return null;
      }
      return partner_sm;
    }
    return null;
  }
}


///////////////////////////////////////
//
// Hex value: [0x00 0x00 0x00 ... ]
//
///////////////////////////////////////
class HexTextForm extends JPanel {
  public JTextArea text_area;
  public JLabel title_label;
  public HexTextForm(String title) {
    this(title,1,3);
  }
  public HexTextForm(String title, int rows, int columns) {
    if( title != null ) add(title_label = new JLabel(title));
    text_area = new JTextArea(rows, columns);
    text_area.setLineWrap(true);
    JScrollPane text_scroll_pane = new JScrollPane(text_area);
    add(text_scroll_pane);
    setLayout(new FlowLayout());
  }
  public String getString() {
    return text_area.getText();
  }
  public byte[] getBytesFromString() {
    return getString().getBytes();
  }
  public byte[] getBytes() {
    String words[] = getString().trim().split(" +");
    ArrayList<Integer> tmp_ba = new ArrayList<Integer>();
    int i;
    for( String w : words ) {
      if( w.length() == 0 ) continue;
      try {
        i = Integer.decode(w).intValue();
      } catch( NumberFormatException e ) {
        JOptionPane.showMessageDialog(
          this,
          w + " : is not a number",
          "MIDI Chord Helper",
          JOptionPane.ERROR_MESSAGE
        );
        return null;
      }
      tmp_ba.add(i);
    }
    byte[] ba = new byte[tmp_ba.size()];
    i = 0;
    for( Integer b : tmp_ba ) {
      ba[i++] = (byte)( b.intValue() & 0xFF );
    }
    return ba;
  }
  public void setTitle( String str ) {
    title_label.setText( str );
  }
  public void setString( String str ) {
    text_area.setText( str );
  }
  public void setValue( int val ) {
    text_area.setText( String.format( " 0x%02X", val ) );
  }
  public void setValue( byte val ) {
    text_area.setText( String.format( " 0x%02X", val ) );
  }
  public void setValue( byte ba[] ) {
    String str = "";
    for( byte b : ba ) {
      str += String.format( " 0x%02X", b );
    }
    text_area.setText(str);
  }
  public void clear() { text_area.setText(""); }
}

///////////////////////////////////////
//
// Hex value: [0x00 0x00 0x00 ... ] v -> Select
//
///////////////////////////////////////
class HexSelecter extends JPanel {
  private JComboBox combo_box;
  private JLabel title;
  public HexSelecter( String title ) {
    if( title != null ) add( this.title = new JLabel(title) );
    add( combo_box = new JComboBox() );
    setLayout(new FlowLayout());
    combo_box.setEditable(true);
    combo_box.setMaximumRowCount(16);
  }
  public JComboBox getComboBox() { return combo_box; }
  public void setTitle( String title ) {
    this.title.setText(title);
  }
  public String getString() {
    return (String)( combo_box.getSelectedItem() );
  }
  public int getValue() {
    ArrayList<Integer> ia = getIntegerList();
    return ia.size() == 0 ? -1 : ia.get(0);
  }
  public ArrayList<Integer> getIntegerList() {
    String words[], str = getString();
    if( str == null )
      words = new String[0];
    else
      words = str.replaceAll( ":.*$", "" ).trim().split(" +");
    int i;
    ArrayList<Integer> ia = new ArrayList<Integer>();
    for( String w : words ) {
      if( w.length() == 0 ) continue;
      try {
        i = Integer.decode(w).intValue();
      } catch( NumberFormatException e ) {
        JOptionPane.showMessageDialog(
          this,
          w + " : is not a number",
          "MIDI Chord Helper",
          JOptionPane.ERROR_MESSAGE
        );
        return null;
      }
      ia.add(i);
    }
    return ia;
  }
  public byte[] getBytes() {
    ArrayList<Integer> ia = getIntegerList();
    byte[] ba = new byte[ia.size()];
    int i = 0;
    for( Integer ib : ia ) {
      ba[i++] = (byte)( ib.intValue() & 0xFF );
    }
    return ba;
  }
  public void setValue( int val ) {
    setValue( (byte)(val & 0xFF) );
  }
  public void setValue( byte val ) {
    int n_item = combo_box.getItemCount();
    String item;
    for( int i=0; i<n_item; i++ ) {
      item = (String)( combo_box.getItemAt(i) );
      if( Integer.decode( item.trim().split(" +")[0] ).byteValue() == val ) {
        combo_box.setSelectedIndex(i);
        return;
      }
    }
    combo_box.setSelectedItem(
      (Object)( String.format( " 0x%02X", val ) )
    );
  }
  public void setValue( byte ba[] ) {
    String str = "";
    for( byte b : ba ) str += String.format( " 0x%02X", b );
    combo_box.setSelectedItem( (Object)str );
  }
  public void clear() {
    combo_box.setSelectedItem( (Object)"" );
  }
}

///////////////////////////////////////
//
// MIDI Channel Selecter (ComboBox or List)
//
///////////////////////////////////////
interface MidiChannelComboBoxModel extends ComboBoxModel {
  int getSelectedChannel();
  void setSelectedChannel(int channel);
}
class DefaultMidiChannelComboBoxModel
  extends DefaultComboBoxModel
  implements MidiChannelComboBoxModel
{
  public DefaultMidiChannelComboBoxModel() {
    for( int ch = 0; ch < MIDISpec.MAX_CHANNELS ; ch++ )
      addElement(ch+1);
  }
  public int getSelectedChannel() {
    return getIndexOf(getSelectedItem());
  }
  public void setSelectedChannel(int channel) {
    setSelectedItem(getElementAt(channel));
  }
}
class MidiChannelComboSelecter extends JPanel {
  JComboBox combo_box = new JComboBox();
  public MidiChannelComboSelecter( String title ) {
    this( title, new DefaultMidiChannelComboBoxModel() );
  }
  public MidiChannelComboSelecter(
    String title, MidiChannelComboBoxModel model
  ) {
    setLayout(new FlowLayout());
    if( title != null ) add( new JLabel(title) );
    combo_box.setModel(model);
    combo_box.setMaximumRowCount(16);
    add(combo_box);
  }
  public JComboBox getComboBox() { return combo_box; }
  public MidiChannelComboBoxModel getModel() {
    return (MidiChannelComboBoxModel)combo_box.getModel();
  }
  public int getSelectedChannel() {
    return combo_box.getSelectedIndex();
  }
  public void setSelectedChannel(int channel) {
    combo_box.setSelectedIndex(channel);
  }
}
class MidiChannelButtonSelecter extends JList {
  private PianoKeyboard keyboard = null;
  public MidiChannelButtonSelecter(MidiChannelComboBoxModel model) {
    super(model);
    setLayoutOrientation(HORIZONTAL_WRAP);
    setVisibleRowCount(1);
    setCellRenderer(new MyCellRenderer());
    setSelectedIndex(model.getSelectedChannel());
    model.addListDataListener(new ListDataListener() {
      public void contentsChanged(ListDataEvent e) {
        MidiChannelButtonSelecter.this.setSelectedIndex(
          MidiChannelButtonSelecter.this.getModel().getSelectedChannel()
        );
      }
      public void intervalAdded(ListDataEvent e) {}
      public void intervalRemoved(ListDataEvent e) {}
    });
    addListSelectionListener(new ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
        MidiChannelButtonSelecter.this.getModel().setSelectedChannel(
          MidiChannelButtonSelecter.this.getSelectedIndex()
        );
      }
    });
  }
  public MidiChannelButtonSelecter(PianoKeyboard keyboard) {
    this(keyboard.midi_ch_combobox_model);
    setPianoKeyboard(keyboard);
  }
  public MidiChannelComboBoxModel getModel() {
    return (MidiChannelComboBoxModel)(super.getModel());
  }
  public void setPianoKeyboard(PianoKeyboard keyboard) {
    (this.keyboard = keyboard).midi_ch_button_selecter = this;
  }
  class MyCellRenderer extends JLabel implements ListCellRenderer {
    private boolean cellHasFocus = false;
    public MyCellRenderer() {
      setOpaque(true);
      setHorizontalAlignment(CENTER);
      setSelectionBackground(Color.yellow);
    }
    public Component getListCellRendererComponent(
      JList list, Object value, int index,
      boolean isSelected, boolean cellHasFocus
    ) {
      this.cellHasFocus = cellHasFocus;
      setText(value.toString());
      if(isSelected) {
        setBackground(list.getSelectionBackground());
        setForeground(list.getSelectionForeground());
      } else {
        setBackground(
          keyboard != null && keyboard.countKeyOn(index) > 0 ?
          Color.pink : list.getBackground()
        );
        setForeground(list.getForeground());
      }
      setEnabled(list.isEnabled());
      setFont(list.getFont());
      return this;
    }
    protected void paintComponent(Graphics g) {
      super.paintComponent(g);
      if( cellHasFocus ) {
        g.setColor(Color.gray);
        g.drawRect(0, 0, this.getWidth() - 1, this.getHeight() - 1);
      }
    }
  }
}

////////////////////////////////////
//
// Tick Position
//
// Mesausre:[xxxx] Beat:[xx] ExTick:[xxx]
//
////////////////////////////////////
class TickPositionModel
  implements ChangeListener
{
  private SequenceIndex seq_index;
  private boolean is_changing = false;
  public SpinnerNumberModel
    tick_model, measure_model, beat_model, extra_tick_model;
  //
  // Constuctor
  //
  public TickPositionModel() {
    tick_model = new SpinnerNumberModel(0L, 0L, 999999L, 1L);
    tick_model.addChangeListener(this);
    measure_model = new SpinnerNumberModel(1, 1, 9999, 1);
    measure_model.addChangeListener(this);
    beat_model = new SpinnerNumberModel(1, 1, 32, 1);
    beat_model.addChangeListener(this);
    extra_tick_model = new SpinnerNumberModel(0, 0, 4*960-1, 1);
    extra_tick_model.addChangeListener(this);
  }
  // ChangeListener
  //
  public void stateChanged(ChangeEvent e) {
    if( seq_index == null ) return;
    Object src = e.getSource();
    if( src == tick_model ) {
      is_changing = true;
      measure_model.setValue(
        1 + seq_index.tickToMeasure(
          tick_model.getNumber().longValue()
        )
      );
      beat_model.setValue( seq_index.last_beat + 1 );
      is_changing = false;
      extra_tick_model.setValue( seq_index.last_extra_tick );
      return;
    }
    if( is_changing ) return;
    tick_model.setValue(
      seq_index.measureToTick(
        measure_model.getNumber().intValue() - 1,
        beat_model.getNumber().intValue() - 1,
        extra_tick_model.getNumber().intValue()
      )
    );
  }
  // Methods
  //
  public SequenceIndex getSequenceIndex() {
    return seq_index;
  }
  public void setSequenceIndex( SequenceIndex seq_index ) {
    this.seq_index = seq_index;
    extra_tick_model.setMaximum( 4 * seq_index.getResolution() - 1 );
  }
  public long getTickPosition() {
    return tick_model.getNumber().longValue();
  }
  public void setTickPosition( long tick ) {
    tick_model.setValue(tick);
  }
}

class TickPositionForm extends JPanel {
  JSpinner
    tick_spinner = new JSpinner(),
    measure_spinner = new JSpinner(),
    beat_spinner = new JSpinner(),
    extra_tick_spinner = new JSpinner();

  public TickPositionForm() {
    setLayout(new GridLayout(2,4));
    add( new JLabel() );
    add( new JLabel() );
    add( new JLabel("Measure:") );
    add( new JLabel("Beat:") );
    add( new JLabel("ExTick:") );
    add( new JLabel("Tick position : ",JLabel.RIGHT) );
    add( tick_spinner );
    add( measure_spinner );
    add( beat_spinner );
    add( extra_tick_spinner );
  }
  public TickPositionForm( TickPositionModel tpm ) {
    this(); setModel(tpm);
  }
  public void setModel( TickPositionModel tpm ) {
    tick_spinner.setModel(tpm.tick_model);
    measure_spinner.setModel(tpm.measure_model);
    beat_spinner.setModel(tpm.beat_model);
    extra_tick_spinner.setModel(tpm.extra_tick_model);
  }
}

////////////////////////////////////
//
// Duration Form
//
class DurationForm extends JPanel
  implements ActionListener, ChangeListener
{
  class NoteIcon extends ButtonIcon {
    public NoteIcon( int kind ) { super(kind); }
    public int getDuration() {
      if(  ! isMusicalNote() ) return -1;
      int duration = (ppq * 4) >> getMusicalNoteValueIndex();
      if( isDottedMusicalNote() ) duration += duration / 2;
      return duration;
    }
  }
  class NoteRenderer extends JLabel
    implements ListCellRenderer
  {
    public NoteRenderer() { setOpaque(true); }
    public Component getListCellRendererComponent(
      JList list,
      Object value,
      int index,
      boolean isSelected,
      boolean cellHasFocus)
    {
      NoteIcon icon = (NoteIcon)value;
      setIcon( icon );
      int duration = icon.getDuration();
      setText( duration < 0 ? null : ("" + duration) );
      setFont( list.getFont() );
      if (isSelected) {
        setBackground( list.getSelectionBackground() );
        setForeground( list.getSelectionForeground() );
      } else {
        setBackground( list.getBackground() );
        setForeground( list.getForeground() );
      }
      return this;
    }
  }
  class NoteComboBox extends JComboBox {
    public NoteComboBox() {
      setRenderer( new NoteRenderer() );
      addItem( new NoteIcon(ButtonIcon.EDIT_ICON) );
      addItem( new NoteIcon(ButtonIcon.WHOLE_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.DOTTED_HALF_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.HALF_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.DOTTED_QUARTER_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.QUARTER_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.DOTTED_8TH_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.A8TH_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.DOTTED_16TH_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.A16TH_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.DOTTED_32ND_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.A32ND_NOTE_ICON) );
      addItem( new NoteIcon(ButtonIcon.A64TH_NOTE_ICON) );
      setMaximumRowCount(16);
      setSelectedIndex(5);
    }
    public int getDuration() {
      NoteIcon icon = (NoteIcon)getSelectedItem();
      if( icon == null ) return -1;
      return icon.getDuration();
    }
    public void setDuration(int duration) {
      int n_items = getItemCount();
      for( int i = 1; i < n_items; i++ ) {
        NoteIcon icon = (NoteIcon)getItemAt(i);
        int icon_duration = icon.getDuration();
        if( icon_duration < 0 || icon_duration != duration ) continue;
        setSelectedItem(icon);
        return;
      }
      setSelectedIndex(0);
    }
  }
  class DurationModel extends SpinnerNumberModel {
    public DurationModel() { super( ppq, 1, ppq*4*4, 1 ); }
    public void setDuration( int value ) {
      setValue( new Integer(value) );
    }
    public int getDuration() {
      return getNumber().intValue();
    }
    public void setPPQ( int ppq ) {
      setMaximum( ppq*4*4 );
      setDuration( ppq );
    }
  }
  DurationModel model;
  JSpinner spinner;
  NoteComboBox note_combo;
  JLabel title_label, unit_label;
  private int ppq = 960;
  //
  public DurationForm() {
    (model = new DurationModel()).addChangeListener(this);
    (note_combo = new NoteComboBox()).addActionListener(this);
    add( title_label = new JLabel("Duration:") );
    add( note_combo );
    add( spinner = new JSpinner( model ) );
    add( unit_label = new JLabel("[Ticks]") );
  }
  // ActionListener
  //
  public void actionPerformed(ActionEvent e) {
    int duration = note_combo.getDuration();
    if( duration < 0 ) return;
    model.setDuration( duration );
  }
  // ChangeListener
  //
  public void stateChanged(ChangeEvent e) {
    note_combo.setDuration( model.getDuration() );
  }
  // Methods
  //
  public void setEnabled( boolean enabled ) {
    super.setEnabled(enabled);
    title_label.setEnabled(enabled);
    spinner.setEnabled(enabled);
    note_combo.setEnabled(enabled);
    unit_label.setEnabled(enabled);
  }
  public void setPPQ( int ppq ) {
    model.setPPQ( this.ppq = ppq );
  }
  public int getDuration() { return model.getDuration(); }
  public void setDuration( int duration ) { model.setDuration(duration); }
}

//////////////////////////////
//
//  Tempo in QPM
//
//////////////////////////////
class TempoSelecter extends JPanel
  implements MouseListener
{
  JSpinner		tempo_spinner;
  SpinnerNumberModel	tempo_spinner_model;
  JLabel		tempo_label, tempo_value_label;
  private boolean	editable;
  static final int	DEFAULT_QPM = 120;
  private long	prev_beat_us_pos = 0;

  public TempoSelecter() {
    String tool_tip = "Tempo in quatrers per minute - e|iP̎l̐j";
    tempo_label = new JLabel(
      "=",
      new ButtonIcon( ButtonIcon.QUARTER_NOTE_ICON ),
      JLabel.CENTER
    );
    tempo_label.setVerticalAlignment( JLabel.CENTER );
    tempo_spinner_model = new SpinnerNumberModel(
      DEFAULT_QPM, 1, 999, 1
    );
    tempo_spinner = new JSpinner( tempo_spinner_model );
    tempo_spinner.setToolTipText( tool_tip );
    tempo_value_label = new JLabel( ""+DEFAULT_QPM );
    tempo_value_label.setToolTipText( tool_tip );
    setLayout( new BoxLayout(this,BoxLayout.X_AXIS) );
    add( tempo_label );
    add( Box.createHorizontalStrut(5) );
    add( tempo_spinner );
    add( tempo_value_label );
    setEditable(true);
    tempo_label.addMouseListener(this);
  }
  //
  // MouseListener
  //
  public void mousePressed(MouseEvent e) {
    Component obj = e.getComponent();
    if( obj == tempo_label && isEditable() ) {
      //
      // Adjust tempo by interval time between two clicks
      //
      long current_us = System.nanoTime()/1000;
      // midi_ch_selecter.noteOn( 9, 37, 100 );
      long interval_us = current_us - prev_beat_us_pos;
      prev_beat_us_pos = current_us;
      if( interval_us < 2000000L /* Shorter than 2 sec only */ ) {
        int tempo_in_bpm = (int)(240000000L / interval_us) >> 2; //  n/4q̏ꍇ݂̂z
        int old_tempo_in_bpm = getTempoInQpm();
        setTempo( ( tempo_in_bpm + old_tempo_in_bpm * 2 ) / 3 );
      }
    }
  }
  public void mouseReleased(MouseEvent e) { }
  public void mouseEntered(MouseEvent e) { }
  public void mouseExited(MouseEvent e) { }
  public void mouseClicked(MouseEvent e) { }
  //
  // Methods
  //
  public boolean isEditable() { return editable; }
  public void setEditable( boolean editable ) {
    this.editable = editable;
    tempo_spinner.setVisible( editable );
    tempo_value_label.setVisible( !editable );
    if( !editable ) {
      // Copy spinner's value to label
      tempo_value_label.setText(
        ""+tempo_spinner_model.getNumber().intValue()
      );
    }
    tempo_label.setToolTipText(
      editable ?
      "Click rhythmically to adjust tempo - NbNăYƂƃe|킹܂"
      : null
    );
  }
  public int getTempoInQpm() {
    return tempo_spinner_model.getNumber().intValue();
  }
  public byte[] getTempoByteArray() {
    return MIDISpec.qpmTempoToByteArray( getTempoInQpm() );
  }
  public void setTempo( int qpm ) {
    tempo_spinner_model.setValue(new Integer(qpm));
    tempo_value_label.setText(""+qpm);
  }
  public void setTempo( byte msgdata[] ) {
    setTempo( MIDISpec.byteArrayToQpmTempo( msgdata ) );
  }
  public void clear() { setTempo( DEFAULT_QPM ); }
}

/////////////////////////////
//
// Time Signature - qI
//
/////////////////////////////
class TimeSignatureLabel extends JLabel {
  private byte upper = -1;
  private byte lower_index = -1;
  public void setTimeSignature(byte upper, byte lower_index) {
    if( this.upper == upper && this.lower_index == lower_index ) {
      return;
    }
    setText(
      "<html><font size=\"+1\">" +
        upper + "/" + (1 << lower_index) +
      "</font></html>"
    );
  }
}
class TimeSignatureSelecter extends JPanel {
  SpinnerNumberModel	upper_timesig_spinner_model;
  JSpinner		upper_timesig_spinner;
  JComboBox		lower_timesig_combobox;
  TimeSignatureLabel	timesig_value_label;
  private boolean	editable;

  public TimeSignatureSelecter() {
    upper_timesig_spinner_model = new SpinnerNumberModel( 4, 1, 32, 1 );
    upper_timesig_spinner = new JSpinner( upper_timesig_spinner_model );
    upper_timesig_spinner.setToolTipText("Time signature (upper digit) - q̕q");
    lower_timesig_combobox = new JComboBox();
    lower_timesig_combobox.setToolTipText("Time signature (lower digit) - q̕");
    for( int i=0; i<6; i++ ) {
      lower_timesig_combobox.addItem( "/" + (1<<i) );
    }
    timesig_value_label = new TimeSignatureLabel();
    timesig_value_label.setToolTipText("Time signature - q");
    lower_timesig_combobox.setSelectedIndex(2);
    //
    add( upper_timesig_spinner );
    add( lower_timesig_combobox );
    add( timesig_value_label );
    setEditable(true);
  }
  public void clear() {
    upper_timesig_spinner_model.setValue(4);
    lower_timesig_combobox.setSelectedIndex(2);
  }
  public int getUpperValue() {
    return upper_timesig_spinner_model.getNumber().intValue();
  }
  public byte getUpperByte() {
    return upper_timesig_spinner_model.getNumber().byteValue();
  }
  public int getLowerValueIndex() {
    return lower_timesig_combobox.getSelectedIndex();
  }
  public byte getLowerByte() {
    return (byte)getLowerValueIndex();
  }
  public byte[] getByteArray() {
    byte[] data = new byte[4];
    data[0] = getUpperByte();
    data[1] = getLowerByte();
    data[2] = (byte)( 96 >> getLowerValueIndex() );
    data[3] = 8;
    return data;
  }
  public void setValue( byte upper, byte lower_index ) {
    upper_timesig_spinner_model.setValue( upper );
    lower_timesig_combobox.setSelectedIndex( lower_index );
    timesig_value_label.setTimeSignature( upper, lower_index );
  }
  public void setValue( byte[] data ) {
    setValue( data[0], data[1] );
  }
  public boolean isEditable() { return editable; }
  public void setEditable( boolean editable ) {
    this.editable = editable;
    upper_timesig_spinner.setVisible(editable);
    lower_timesig_combobox.setVisible(editable);
    timesig_value_label.setVisible(!editable);
    if( !editable ) {
      timesig_value_label.setTimeSignature(
        getUpperByte(), getLowerByte()
      );
    }
  }
}

/////////////////////////////
//
// Key Signature - I
//
/////////////////////////////
class KeySignatureSelecter extends JPanel implements ActionListener {
  JComboBox	keysig_combobox;
  JCheckBox	minor_checkbox = null;

  public KeySignatureSelecter() { this(true); }
  public KeySignatureSelecter(boolean use_minor_checkbox) {
    add( new JLabel("Key:") );
    add( keysig_combobox = new JComboBox() );
    if( use_minor_checkbox ) {
      add( minor_checkbox = new JCheckBox("minor") );
      minor_checkbox.addActionListener(this);
    }
    int i;
    String str;
    Music.Key key;
    for( i = -7 ; i <= 7 ; i++ ) {
      str = (key = new Music.Key(i)).toString();
      if( i != 0 ) {
        str = key.signature() + " : " + str ;
      }
      keysig_combobox.addItem(str);
    }
    keysig_combobox.setMaximumRowCount(15);
    keysig_combobox.addActionListener(this);
    clear();
  }
  // ActionListener
  //
  public void actionPerformed(ActionEvent e) {
    // Object src = e.getSource();
    updateToolTipText();
  }
  //
  // Methods
  //
  private void updateToolTipText() {
    Music.Key key = getKey();
    keysig_combobox.setToolTipText(
      "Key: " + key.toStringIn( Music.NoteSymbol.NOTE_NAME )
      + " "  + key.toStringIn( Music.NoteSymbol.NOTE_IN_JAPANESE )
      + " (" + key.signatureDescription() + ")"
    );
  }
  public void clear() {
    setKey( new Music.Key("C") );
  }
  public void setKey( Music.Key key ) {
    if( key == null ) { clear(); return; }
    keysig_combobox.setSelectedIndex( key.toCo5() + 7 );
    if( minor_checkbox == null ) return;
    switch( key.majorMinor() ) {
    case Music.Key.MINOR : minor_checkbox.setSelected(true); break;
    case Music.Key.MAJOR : minor_checkbox.setSelected(false); break;
    }
  }
  public Music.Key getKey() {
    int minor = (
      minor_checkbox == null ? Music.Key.MAJOR_OR_MINOR :
      isMinor() ? Music.Key.MINOR :
      Music.Key.MAJOR
    );
    return new Music.Key(getKeyCo5(),minor);
  }
  public int getKeyCo5() {
    return keysig_combobox.getSelectedIndex() - 7;
  }
  public boolean isMinor() {
    return ( minor_checkbox != null && minor_checkbox.isSelected() );
  }
}
class KeySignatureLabel extends JLabel {
  private Music.Key key;
  public KeySignatureLabel() { clear(); }
  public Music.Key getKey() { return key; }
  public void setKeySignature( Music.Key key ) {
    this.key = key;
    if( key == null ) {
      setText("Key:C");
      setToolTipText("Key: Unknown");
      setEnabled(false);
      return;
    }
    setText( "key:" + key.toString() );
    setToolTipText(
      "Key: " + key.toStringIn(Music.NoteSymbol.NOTE_NAME)
      + " "  + key.toStringIn(Music.NoteSymbol.NOTE_IN_JAPANESE)
      + " (" + key.signatureDescription() + ")"
    );
    setEnabled(true);
  }
  public void clear() { setKeySignature( (Music.Key)null ); }
}

///////////////////////////////////////////////
//
// Velocity
//
///////////////////////////////////////////////
class VelocityModel extends DefaultBoundedRangeModel {
  public VelocityModel() { super( 64, 0, 0, 127 ); }
}
class VelocitySelecter extends JPanel {
  private static final String	LABEL_PREFIX = "Velocity=";
  public JSlider slider = null;
  public JLabel label;
  public VelocitySelecter( VelocityModel model ) {
    slider = new JSlider(model);
    slider.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        label.setText( LABEL_PREFIX + getValue() );
      }
    });
    slider.setToolTipText("Velocity");
    label = new JLabel( LABEL_PREFIX + model.getValue(), Label.RIGHT );
    label.setToolTipText("Velocity");
    setLayout( new BoxLayout( this, BoxLayout.X_AXIS ) );
    add(label);
    add(slider);
  }
  public void setBackground(Color c) {
    super.setBackground(c);
    if( slider != null ) slider.setBackground(c);
  }
  public int getValue() {
    return slider.getValue();
  }
  public void setValue(int velocity) {
    slider.setValue(velocity);
  }
}

///////////////////////////////////////////////
//
// MIDI Instrument (Program) - FI
//
///////////////////////////////////////////////
class MidiProgramSelecter extends JComboBox {
  private int family;
  private MidiProgramFamilySelecter family_selecter = null;
  public MidiProgramSelecter() {
    setFamily(-1);
  }
  public void setFamilySelecter( MidiProgramFamilySelecter mpfs ) {
    family_selecter = mpfs;
  }
  public void setFamily( int family ) {
    int program_no = getProgram();
    this.family = family;
    removeAllItems();
    if( family < 0 ) {
      setMaximumRowCount(16);
      for( int i=0; i < MIDISpec.instrument_names.length; i++ ) {
        addItem(i+": " + MIDISpec.instrument_names[i]);
      }
      setSelectedIndex(program_no);
    }
    else {
      setMaximumRowCount(8);
      for( int i=0; i < 8; i++ ) {
        program_no = i + family * 8;
        addItem( program_no + ": " + MIDISpec.instrument_names[program_no] );
      }
      setSelectedIndex(0);
    }
  }
  public int getProgram() {
    int program_no = getSelectedIndex();
    if( family > 0 && program_no >= 0 ) program_no += family * 8;
    return program_no;
  }
  public String getProgramName() { return (String)( getSelectedItem() ); }
  public void setProgram( int program_no ) {
    if( getItemCount() == 0 ) return; // To ignore event triggered by removeAllItems() 
    if( family >= 0 && program_no >= 0 && family == program_no / 8 ) {
      setSelectedIndex(program_no % 8);
    }
    else {
      if( family >= 0 ) setFamily(-1);
      if( family_selecter != null ) family_selecter.setSelectedIndex(0);
      if( program_no < getItemCount() ) setSelectedIndex(program_no);
    }
  }
}
class MidiProgramFamilySelecter extends JComboBox implements ActionListener {
  private MidiProgramSelecter program_selecter = null;
  public MidiProgramFamilySelecter() { this(null); }
  public MidiProgramFamilySelecter( MidiProgramSelecter mps ) {
    program_selecter = mps;
    setMaximumRowCount(17);
    addItem("Program:");
    for( int i=0; i < MIDISpec.instrument_family_names.length; i++ ) {
      addItem( (i*8) + "-" + (i*8+7) + ": " + MIDISpec.instrument_family_names[i] );
    }
    setSelectedIndex(0);
    addActionListener(this);
  }
  public void actionPerformed(ActionEvent evt) {
    if( program_selecter == null ) return;
    int i = getSelectedIndex();
    program_selecter.setFamily( i < 0 ? i : i-1 );
  }
  public int getProgram() {
    int i = getSelectedIndex();
    if( i <= 0 ) return -1;
    else return (i-1)*8;
  }
  public String getProgramFamilyName() { return (String)( getSelectedItem() ); }
  public void setProgram( int program_no ) {
    if( program_no < 0 ) program_no = 0;
    else program_no = program_no / 8 + 1;
    setSelectedIndex( program_no );
  }
}
