/* Copyright 2006-2007 (c) Takefumi MIYOSHI All rights reserved. */

package net.wasamon.wallet.binutils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;
import net.wasamon.mjlib.xml.XMLUtil;

import org.w3c.dom.Node;

public class TextRegion implements Section {

	private ArrayList programs = new ArrayList();

	/**
	 * プログラム中のラベルを保持するハッシュテーブル．ラベル名がキー，ラベルの指示する番地を値として保持する
	 */
//	private Hashtable<String, Integer> labels = new Hashtable<String, Integer>();
	private Hashtable labels = new Hashtable();

	/**
	 * プログラム中の定数データを保持するハッシュテーブル．定数データに振られたラベルがキー，DataElementを値として保持する
	 */
	private Hashtable constants = new Hashtable();

	private int offset;

	public void clear() {
		programs.clear();
		labels.clear();
		constants.clear();
	}

	private int align;
	
	public TextRegion(){
	}

	/** 与えられた文字列が即値を表わしているかどうか判定する */
	private boolean isImmidiate(String s) {
		if (s == null || s.equals(""))
			return false; // sがnullか""の場合
		else if (s.charAt(0) == 'r')
			return false;
		else
			return true;
	}

	public void addData(Parser parser, String value) throws AssemblerException {
		String[] inst = new String[2];
		int offset = 0;
		String[] token = parser.getLineItems(value);
		for (int i = 0; i < token.length; i++) {
			if (token[i].equals("")) {
			} else if (token[i].charAt(0) == ';') { // 以降コメント
				break;
			} else if (token[i].charAt(0) == '#') { // 以降コメント
				break;
			} else if (token[i].startsWith("/*")) { // 以降コメント
				break;
			} else { // 命令として登録
				if (offset < 2) {
					inst[offset] = token[i];
					offset++;
				} else { // プログラム一行は命令と引数で構成され，大きい場合はエラー
					throw new AssemblerException(value);
				}
			}
		}
		if (offset > 0) { // 命令文だった
			if (inst[1] != null) {
				programs.add(new ProgramCode(inst[0], parser.getArgumentItems(inst[1])));
			} else {
				programs.add(new ProgramCode(inst[0], null));
			}
		}
	}

	public void execCommand(Parser parser, String value) throws AssemblerException{
		if (value.startsWith(".constant")) {
			String a[] = parser.getArgumentItems(parser.getLineArgument(value));
			try{
				if (constants.containsKey(a[0]) == false) {
					DataElement d = new DataElement(constants.size());
					if(a[0].startsWith("c_")){
						d.type = DataElement.TYPE_INT;
						d.data.write(DataUtil.toByteArray(DataUtil.parseInt(a[0].substring(2))));
					}else if(a[0].startsWith("f_")){
						d.type = DataElement.TYPE_INT;
						d.data.write(DataUtil.toByteArray(Float.floatToIntBits(Float.parseFloat(a[0].substring(2)))));
					}
					constants.put(a[0], d);
				}
			}catch(IOException e){
				throw new AssemblerException(e.getMessage());
			}
		}else if(value.startsWith(".align")){
			align = DataUtil.parseInt(parser.getLineArgument(value));
		}
	}

	public void addLabel(String label) {
		labels.put(label, new Integer(programs.size()));
	}

	private void writeProgramCode(ByteArrayOutputStream out, Linker linker, InstructionTable inst) throws AssemblerException, IOException{
		Iterator it = programs.iterator();
		while(it.hasNext()){
			ProgramCode p = (ProgramCode)(it.next());
			ByteArrayOutputStream tmp = new ByteArrayOutputStream();
			tmp.write((byte) (inst.toInstructionCode(p.inst) & 0xff));
			for(int i = 0; i < 3; i++){
				if (p.args[i] == null){
					tmp.write((byte)0); // ダミーバイト
				}else if(isImmidiate(p.args[i])){
					tmp.write(DataUtil.toByteArray(linker.toValue(p.args[i]), 2));
				}else{
					tmp.write((byte)(linker.toValue(p.args[i])));
				}
			}
			out.write(tmp.toByteArray(), 0, 4);
		}
	}
	

	private void writeConstants(ByteArrayOutputStream out) throws IOException{
		ArrayList dataList = new ArrayList(constants.values());
		Collections.sort(dataList, new DataElementComparator());
		Iterator it = dataList.iterator();
		while(it.hasNext()){
			DataElement data = (DataElement)(it.next());
			out.write(data.data.toByteArray());
		}
	}

	public ByteArrayOutputStream genByteStream(Linker linker, InstructionTable inst) throws IOException, AssemblerException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		writeProgramCode(out, linker, inst);
		writeConstants(out);
		out.flush();
		return out;
	}

	public int getLabelValue(String label) {
		int val = -1;
		if(label.startsWith("_") && labels.containsKey(label.substring(1))){
			val = ((Integer)(labels.get(label.substring(1)))).intValue() * 4 + offset;
		}else if((label.startsWith("c_") || label.startsWith("f_")) && constants.containsKey(label)){
			val = (((DataElement)(constants.get(label))).offset + programs.size()) * 4 + offset;
		}
		return val;
	}

	public void setOffset(int offset){
		this.offset = offset;
	}
	
	public int size(){
		return (programs.size() + constants.size())*4;
	}

	public void outputAsXML(PrintWriter writer) throws IOException {
		Iterator it;
		it = programs.iterator();
		while(it.hasNext()){
			ProgramCode program = (ProgramCode)(it.next());
			program.outputAsXML(writer);
		}
		it = labels.entrySet().iterator();
		while(it.hasNext()){
			Map.Entry label = (Map.Entry)(it.next());
			writer.println(XMLUtil.toElementString("label", new String[]{"label", (String)(label.getKey()), "offset", label.getValue().toString()}));
		}
		it = constants.entrySet().iterator();
		while(it.hasNext()){
			Map.Entry constant = (Map.Entry)(it.next());
			((DataElement)(constant.getValue())).outputAsXML(writer, (String)(constant.getKey()));
		}
	}
	
	public void inputFromXML(Node node) throws XMLParserException{
		Node[] nodes;
		nodes = XMLParser.getNamedNodeArray(node, "code");
		for(int i = 0; i < nodes.length; i++){
			programs.add(ProgramCode.inputFromXML(nodes[i]));
		}
		nodes = XMLParser.getNamedNodeArray(node, "label");
		for(int i = 0 ; i < nodes.length; i++){
			String label = XMLParser.getAttribute(nodes[i], "label").getNodeValue();
			int offset = DataUtil.parseInt(XMLParser.getAttribute(nodes[i], "offset").getNodeValue());
			labels.put(label, new Integer(offset));
		}
		nodes = XMLParser.getNamedNodeArray(node, "data");
		for(int i = 0; i < nodes.length; i++){
			String label = XMLParser.getAttribute(nodes[i], "label").getNodeValue();
			constants.put(label, DataElement.inputFromXML(nodes[i]));
		}
	}
	
}
