/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * リコンフィギュラブルプロセッサRLUに関するパッケージ
 */
package net.wasamon.mics.processor.rlu;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.swing.JFrame;
import javax.swing.JPanel;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsViewable;
import net.wasamon.mics.memory.RandomAccessMemory;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.print.PrintFormat;
import net.wasamon.mjlib.util.DataUtil;

import org.w3c.dom.Node;

/**
 * 回路ユニット全体を表わす
 * @author Takefumi MIYOSHI
 */
public class ReconfigurableUnit extends MicsElement implements
		ExecutableElement, ChannelConnectable, DataBuffer, MicsViewable {

	/**
	 * 内部にもっているLogicUnitの配列
	 */
	LogicUnit[] unit;

	/**
	 * 内部にもっているLogicUnitの一辺のサイズ
	 */
	private int size;

	/** リコンフィギュラブルプロセッサへの入力バッファ */
	private ExternalDataInterface[] input;

	private String busID;

	private int ruAddr;

	private int contextAddr;

	private RandomAccessMemory context;

	/**
	 * コンストラクタ
	 * 
	 * @param size
	 *          一辺のRLBの個数．実際にはsize*size個のRLBを持つ．
	 * @param context
	 *          コンフィグ情報の格納されているメモリ
	 * @param offset
	 *          コンフィグ情報の格納されているメモリ上のオフセット
	 */
	public ReconfigurableUnit() {
	}

	private void initialize(int size) {
		this.context = new RandomAccessMemory(4 * size * size);
		this.contextAddr = 0;
		this.unit = new LogicUnit[size * size];
		this.input = new ExternalDataInterface[size];
	}
	
	private DataBuffer[] getSrcLogicUnitArray(int id){
		DataBuffer[] connections = new DataBuffer[6];
		if(id / size == 0){
			connections[ConfigDataAssembler.NonDirection] = input[id % size];
		}
		if(id % size == 0){
			connections[ConfigDataAssembler.West] = null;
		}else{
			connections[ConfigDataAssembler.West] = unit[id - 1];
		}
		if(id % size == 0){
			connections[ConfigDataAssembler.NorthWest] = null;
		}else if(id / size == 0){
			connections[ConfigDataAssembler.NorthWest] = input[id % size - 1];
		}else{
			connections[ConfigDataAssembler.NorthWest] = unit[id - size - 1];
		}
		if(id / size == 0){
			connections[ConfigDataAssembler.North] = null;
		}else{
			connections[ConfigDataAssembler.North] = unit[id - size];
		}
		if(id % size == size - 1){
			connections[ConfigDataAssembler.NorthEast] = null;
		}else if(id / size == 0){
			connections[ConfigDataAssembler.NorthEast] = input[id % size + 1];
		}else{
			connections[ConfigDataAssembler.NorthEast] = unit[id - size + 1];
		}
		if(id % size == size - 1){
			connections[ConfigDataAssembler.East] = null;
		}else{
			connections[ConfigDataAssembler.East] = unit[id + 1];
		}
		return connections;
	}

	private void setSize(String base, Node n) throws NumberFormatException{
		Integer value = getAttributeAsInt(n, "size");
		if(value == null){
			throw new NumberFormatException("size value is illegal");
		}
		this.size = value.intValue(); 
		initialize(this.size);
	}
	
	private void setChannel(String base, Node n) throws MicsException{
		Node n0 = getNamedNode(n, "channel");
		if(n0 == null){
			throw new MicsException("configuration syntax error for " + getClass().getName() + ": channel not found.");
		}
		Integer value = getAttributeAsInt(n0, "offset");
		if(value == null){
			throw new MicsException("configuration syntax error for " + getClass().getName() + ": channel offset is not defined.");
		}
		String id = getAttributeAsString(n0, "id");
		if(id == null){
			throw new MicsException("configuration syntax error for " + getClass().getName() + ": channel id is not defined.");
		}
		this.busID = id;
		this.ruAddr = value.intValue();
		for (int i = 0; i < size; i++) {
			// 一つのRLBの入力は 16bit x 2 なので，アドレスを4バイトずつずらす
			input[i] = new ExternalDataInterface(composite, busID, ruAddr + i * 4, null);
		}
		for (int i = 0; i < size; i++) { // 最後の行のRLBをマッピングする
			unit[size * (size - 1) + i].setChannel(busID, ruAddr + size * 4 + 4 * i, true);
		}
	}
	
	private void setInitialMemoryValue(String base, Node n) throws MicsException{
		Node[] init_var__init_obj = getNamedNodeArray(n, "init");
		for (int i = 0; i < init_var__init_obj.length; i++) {
			Node init_var__init_node = init_var__init_obj[i];
			String init_var__init_file;
			init_var__init_file = getAttributeAsString(init_var__init_node,	"file");
			if (init_var__init_file.charAt(0) != '/') {
				init_var__init_file = base + "/" + init_var__init_file;
			}
			Integer value;
			value = getAttributeAsInt(init_var__init_node, "offset");
			if(value == null){
				throw new MicsException("configuration syntax error for " + getClass().getName() + ": init offset is not defined.");
			}
			try {
				write(value.intValue(), new BufferedInputStream(new FileInputStream(init_var__init_file)));
			} catch (FileNotFoundException e) {
				System.out.println("no such file: " + init_var__init_file);
				System.out.println("[W] no data is written as initialize.");
			}
		}
	}
	
	public void initialize(String base, Node n) throws MicsException {
		try {
			setSize(base, n);
			for(int i = 0; i < size * size; i++){
				unit[i] = new LogicUnit(composite);
			}
			setChannel(base, n);
			for(int i = 0; i < size*size; i++){
				unit[i].initialize(getSrcLogicUnitArray(i));
			}
			setInitialMemoryValue(base, n);
		} catch (NumberFormatException e) {
			throw new MicsException("configuration syntax error for " + getClass().getName());
		}
	}

	public int size() {
		return unit.length;
	}

	public MicsDataPacket read(MicsDataPacket d) {
		return context.read(d);
	}

	public void write(MicsDataPacket d) {
		context.write(d);
		unit[((RandomAccessMemoryDataPacket)d).addr/4].setLogicUnit(d);
	}

	public void write(int addr, InputStream reader) throws DataBufferException {
		context.write(addr, reader);
		setLogicUnit();
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		context.dump(offset, len, writer);
	}

	public String toString(int addr, int length) {
		return context.toString(addr, length);
	}

	public void reset() {
		for (int i = 0; i < unit.length; i++) {
			unit[i].reset();
		}
	}

	private char busbuffer;

	public void writeback(Channel src, MicsDataPacket value) {
		busbuffer = DataUtil.toChar(((RandomAccessMemoryDataPacket) value).data, 0,	2);
	}

	public int getChannelOffset(Channel c) {
		return ruAddr;
	}

	private Channel busInstance = null;

	private Channel bus() throws MicsException {
		if (busInstance == null) {
			busInstance = composite.getChannel(busID);
		}
		return busInstance;
	}

	public char getInputData(int addr) throws MicsException {
		bus().readRequest(this,	RandomAccessMemoryDataPacket.readPacket(addr, 1, 16));
		return busbuffer;
	}

	/**
	 * ReconfigurableUnitの1ステップの実行
	 */
	public ExecInfo exec_first() throws MicsException {
		ExecInfo info = new ExecInfo();
		for (int i = 0; i < unit.length; i++) {
			unit[i].exec_first();
		}
		info.setTerminatableFlag(true);
		return info;
	}

	public ExecInfo exec_second() throws MicsException{
		for (int i = 0; i < unit.length; i++) {
			unit[i].exec_second();
		}
		return null;
	}

	private char readData(DataBuffer buf, int reg) throws MicsException{
		MicsDataPacket p = buf.read(RandomAccessMemoryDataPacket.readPacket(reg*2, 2, 8));
		return DataUtil.toChar(((RandomAccessMemoryDataPacket)p).data, 0, 2);
	}
	
	public String toString() {
		String str = "";
		try {
			str += "==\n";
			for (int j = 0; j < size; j++) {
				str += PrintFormat.print("%04x", (int) (readData(input[j], 0)));
				str += ":";
				str += PrintFormat.print("%04x", (int) (readData(input[j], 1)));
				str += ":";
				str += PrintFormat.print("%04x", (int) (readData(input[j], 2)));
				str += " ";
			}
			str += "\n";
			str += "--\n";
			for (int i = 0; i < size; i++) {
				for (int j = 0; j < size; j++) {
					str += PrintFormat.print("%04x", (int) (readData(unit[i * size + j], 0)));
					str += ":";
					str += PrintFormat.print("%04x", (int) (readData(unit[i * size + j], 1)));
					str += ":";
					str += PrintFormat.print("%04x", (int) (readData(unit[i * size + j], 2)));
					str += " ";
				}
				str += "\n";
			}
		} catch (FormatException e) {
			str += "catch: " + e.getMessage();
		} catch (MicsException e) {
			str += "catch: " + e.getMessage();
		}

		return str;
	}

	private void setLogicUnit(){
		for(int i = 0; i < size*size; i++){
			unit[i].setLogicUnit(context.read(RandomAccessMemoryDataPacket.readPacket(contextAddr + i * 4, 4, 8)));
		}
	}

	public String[] getConnectedElements() {
		return new String[] { busID };
	}

	public String getInfo() {
		String s = "";
		s += "ReconfigurableUnit\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  ID: " + id() + "\n";
		s += "  Local Memory ID: " + ((MicsElement) context).id() + ",";
		s += " offset = " + contextAddr + "\n";
		s += "  Channel ID: ";
		try {
			s += ((MicsElement) bus()).id() + ",";
		} catch (MicsException e) {
			s += "unconnected,";
		}
		s += " input offset = " + ruAddr + ",";
		s += " output offset = " + (ruAddr + size * 4);
		return s;
	}

	private ConfigDataViewer viewer = null;

	public void show() {
		if (viewer == null) {
			viewer = new ConfigDataViewer();
		}
		viewer.setReconfigurableUnit(this);
		viewer.repaint();
	}

	public String getImagePath() {
		return "cb_rlu.png";
	}


	public String getDescription(){
		return "TODO";
	}

}

class ConfigDataViewer {

	private ConfigDataCanvas canvas;

	private JFrame frame;

	public ConfigDataViewer() {
		frame = new JFrame("Config Data Viewer");
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				frame.setVisible(false);
			}
		});
		frame.setSize(600, 600);
		frame.getContentPane().add((canvas = new ConfigDataCanvas()));
		frame.setVisible(false);
	}

	public void setReconfigurableUnit(ReconfigurableUnit rp) {
		canvas.setReconfigurableUnit(rp);
		frame.setTitle("Config Data Viewer (ID=" + rp.id() + ")");
		frame.setVisible(true);
	}

	public void repaint() {
		canvas.repaint();
	}

	class ConfigDataCanvas extends JPanel {
		/**
		 * generated by eclipse
		 */
		private static final long serialVersionUID = 1L;

		private ReconfigurableUnit rp;

		private ConfigDataCanvas() {
			this.rp = null;
		}

		private void setReconfigurableUnit(ReconfigurableUnit rp) {
			this.rp = rp;
			repaint();
		}

		private int offset = 60;

		private int luwidth = 60;

		private int luheight = 60;

		private int px(int x) {
			return offset + x * luwidth;
		}

		private int py(int y) {
			return offset + y * luheight;
		}

		private void paintLogicUnit(Graphics g, int n) {
			for (int i = 0; i < n; i++) {
				for (int j = 0; j < n; j++) {
					g.drawRect(px(j), py(i), luwidth, luheight);
					LogicUnit lu0 = rp.unit[i * n + j];
					g.drawString(lu0.getLogicUnitInstString(), px(j) + 5, py(i) + 14);
					g.drawString(lu0.getShiftOpString(), px(j) + 5, py(i) + 28);
				}
			}
		}

		private Color getRegColor(int regID){
			if (regID == 0) {
				return Color.blue;
			} else {
				return Color.red;
			}
		}
		
		private Point getDirectinValue(int dir){
			switch (dir) {
			case ConfigDataAssembler.NorthWest:
				return new Point(-1, -1);
			case ConfigDataAssembler.North:
				return new Point(0,  -1);
			case ConfigDataAssembler.NorthEast:
				return new Point(1, -1);
			case ConfigDataAssembler.West:
				return new Point(-1, 0);
			case ConfigDataAssembler.East:
				return new Point(1, 0);
			default:
				return new Point(0, 0);
			}
		}
		
		private void paintDirection(Graphics g, int n) {
			for (int i = 0; i < n; i++) {
				for (int j = 0; j < n; j++) {
					LogicUnit lu0 = rp.unit[i * n + j];
					{
						g.setColor(getRegColor(lu0.srcReg[0].regID));
						Point p = getDirectinValue(lu0.srcReg[0].dir);
						g.drawLine(px(j) + 28, py(i) + 28, px(j) + 28 + p.x * 60, py(i) + 28 + p.y * 60);
					}
					{
						g.setColor(getRegColor(lu0.srcReg[1].regID));
						Point p = getDirectinValue(lu0.srcReg[1].dir);
						g.drawLine(px(j) + 32, py(i) + 32, px(j) + 32 + p.x * 60, py(i) + 32 + p.y * 60);
					}
				}
			}
			g.setColor(Color.black);
		}

		public void paintComponent(Graphics g) {
			super.paintComponent(g);
			if (rp == null) {
				return;
			} else {
				int n = (int) (Math.sqrt(rp.size()));
				paintDirection(g, n);
				paintLogicUnit(g, n);
			}
		}
	}

}
