/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * リコンフィギュラブルプロセッサRLUに関するパッケージ
 */
package net.wasamon.mics.processor.rlu;

import java.io.InputStream;
import java.io.OutputStream;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.ConfigErrorException;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.MicsCompositeElement;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.wallet.binutils.UnknownInstructionException;

import org.w3c.dom.Node;

/**
 * 一つの演算ユニット
 * @author Takefumi MIYOSHI
 *
 */
public class LogicUnit extends MicsElement implements DataBuffer, ExecutableElement, ChannelConnectable{

	public static final int major_version = 1;
	public static final int minor_version = 0;
	public static final int revision = 0;

	public static final int INST_NOP = 0;
	public static final int INST_ADD = 1;
	public static final int INST_SUB = 2;
	public static final int INST_AND = 3;
	public static final int INST_OR = 4;
	public static final int INST_MUL0 = 5;
	public static final int INST_MULT = 6;

	public static final int SHIFT_NONE = 0;
	public static final int SHIFT_LEFT = 1;
	public static final int SHIFT_RIGHT = 2;

	private char[] input = new char[3];
	private char[] output = new char[3];

	private int preShifter = SHIFT_NONE;
	private int postShifter = SHIFT_NONE;
	private int inst = INST_NOP;
	
	private DataBuffer[] connects;
	
	InputRegister[] srcReg = new InputRegister[2];
	
	/**
	 * エンジン経由ではなく独自に生成する場合のコンストラクタ
	 * @param composite
	 */
	LogicUnit(MicsCompositeElement composite){
		this.composite = composite;
	}
	
	public void initialize(DataBuffer[] connects){
		this.connects = connects;
		srcReg[0] = new InputRegister();
		srcReg[1] = new InputRegister();
		reset();
	}

	public void reset() {
		input[0] = (char) 0;
		input[1] = (char) 0;
		input[2] = (char) 0;
		output[0] = (char) 0;
		output[1] = (char) 0;
		output[2] = (char) 0;
	}

	private int shifter(int s, int value) throws ConfigErrorException {
		switch (s) {
		case SHIFT_NONE:
			break;
		case SHIFT_LEFT:
			value = value << 1;
			break;
		case SHIFT_RIGHT:
			value = value >> 1;
			break;
		default:
			throw new ConfigErrorException();
		}
		return value;
	}

	private String busID;
	private Channel bus = null;
	private int addr;
	private boolean fWriteOutput = false;
	
	void setChannel(String busID, int addr, boolean fWriteOutput){
		this.busID = busID;
		this.addr = addr;
		this.fWriteOutput = fWriteOutput;
	}
	
  /**
   * この演算にわりあたっている演算を入力データに施す
   */
	public ExecInfo exec_first() throws MicsException{
		ExecInfo info = new ExecInfo();
		DataBuffer src;
		if((src = connects[srcReg[0].dir]) != null){
			RandomAccessMemoryDataPacket d = (RandomAccessMemoryDataPacket)(src.read(RandomAccessMemoryDataPacket.readPacket(srcReg[0].regID * 2, 2, 8)));
			input[0] = DataUtil.toChar(d.data, 0, 2); 
		}
		if((src = connects[srcReg[1].dir]) != null){
			RandomAccessMemoryDataPacket d = (RandomAccessMemoryDataPacket)(src.read(RandomAccessMemoryDataPacket.readPacket(srcReg[1].regID * 2, 2, 8)));
			input[1] = DataUtil.toChar(d.data, 0, 2); 
		}
		if((src = connects[srcReg[0].dir]) != null){
			RandomAccessMemoryDataPacket d = (RandomAccessMemoryDataPacket)(src.read(RandomAccessMemoryDataPacket.readPacket(2 * 2, 2, 8)));
			input[2] = DataUtil.toChar(d.data, 0, 2); 
		}
		info.setTerminatableFlag(true);
		return info;
	}
	
	public ExecInfo exec_second() throws MicsException{
		ExecInfo info = new ExecInfo();
		int tmp = DataUtil.toInteger(input[0], input[1]);
		tmp = shifter(preShifter, tmp);

		switch (inst) {
		case INST_NOP:
			break;
		case INST_ADD:
			tmp = input[0] + input[1];
			break;
		case INST_SUB:
			tmp = input[0] - input[1];
			break;
		case INST_AND:
			tmp = input[0] & input[1];
			break;
		case INST_OR:
			tmp = input[0] | input[1];
			break;
		case INST_MUL0:
			if ((input[1] & 0x01) == 1) {
				tmp = input[0];
				tmp = DataUtil.toInteger(DataUtil.toCharL(tmp), input[1]);
			} else {
				tmp = DataUtil.toInteger((char) 0, input[1]);
			}
			break;
		case INST_MULT:
			if ((input[1] & 0x01) == 1) {
				tmp = input[0] + input[2];
				tmp = DataUtil.toInteger(DataUtil.toCharL(tmp), input[1]);
			}
			break;
		default:
			throw new ConfigErrorException();
		}

		tmp = shifter(postShifter, tmp);

		output[0] = DataUtil.toCharH(tmp);
		output[1] = DataUtil.toCharL(tmp);
		output[2] = input[2];
		if(fWriteOutput) writeOutput();
		info.setTerminatableFlag(true);
		return info;
	}
	
	private Channel bus() throws MicsException{
		if(bus == null){
			bus = composite.getChannel(busID);
		}
		return bus;
	}
	
	private void writeOutput() throws MicsException{
		MicsDataPacket p;
		p = RandomAccessMemoryDataPacket.writePacket(addr, 1, 16, DataUtil.toByteArray(output[0], 2));
		bus().writeRequest(this, p);
		p = RandomAccessMemoryDataPacket.writePacket(addr+2, 1, 16, DataUtil.toByteArray(output[1], 2));
		bus().writeRequest(this, p);
	}
	
	public void setLogicUnit(MicsDataPacket p){
		byte[] data = ((RandomAccessMemoryDataPacket)p).data;
		setInst(data[0] & 0x00ff);
		setPreSfhiter((data[1] >> 2) & 0x03);
		setPostShifter(data[1] & 0x03);
		setSrcRegisterRegisterID(0, (data[1] >> 7) & 0x01);
		setSrcRegisterRegisterID(1, (data[1] >> 6) & 0x01);
		setSrcRegisterDir(0, data[2] & 0x00ff);
		setSrcRegisterDir(1, data[3] & 0x00ff);
	}

	private void setInst(int value){
		this.inst = value;
	}

	private void setPreSfhiter(int value){
		this.preShifter = value;
	}
	
	private void setPostShifter(int value){
		this.postShifter = value;
	}
	
	private void setSrcRegisterRegisterID(int reg, int value){
		srcReg[reg].regID = value;
	}
	
	private void setSrcRegisterDir(int reg, int value){
		srcReg[reg].dir = value;
	}
	
	public String getShiftOpString() {
		String str = "";
		switch (preShifter) {
		case SHIFT_RIGHT:
			str += "R";
			break;
		case SHIFT_LEFT:
			str += "L";
			break;
		case SHIFT_NONE:
			str += "N";
			break;
		default:
			str += "E";
		}
		switch (postShifter) {
		case SHIFT_RIGHT:
			str += "R";
			break;
		case SHIFT_LEFT:
			str += "L";
			break;
		case SHIFT_NONE:
			str += "N";
			break;
		default:
			str += "E";
		}
		return str;
	}

	public String getLogicUnitInstString() {
		switch (inst) {
		case LogicUnit.INST_NOP:
			return "NOP  ";
		case LogicUnit.INST_ADD:
			return "ADD  ";
		case LogicUnit.INST_SUB:
			return "SUB  ";
		case LogicUnit.INST_AND:
			return "AND  ";
		case LogicUnit.INST_OR:
			return "OR   ";
		case LogicUnit.INST_MUL0:
			return "MUL0 ";
		case LogicUnit.INST_MULT:
			return "MULT ";
		default:
			return "ERROR";
		}
	}

	public static int getInstructionCode(String str)
			throws UnknownInstructionException {
		if (str == null || str.equals(""))
			throw new UnknownInstructionException();
		else if (str.equals("NOP"))
			return INST_NOP;
		else if (str.equals("ADD"))
			return INST_ADD;
		else if (str.equals("SUB"))
			return INST_SUB;
		else if (str.equals("AND"))
			return INST_AND;
		else if (str.equals("OR"))
			return INST_OR;
		else if (str.equals("MUL0"))
			return INST_MUL0;
		else if (str.equals("MULT"))
			return INST_MULT;
		else
			throw new UnknownInstructionException();
	}

	public static int getShiftOpCode(String str)
			throws UnknownInstructionException {
		if (str == null || str.equals(""))
			throw new UnknownInstructionException();
		else if (str.equals("RIGHT"))
			return SHIFT_RIGHT;
		else if (str.equals("LEFT"))
			return SHIFT_LEFT;
		else if (str.equals("NONE"))
			return SHIFT_NONE;
		else
			throw new UnknownInstructionException();
	}

	
	class InputRegister{

		int regID;
		int dir;

	}


	public String[] getConnectedElements() {
		// TODO Auto-generated method stub
		return null;
	}

	public String getImagePath() {
		// TODO Auto-generated method stub
		return null;
	}

	public void initialize(String base, Node node) throws MicsException {
		// TODO Auto-generated method stub
		
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		// TODO Auto-generated method stub
		
	}

	public MicsDataPacket read(MicsDataPacket data) {
		RandomAccessMemoryDataPacket p = (RandomAccessMemoryDataPacket)data;
		return RandomAccessMemoryDataPacket.writePacket(p.addr, 1, 16, DataUtil.toByteArray(output[p.addr/2]));
	}

	public MicsDataPacket read(Channel src, MicsDataPacket data) {
		return read(data);
	}

	public int size() {
		// TODO Auto-generated method stub
		return 0;
	}

	public String toString(int addr, int length) {
		// TODO Auto-generated method stub
		return null;
	}

	public void write(MicsDataPacket data) {
		// TODO Auto-generated method stub
		
	}

	public void write(Channel src, MicsDataPacket data) {
		// TODO Auto-generated method stub
		
	}

	public void write(int offset, InputStream reader) throws DataBufferException {
		// TODO Auto-generated method stub
		
	}

	public int getChannelOffset(Channel c) {
		// TODO Auto-generated method stub
		return 0;
	}

	public void writeback(Channel src, MicsDataPacket data) throws MicsException {
		// TODO Auto-generated method stub
		
	}
	public String getDescription(){
		return "TODO";
	}

}
