/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * リコンフィギュラブルプロセッサRLUに関するパッケージ
 */
package net.wasamon.mics.processor.monorlu;

import net.wasamon.mics.ConfigErrorException;
import net.wasamon.mics.MicsException;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.wallet.binutils.UnknownInstructionException;

/**
 * 一つの演算ユニット
 * @author Takefumi MIYOSHI
 *
 */
public class LogicUnit implements DataProvider {

	public static final int major_version = 0;
	public static final int minor_version = 1;

	public static final int INST_NOP = 0;
	public static final int INST_ADD = 1;
	public static final int INST_SUB = 2;
	public static final int INST_AND = 3;
	public static final int INST_OR = 4;
	public static final int INST_MUL0 = 5;
	public static final int INST_MULT = 6;

	public static final int SHIFT_NONE = 0;
	public static final int SHIFT_LEFT = 1;
	public static final int SHIFT_RIGHT = 2;

	private char[] input = new char[3];
	private char[] output = new char[3];

	private int id;

	private ReconfigurableUnit unit;

	public LogicUnit(ReconfigurableUnit unit, int id) {
		this.id = id;
		this.unit = unit;
		reset();
	}

	public void reset() {
		input[0] = (char) 0;
		input[1] = (char) 0;
		input[2] = (char) 0;
		output[0] = (char) 0;
		output[1] = (char) 0;
		output[2] = (char) 0;
	}

  /**
   * DataProviderで規定されているメソッド．指定した番号のレジスタの値を返す．
   */
	public char getData(int i) {
		return output[i];
	}

	public void load() throws MicsException {
		input[0] = unit.getInputSrcLogicUnit(id, 0).getData(
				unit.getInputSrcRegister(id, 0));
		input[1] = unit.getInputSrcLogicUnit(id, 1).getData(
				unit.getInputSrcRegister(id, 1));
		input[2] = unit.getInputSrcLogicUnit(id, 0).getData(2);
	}

	private int shifter(int s, int value) throws ConfigErrorException {
		switch (s) {
		case SHIFT_NONE:
			break;
		case SHIFT_LEFT:
			value = value << 1;
			break;
		case SHIFT_RIGHT:
			value = value >> 1;
			break;
		default:
			throw new ConfigErrorException();
		}
		return value;
	}

  /**
   * この演算にわりあたっている演算を入力データに施す
   */
	public void exec() throws ConfigErrorException {

		int tmp = DataUtil.toInteger(input[0], input[1]);
		tmp = shifter(unit.getLogicUnitPreShift(id), tmp);

		switch (unit.getLogicUnitInst(id)) {
		case INST_NOP:
			break;
		case INST_ADD:
			tmp = input[0] + input[1];
			break;
		case INST_SUB:
			tmp = input[0] - input[1];
			break;
		case INST_AND:
			tmp = input[0] & input[1];
			break;
		case INST_OR:
			tmp = input[0] | input[1];
			break;
		case INST_MUL0:
			if ((input[1] & 0x01) == 1) {
				tmp = input[0];
				tmp = DataUtil.toInteger(DataUtil.toCharL(tmp), input[1]);
			} else {
				tmp = DataUtil.toInteger((char) 0, input[1]);
			}
			break;
		case INST_MULT:
			if ((input[1] & 0x01) == 1) {
				tmp = input[0] + input[2];
				tmp = DataUtil.toInteger(DataUtil.toCharL(tmp), input[1]);
			}
			break;
		default:
			throw new ConfigErrorException();
		}

		tmp = shifter(unit.getLogicUnitPostShift(id), tmp);

		output[0] = DataUtil.toCharH(tmp);
		output[1] = DataUtil.toCharL(tmp);
		output[2] = input[2];

	}

	public String getShiftOpString() {
		int preShift = unit.getLogicUnitPreShift(this.id);
		int postShift = unit.getLogicUnitPostShift(this.id);
		String str = "";
		switch (preShift) {
		case SHIFT_RIGHT:
			str += "R";
			break;
		case SHIFT_LEFT:
			str += "L";
			break;
		case SHIFT_NONE:
			str += "N";
			break;
		default:
			str += "E";
		}
		switch (postShift) {
		case SHIFT_RIGHT:
			str += "R";
			break;
		case SHIFT_LEFT:
			str += "L";
			break;
		case SHIFT_NONE:
			str += "N";
			break;
		default:
			str += "E";
		}
		return str;
	}

	public String getLogicUnitInstString() {
		switch (unit.getLogicUnitInst(id)) {
		case LogicUnit.INST_NOP:
			return "NOP  ";
		case LogicUnit.INST_ADD:
			return "ADD  ";
		case LogicUnit.INST_SUB:
			return "SUB  ";
		case LogicUnit.INST_AND:
			return "AND  ";
		case LogicUnit.INST_OR:
			return "OR   ";
		case LogicUnit.INST_MUL0:
			return "MUL0 ";
		case LogicUnit.INST_MULT:
			return "MULT ";
		default:
			return "ERROR";
		}
	}

	public static int getInstructionCode(String inst)
			throws UnknownInstructionException {
		if (inst == null || inst.equals(""))
			throw new UnknownInstructionException();
		else if (inst.equals("NOP"))
			return INST_NOP;
		else if (inst.equals("ADD"))
			return INST_ADD;
		else if (inst.equals("SUB"))
			return INST_SUB;
		else if (inst.equals("AND"))
			return INST_AND;
		else if (inst.equals("OR"))
			return INST_OR;
		else if (inst.equals("MUL0"))
			return INST_MUL0;
		else if (inst.equals("MULT"))
			return INST_MULT;
		else
			throw new UnknownInstructionException();
	}

	public static int getShiftOpCode(String str)
			throws UnknownInstructionException {
		if (str == null || str.equals(""))
			throw new UnknownInstructionException();
		else if (str.equals("RIGHT"))
			return SHIFT_RIGHT;
		else if (str.equals("LEFT"))
			return SHIFT_LEFT;
		else if (str.equals("NONE"))
			return SHIFT_NONE;
		else
			throw new UnknownInstructionException();
	}

	public String getShiftStr(int shift) {
		switch (shift) {
		case SHIFT_NONE:
			return "NONE ";
		case SHIFT_LEFT:
			return "LEFT ";
		case SHIFT_RIGHT:
			return "RIGHT";
		default:
			return "ERROR";
		}
	}

}
