/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * リコンフィギュラブルプロセッサRLUに関するパッケージ
 */
package net.wasamon.mics.processor.monorlu;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

import net.wasamon.wallet.binutils.AssemblerException;
import net.wasamon.wallet.binutils.UnknownInstructionException;

/**
 * テキストデータからコンフィギュレーションデータを作成するRLUのためのアセンブラ
 * @author Takefumi MIYOSHI
 *
 */
public class ConfigDataAssembler {

	/**
	 * ConfigDataを保持する
	 */
	private ArrayList programs = new ArrayList();

	private String currentFile;

	private int programsize;

	public void parser(String f) throws IOException, FileNotFoundException {
		try {
			currentFile = f;
			programs.clear();
			File file = new File(f);
			File dest = parser(file);
			System.out.println(f + " -> " + file.getParent() + "/" + dest.getName());
		} catch (Exception e) {
			System.out.println(e);
			System.exit(0);
		}
	}

	public File parser(File file) throws UnknownInstructionException,
			AssemblerException, IOException, FileNotFoundException {
		BufferedReader reader = new BufferedReader(new FileReader(file));
		String line = "";
		int count = 0;
		while ((line = reader.readLine()) != null) {
			count++;
			parseLine(line, count);
		}
		ByteArrayOutputStream hex = null;
		hex = getObjectCode();
		int i = file.getName().lastIndexOf('.');
		File dest = new File(file.getParent(), file.getName().substring(0, i)
				+ ".cnf");
		FileOutputStream fos = new FileOutputStream(dest);
		hex.writeTo(fos);
		fos.flush();
		fos.close();
		hex.close();
		return dest;
	}

	private ByteArrayOutputStream getObjectCode() throws IOException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		byte[] context = new byte[programsize * programsize * 4];
		ConfigData data = null;
		for (int i = 0; i < programs.size(); i++) {
			data = (ConfigData) (programs.get(i));
			int id = (data.id[0] * programsize) + data.id[1];
			context[4 * id] = data.inst;
			context[4 * id + 1] = data.shiftOp;
			context[4 * id + 2] = (byte) ((data.src0[0] * programsize) + data.src0[1]);
			context[4 * id + 3] = (byte) ((data.src1[0] * programsize) + data.src1[1]);
		}
		out.write(context, 0, context.length);
		out.flush();
		return out;
	}

  /** アセンブラ指示文のパーズ */
	private void parseAssemblerIndicate(String str) throws AssemblerException {
		StringTokenizer st = new StringTokenizer(str, " ");
		String token[] = new String[st.countTokens()];
		for (int i = 0; i < token.length; i++) {
			token[i] = st.nextToken();
		}
		if (".size".equals(token[0])) {
			try {
				this.programsize = Integer.parseInt(token[1]);
			} catch (NumberFormatException e) {
				throw new AssemblerException("syntax error");
			}
		} else {
			throw new AssemblerException("Unknown parameter: " + token[0]);
		}
	}

  /** 一行の空白を取りのぞくためのパターン */
	private Pattern linePattern = Pattern.compile("[, ]+");

  /**
   * linePatternを使ってパーズする．
   * @param str 行の文字列
   * @param number 現在の行番号
   */
	private void parseLine(String str, int number)
			throws UnknownInstructionException, AssemblerException {
		if (str.equals(""))
			return;
		String[] token = null;
		token = linePattern.split(str);
		if (token[0].charAt(0) == '#') {
			return; // コメント
		} else if (token[0].charAt(0) == '.') {
			try {
				parseAssemblerIndicate(str); // 指示文
				return;
			} catch (AssemblerException e) {
				throw new AssemblerException(e.getMessage() + " " + currentFile
						+ " at line " + number);
			}
		} else if (token.length != 6) {
			for (int i = 0; i < token.length; i++) {
			}
			throw new AssemblerException("syntax error: " + currentFile + " at line "
					+ number);
		}
		programs.add(new ConfigData(token));
	}

	class ConfigData {
		public int[] id;
		public byte inst;
		public byte shiftOp;
		public int[] src0;
		public int[] src1;

		public ConfigData(String[] token) throws UnknownInstructionException {
			inst = (byte) (LogicUnit.getInstructionCode(token[1].toUpperCase()));
			src0 = getLogicUnitID(token[4]);
			src1 = getLogicUnitID(token[5]);
			id = getLogicUnitID(token[0]);
			shiftOp = (byte) ((LogicUnit.getShiftOpCode(token[2].toUpperCase()) << 2) + LogicUnit
					.getShiftOpCode(token[3].toUpperCase()));
			if (src0[2] == 0) {
			} else if (src0[2] == 1) {
				shiftOp |= (byte) 0x80;
			} else {
				throw new UnknownInstructionException("Illegal source register id = "
						+ src0[2] + " for logic unit(0)");
			}
			if (src1[2] == 0) {
			} else if (src1[2] == 1) {
				shiftOp |= (byte) 0x40;
			} else {
				throw new UnknownInstructionException("Illegal source register id = "
						+ src1[2] + " for logic unit(1)");
			}
		}

    /** 引数を:で分割するパターン */
		private Pattern argPattern = Pattern.compile(":");

		private int[] getLogicUnitID(String s) throws UnknownInstructionException {
			String[] token = argPattern.split(s);
			int[] id = new int[token.length];
			for (int i = 0; i < token.length; i++) {
				id[i] = Integer.parseInt(token[i]);
			}
			return id;
		}

	}

	public static void main(String[] args) throws Exception {
		if (args.length == 0) {
			System.out.println("no input file");
			System.exit(0);
		}
		ConfigDataAssembler asm = new ConfigDataAssembler();
		for (int i = 0; i < args.length; i++) {
			asm.parser(args[i]);
		}
	}

}
