package net.wasamon.mics.processor.mips;

public class R3010 {

    private static final int REGISTER_SIZE = 32;

    private MipsFloatRegister[] register = new MipsFloatRegister[REGISTER_SIZE];

    private MipsIntRegister fcr0, fcr31;

    public R3010() {
	for(int i = 0; i < REGISTER_SIZE; ++i){
	    register[i] = new MipsFloatRegister();
	}
	fcr0 = new MipsIntRegister();
	fcr31 = new MipsIntRegister();
    }

    protected int getC(){
	return (fcr0.getAsIntValue() & 0x00800000) >>> 23;
    }

    private int getExceptions(){
	return (fcr0.getAsIntValue() & 0x0003f000) >>> 12;
    }

    private int getTrapEnable(){
	return (fcr0.getAsIntValue() & 0x00000f80) >>> 7;
    }

    private int getStickyBits(){
	return (fcr0.getAsIntValue() & 0x0000007c) >>> 2;
    }

    private int getRM(){
	return (fcr0.getAsIntValue() & 0x00000003);
    }

    private int getImp(){
	return (fcr31.getAsIntValue() & 0x0000ff00) >>> 8;
    }

    private int getRev(){
	return (fcr31.getAsIntValue() & 0x000000ff);
    }


    private void setC(int val){
	fcr0.setAsBits((fcr0.getAsIntValue() & 0xff7fffff) | ((val << 23) & 0x00800000));
    }

    private void setExceptions(int val){
	fcr0.setAsBits((fcr0.getAsIntValue() & 0xfffc0fff) | ((val << 12) & 0x0003f000));
    }

    private void setTrapEnable(int val){
	fcr0.setAsBits((fcr0.getAsIntValue() & 0xfffff07f) | ((val << 7) & 0x00000f80));
    }

    private void setStickyBits(int val){
	fcr0.setAsBits((fcr0.getAsIntValue() & 0xffffff83) | ((val << 2) & 0x0000007c));
    }

    private void setRM(int val){
	fcr0.setAsBits((fcr0.getAsIntValue() & 0xfffffffc) | (val & 0x00000003));
    }

    protected MipsFloatRegister getRegister(int index) {
	return register[index];
    }

    protected MipsIntRegister getControlRegister(int index) {
	if(index == 0){
	    return fcr0;
	}
	else if(index == 31){
	    return fcr31;
	}
	else{
	    return null;
	}
    }

    private void setFPR(int index, double val){
	setFPR(index, Double.doubleToLongBits(val));
    }

    private void setFPR(int index, long bits){
	int valh = (int)((bits & 0xffffffff00000000l) >>> 32);
	int vall = (int)(bits & 0x00000000ffffffffl);
	getRegister(index + 1).setAsBits(valh);
	getRegister(index).setAsBits(vall);
    }

    private double getFPRasDouble(int index){
	return Double.longBitsToDouble(getFPRasBits(index));
    }

    private long getFPRasBits(int index){
	long ftl = (getRegister(index).getAsIntValue() & 0x00000000ffffffffl);
	long fth = (getRegister(index + 1).getAsIntValue() & 0x00000000ffffffffl);
	long bits = 0x0000000000000000l;

	bits |= (fth << 32);
	bits |= ftl;

	return bits;
    }

    protected void exec(MipsOperation op){
	MipsFloatRegister regFT, regFS, regFD;

	switch (op.getInstruction()) {
	case MipsOperation.INST_ABS_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(Math.abs(regFS.getAsFloatValue()));
	    break;

	case MipsOperation.INST_ABS_D:
	    {
		setFPR(op.getZeroExtendedFD(), Math.abs(getFPRasDouble(op.getZeroExtendedFS())));
	    }
	    break;

	case MipsOperation.INST_ADD_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFT = getRegister(op.getZeroExtendedFT());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(regFS.getAsFloatValue() + regFT.getAsFloatValue());
	    break;

	case MipsOperation.INST_ADD_D:
	    {
		double result = getFPRasDouble(op.getZeroExtendedFS()) + getFPRasDouble(op.getZeroExtendedFT());
		setFPR(op.getZeroExtendedFD(), result);
	    }
	    break;

	case MipsOperation.INST_C_F_S:
	    setC(0);
	    break;

	case MipsOperation.INST_C_F_D:
	    setC(0);
	    break;

	case MipsOperation.INST_C_UN_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_UN_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_EQ_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft == ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_EQ_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft == dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_UEQ_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft == ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_UEQ_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft == dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_OLT_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft > ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_OLT_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft > dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_ULT_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft > ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_ULT_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft > dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_OLE_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft >= ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_OLE_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft >= dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_ULE_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft >= ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_ULE_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft >= dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_SF_S:
	    setC(0);
	    break;

	case MipsOperation.INST_C_SF_D:
	    setC(0);
	    break;

	case MipsOperation.INST_C_NGLE_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGLE_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_SEQ_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft == ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_SEQ_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft == dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGL_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft == ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGL_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft == dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_LT_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft > ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_LT_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft > dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGE_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft > ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGE_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft > dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_LE_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN){
		    setC(0);
		}
		else if(fft >= ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_LE_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN){
		    setC(0);
		}
		else if(dft >= dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGT_S:
	    {
		float fft = getRegister(op.getZeroExtendedFT()).getAsFloatValue();
		float ffs = getRegister(op.getZeroExtendedFS()).getAsFloatValue();
		
		if(fft == Float.NaN || ffs == Float.NaN || fft >= ffs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_C_NGT_D:
	    {
		double dft = getFPRasDouble(op.getZeroExtendedFT());
		double dfs = getFPRasDouble(op.getZeroExtendedFS());
		
		if(dft == Double.NaN || dfs == Double.NaN || dft >= dfs){
		    setC(1);
		}
		else{
		    setC(0);
		}
	    }
	    break;

	case MipsOperation.INST_CVT_D_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    setFPR(op.getZeroExtendedFD(), (double)(regFS.getAsFloatValue()));
	    break;

	case MipsOperation.INST_CVT_D_W:
	    regFS = getRegister(op.getZeroExtendedFS());
	    setFPR(op.getZeroExtendedFD(), (double)(regFS.getAsIntValue()));
	    break;

	case MipsOperation.INST_CVT_S_D:
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits((float)getFPRasDouble(op.getZeroExtendedFS()));
	    break;

	case MipsOperation.INST_CVT_S_W:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits((float)regFS.getAsIntValue());
	    break;

	case MipsOperation.INST_CVT_W_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits((int)(regFS.getAsFloatValue()));
	    break;

	case MipsOperation.INST_CVT_W_D:
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits((int)getFPRasDouble(op.getZeroExtendedFS()));
	    break;

	case MipsOperation.INST_DIV_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFT = getRegister(op.getZeroExtendedFT());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(regFS.getAsFloatValue() / regFT.getAsFloatValue());
	    break;

	case MipsOperation.INST_DIV_D:
	    {
		double result = getFPRasDouble(op.getZeroExtendedFS()) / getFPRasDouble(op.getZeroExtendedFT());
		setFPR(op.getZeroExtendedFD(), result);
	    }
	    break;

	case MipsOperation.INST_MOV_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(regFS.getAsFloatValue());
	    break;

	case MipsOperation.INST_MOV_D:
	    setFPR(op.getZeroExtendedFD(), getFPRasDouble(op.getZeroExtendedFS()));
	    break;

	case MipsOperation.INST_MUL_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFT = getRegister(op.getZeroExtendedFT());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(regFS.getAsFloatValue() * regFT.getAsFloatValue());
	    break;

	case MipsOperation.INST_MUL_D:
	    {
		double result = getFPRasDouble(op.getZeroExtendedFS()) * getFPRasDouble(op.getZeroExtendedFT());
		setFPR(op.getZeroExtendedFD(), result);
	    }
	    break;

	case MipsOperation.INST_NEG_S:
	    {
		regFS = getRegister(op.getZeroExtendedFS());
		regFD = getRegister(op.getZeroExtendedFD());
		int ifs = regFS.getAsIntValue();
		regFD.setAsBits(~ifs);
	    }
	    break;

	case MipsOperation.INST_NEG_D:
	    {
		long lfs = getFPRasBits(op.getZeroExtendedFS());
		setFPR(op.getZeroExtendedFD(), ~lfs);
	    }
	    break;

	case MipsOperation.INST_SUB_S:
	    regFS = getRegister(op.getZeroExtendedFS());
	    regFT = getRegister(op.getZeroExtendedFT());
	    regFD = getRegister(op.getZeroExtendedFD());
	    regFD.setAsBits(regFS.getAsFloatValue() - regFT.getAsFloatValue());
	    break;

	case MipsOperation.INST_SUB_D:
	    {
		double result = getFPRasDouble(op.getZeroExtendedFS()) - getFPRasDouble(op.getZeroExtendedFT());
		setFPR(op.getZeroExtendedFD(), result);
	    }
	    break;

	default:
	    System.out.println("unkown operation !!!!!!");
	    //throw new ExecutableElementException("unknow op code " + String.valueOf(op.getInstruction()));

	}
    }
}
