package net.wasamon.mics.processor.mips;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.ExecutableElementException;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsViewable;
import net.wasamon.mics.memory.InterruptDataPacket;
import net.wasamon.mics.memory.RandomAccessMemory;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mics.util.ChannelManager;
import net.wasamon.mics.util.MicsTableModel;
import net.wasamon.mics.util.MicsTablePanel;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

public class Mips extends MicsElement implements ExecutableElement,
		ChannelConnectable, DataBuffer, MicsViewable {

	private static final int DELAYING_STATE = 0;

	private static final int DELAYSLOT_STATE = 1;

	private static final int NORMAL_STATE = 2;

	private static final int SIGNED = 0;

	private static final int UNSIGNED = 1;

	private static final int REGISTER_SIZE = 32;

	private static final int CODE_LENGTH = 4;

	private int delayState;

	private int delayPC;

	private MipsIntRegister[] register = new MipsIntRegister[REGISTER_SIZE];

	private MipsIntRegister hiRegister, loRegister;

	private MipsProgramCounter programCounter;

	private RandomAccessMemory memory;

	private int programOffset;

	private MipsRegister readRequestRegister;

	private int readRequestDataSign;

	private ArrayList<DelayLoadSet> delayLoadSetList;

	private boolean waitFlag;

	private MipsCOP0 cop0;

	private R3010 cop1;

	private ChannelManager channelManager;

	private VirtualMemory virtualMemory;

	private boolean started;

	MipsOperation op = null;

	public Mips() {
		register[0] = new MipsZeroWiredIntRegister();
		for (int i = 1; i < REGISTER_SIZE; i++) {
			register[i] = new MipsIntRegister();
		}
		programCounter = new MipsProgramCounter();
		hiRegister = new MipsIntRegister();
		loRegister = new MipsIntRegister();
		delayLoadSetList = new ArrayList<DelayLoadSet>();

		cop0 = new MipsCOP0();
		cop1 = new R3010();

		delayState = NORMAL_STATE;
		delayPC = 0;

		started = false;
	}

	private MipsIntRegister getRegister(int index) {
		return register[index];
	}

	private int zeroExtend(byte[] data) {
		int v = 0x00000000;
		switch (data.length) {
		case 1:
			v |= (data[0] & 0x000000ff) << 0;
			break;
		case 2:
			v |= (data[0] & 0x000000ff) << 8;
			v |= (data[1] & 0x000000ff) << 0;
			break;
		case 4:
			v |= (data[0] & 0x000000ff) << 24;
			v |= (data[1] & 0x000000ff) << 16;
			v |= (data[2] & 0x000000ff) << 8;
			v |= (data[3] & 0x000000ff) << 0;
			break;
		}
		return v;
	}

	private int signExtend(byte[] data) {
		int v;
		if ((data[0] & 0x80) == 0x80) {
			v = 0xffffffff;
		} else {
			v = 0x00000000;
		}

		switch (data.length) {
		case 1:
			v &= 0xffffff00;
			v |= (data[0] & 0x000000ff) << 0;
			break;
		case 2:
			v &= 0xffff0000;
			v |= (data[0] & 0x000000ff) << 8;
			v |= (data[1] & 0x000000ff) << 0;
			break;
		case 4:
			v &= 0x00000000;
			v |= (data[0] & 0x000000ff) << 24;
			v |= (data[1] & 0x000000ff) << 16;
			v |= (data[2] & 0x000000ff) << 8;
			v |= (data[3] & 0x000000ff) << 0;
			break;
		}
		return v;
	}

	public void initialize(String base, Node n) throws MicsException {
		channelManager = new ChannelManager(composite);
		try {
			{
				Node init_var_node = n;
				int init_var_memory;
				init_var_memory = DataUtil.parseInt(XMLParser.getAttribute(
						init_var_node, "memory").getNodeValue());

				memory = new RandomAccessMemory(init_var_memory);

				{
					Node[] init_var__channel_obj = XMLParser.getNamedNodeArray(n,
							"channel");
					for (int i = 0; i < init_var__channel_obj.length; i++) {
						Node init_var__channel_node = init_var__channel_obj[i];
						String init_var__channel_id;
						init_var__channel_id = XMLParser.getAttribute(
								init_var__channel_node, "id").getNodeValue();
						int init_var__channel_offset;
						init_var__channel_offset = DataUtil.parseInt(XMLParser
								.getAttribute(init_var__channel_node, "offset").getNodeValue());

						channelManager.add(init_var__channel_id, init_var__channel_offset);

					}
				}
				{
					Node[] init_var__init_obj = XMLParser.getNamedNodeArray(n, "init");
					for (int i = 0; i < init_var__init_obj.length; i++) {
						Node init_var__init_node = init_var__init_obj[i];
						String init_var__init_file;
						init_var__init_file = XMLParser.getAttribute(init_var__init_node,
								"file").getNodeValue();
						if (init_var__init_file.charAt(0) != '/') {
							init_var__init_file = base + "/" + init_var__init_file;
						}
						int init_var__init_offset;
						init_var__init_offset = DataUtil.parseInt(XMLParser.getAttribute(
								init_var__init_node, "offset").getNodeValue());

						try {
							write(init_var__init_offset, new BufferedInputStream(
									new FileInputStream(init_var__init_file)));
						} catch (FileNotFoundException e) {
							System.out.println("no such file: " + init_var__init_file);
							System.out.println("[W] no data is written as initialize.");
						}

					}
				}

			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.Mips");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.Mips");
		}
	}

	public MicsDataPacket read(MicsDataPacket data) {
		return memory.read(data);
	}

	public void write(MicsDataPacket data) {
		memory.write(data);
	}

	public void write(int addr, InputStream reader) throws DataBufferException {
		memory.write(addr, reader);
		ELFHeader elfHeader = new ELFHeader(memory);
		if (elfHeader.isELFFormat()) {
			System.out.println("\n");
			System.out.println(elfHeader.printELFHeader());
			System.out.println(elfHeader.printProgramHeaders());
			System.out.println(elfHeader.printSectionHeaders());
			virtualMemory = new VirtualMemory(elfHeader);
			programCounter.setMemory(memory, virtualMemory);
			getRegister(29).setAsBits(
					VirtualMemory.START_OF_PROTRAM_POINT + memory.size() - 1);
			getRegister(31).setAsBits(virtualMemory.getEntryPoint());
		}
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		memory.dump(offset, len, writer);
	}

	public int size() {
		return memory.size();
	}

	public String toString(int offset, int length) {
		return memory.toString(offset, length);
	}

	public void execWait() {
		waitFlag = true;
	}

	public void execRestart() {
		waitFlag = false;
	}

	public void reset() {
		this.waitFlag = false;
		for (int i = 0; i < REGISTER_SIZE; i++) {
			register[i].reset();
		}
		programCounter.reset();
		hiRegister.reset();
		loRegister.reset();

		delayState = NORMAL_STATE;
		delayPC = 0;

		started = false;
	}

	public ExecInfo exec_second() {
		return null;
	}

	public ExecInfo exec_first() throws ExecutableElementException {
		// //////MipsOperation op = null;
		ExecInfo info = new ExecInfo();
		info.setCycle(1);
		info.setTerminatableFlag(false);

		if (programCounter.getVirtualCounter() == virtualMemory.getEntryPoint()) {
			if (started) {
				info.setTerminatableFlag(true);
				return info;
			}
			started = true;
		}

		try {
			op = readProgram(programCounter);
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new ExecutableElementException("program address error");
		} catch (MicsException e) {
			info.setTerminatableFlag(true);
			return info;
		}

		// System.out.printf("0x%08x: %s\n", programCounter.getVirtualCounter(),
		// op);

		if (waitFlag == true) {
			return info;
		}

		switch (op.getInstruction()) {
		case MipsOperation.INST_ADD: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() + reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
			// overflow exception
		}
			break;

		case MipsOperation.INST_ADDI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits(reg0.getAsIntValue() + op.getSignExtendedImmediate());
			programCounter.add(CODE_LENGTH);
			// overflow exception
		}
			break;

		case MipsOperation.INST_ADDIU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits(reg0.getAsIntValue() + op.getSignExtendedImmediate());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_ADDU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() + reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_AND: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() & reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_ANDI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits(reg0.getAsIntValue() & op.getZeroExtendedImmediate());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC0F: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == false
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC1F: {
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (cop1.getC() == 0) {
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC2F: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == false
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC3F: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == false
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC0T: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == true
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC1T: {
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (cop1.getC() == 1) {
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC2T: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == true
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BC3T: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			int offset = op.getSignExtendedImmediate() << 2;
			int delaySlotAddress = programCounter.getVirtualCounter() + CODE_LENGTH;
			if (true) {// CpCond == true
				delayPC = delaySlotAddress + offset;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BEQ: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			if (reg0.getAsIntValue() == reg1.getAsIntValue()) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BGEZ: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			if ((reg0.getAsIntValue() & 0x80000000) == 0x00000000) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BGEZAL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(31);
			reg1.setAsBits(programCounter.getVirtualCounter() + CODE_LENGTH * 2);
			if ((reg0.getAsIntValue() & 0x80000000) == 0x00000000) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BGTZ: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			// if (reg0.getAsIntValue() > 0) {//BUG
			if (((reg0.getAsIntValue() & 0x80000000) == 0x00000000)
					&& reg0.getAsIntValue() != 0) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BLEZ: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			// if (reg0.getAsIntValue() <= 0) {
			if (((reg0.getAsIntValue() & 0x80000000) == 0x80000000)
					|| reg0.getAsIntValue() == 0) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BLTZ: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			// if (reg0.getAsIntValue() < 0) {
			if ((reg0.getAsIntValue() & 0x80000000) == 0x80000000) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BLTZAL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(31);
			reg1.setAsBits(programCounter.getVirtualCounter() + CODE_LENGTH * 2);
			if ((reg0.getAsIntValue() & 0x80000000) == 0x80000000) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BNE: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			if (reg0.getAsIntValue() != reg1.getAsIntValue()) {
				int target = op.getSignExtendedImmediate() << 2;
				delayPC = programCounter.getVirtualCounter() + target + CODE_LENGTH;
				delayState = DELAYING_STATE;
			}
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_BREAK:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// 制御権を例外ハンドラに移す
			programCounter.add(CODE_LENGTH);
			info.setTerminatableFlag(true);
			return info;

		case MipsOperation.INST_CFC0: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// cop0では無効な命令
			// コプロセッサ使用不能例外
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CFC1: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg1 = cop1.getControlRegister(op.getZeroExtendedRD());
			if (reg1 == null) {
				System.out.println("unkown COP1 control register");
				info.setTerminatableFlag(true);
				return info;
			}
			reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CFC2: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			// 制御レジスタ
			// reg1 = cop2.getRegister(op.getZeroExtendedRD());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CFC3: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			// 制御レジスタ
			// reg1 = cop3.getRegister(op.getZeroExtendedRD());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		/*
		 * case MipsOperation.INST_COP0: break;
		 * 
		 * case MipsOperation.INST_COP1: break;
		 * 
		 * case MipsOperation.INST_COP2: break;
		 * 
		 * case MipsOperation.INST_COP3: break;
		 */

		case MipsOperation.INST_CTC0: {
			// cop0では無効な命令
			// コプロセッサ使用不能例外
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CTC1: {
			MipsIntRegister reg0 = cop1.getControlRegister(op.getZeroExtendedRD());
			if (reg0 == null) {
				System.out.println("unkown COP1 control register");
				info.setTerminatableFlag(true);
				return info;
			}
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CTC2: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// 制御レジスタ
			// reg0 = cop2.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_CTC3: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// 制御レジスタ
			// reg0 = cop3.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_DIV: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			hiRegister.setAsBits(reg0.getAsIntValue() % reg1.getAsIntValue());
			loRegister.setAsBits(reg0.getAsIntValue() / reg1.getAsIntValue());
			info.setCycle(35);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_DIVU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			long unsignedRS = (long) reg0.getAsIntValue() & 0x00000000ffffffffl;
			long unsignedRT = (long) reg1.getAsIntValue() & 0x00000000ffffffffl;
			hiRegister.setAsBits((int) (unsignedRS % unsignedRT));
			loRegister.setAsBits((int) (unsignedRS / unsignedRT));
			info.setCycle(35);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_J: {
			int target = op.getSignExtendedTarget() << 2;
			delayPC = ((programCounter.getVirtualCounter() + CODE_LENGTH) & 0xf0000000)
					| target;
			delayState = DELAYING_STATE;
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_JAL: {
			MipsIntRegister reg1 = getRegister(31);
			reg1.setAsBits(programCounter.getVirtualCounter() + CODE_LENGTH * 2);
			int target = op.getSignExtendedTarget() << 2;
			delayPC = ((programCounter.getVirtualCounter() + CODE_LENGTH) & 0xf0000000)
					| target;
			delayState = DELAYING_STATE;
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_JALR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRD());
			reg1.setAsBits(programCounter.getVirtualCounter() + CODE_LENGTH * 2);
			delayPC = reg0.getAsIntValue();
			delayState = DELAYING_STATE;
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_JR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			delayPC = reg0.getAsIntValue();
			delayState = DELAYING_STATE;
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LB: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			loadData(reg1, vaddr, 1, SIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LBU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			loadData(reg1, vaddr, 1, UNSIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LH: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			loadData(reg1, vaddr, 2, SIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LHU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			loadData(reg1, vaddr, 2, UNSIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LUI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			reg0.setAsBits(op.getZeroExtendedImmediate() << 16);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LW: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			loadData(reg1, vaddr, 4, SIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LWC0:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_LWC1: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsFloatRegister freg = cop1.getRegister(op.getZeroExtendedRT());
			int vaddr = op.getSignExtendedImmediate() + reg0.getAsIntValue();
			loadData(freg, vaddr, 4, SIGNED);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_LWC2:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_LWC3:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_LWL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			int paddr = virtualMemory.getPhysicalAddress(vaddr);
			if (paddr == 0xffffffff) {
				// 当座
				// address error exception or TLB exception
				System.out.println("address error exception or TLB exception");
				info.setTerminatableFlag(true);
				return info;
			}
			int value = reg1.getAsIntValue();

			switch (paddr & 0x00000003) {
			case 0:
				reg1.setAsBits((value & 0x00000000)
						| (zeroExtend(memory.read(paddr, 4)) << 0));
				break;
			case 1:
				reg1.setAsBits((value & 0x000000ff)
						| (zeroExtend(memory.read(paddr, 3)) << 8));
				break;
			case 2:
				reg1.setAsBits((value & 0x0000ffff)
						| (zeroExtend(memory.read(paddr, 2)) << 16));
				break;
			case 3:
				reg1.setAsBits((value & 0x00ffffff)
						| (zeroExtend(memory.read(paddr, 1)) << 24));
				break;
			}

			programCounter.add(CODE_LENGTH);
			break;
		}

		case MipsOperation.INST_LWR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			int paddr = virtualMemory.getPhysicalAddress(vaddr);
			if (paddr == 0xffffffff) {
				// 当座
				// address error exception or TLB exception
				System.out.println("address error exception or TLB exception");
				info.setTerminatableFlag(true);
				return info;
			}
			int value = reg1.getAsIntValue();

			switch (paddr & 0x00000003) {
			case 0:
				reg1
						.setAsBits((value & 0xffffff00) | zeroExtend(memory.read(paddr, 1)));
				break;
			case 1:
				reg1.setAsBits((value & 0xffff0000)
						| zeroExtend(memory.read(paddr - 1, 2)));
				break;
			case 2:
				reg1.setAsBits((value & 0xff000000)
						| zeroExtend(memory.read(paddr - 2, 3)));
				break;
			case 3:
				reg1.setAsBits((value & 0x00000000)
						| zeroExtend(memory.read(paddr - 3, 4)));
				break;
			}

			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFC0: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			// reg1 = cop3.getRegister(op.getZeroExtendedRD());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFC1: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			MipsFloatRegister freg = cop1.getRegister(op.getZeroExtendedRD());
			// reg1 = cop1.getRegister(op.getZeroExtendedRD());
			reg0.setAsBits(freg.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFC2: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			// reg1 = cop3.getRegister(op.getZeroExtendedRD());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFC3: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			// reg1 = cop3.getRegister(op.getZeroExtendedRD());
			// reg0.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFHI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRD());
			reg0.setAsBits(hiRegister.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MFLO: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRD());
			reg0.setAsBits(loRegister.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTC0: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// reg0 = cop0.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			// reg1.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTC1: {
			// reg0 = cop1.getRegister(op.getZeroExtendedRD());
			MipsFloatRegister freg = cop1.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			delayLoadSetList.add(new DelayLoadSet(freg, reg1.getAsFloatValue()));
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTC2: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// reg0 = cop2.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			// reg1.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTC3: {
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			// reg0 = cop3.getRegister(op.getZeroExtendedRD());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			// reg1.setAsBits(reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTHI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRD());
			hiRegister.setAsBits(reg0.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MTLO: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRD());
			loRegister.setAsBits(reg0.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MULT: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			long signedRS = reg0.getAsIntValue();
			long signedRT = reg1.getAsIntValue();
			long ans = signedRS * signedRT;
			hiRegister.setAsBits((int) ((ans & 0xffffffff00000000l) >>> 32));
			loRegister.setAsBits((int) ((ans & 0x00000000ffffffffl)));
			info.setCycle(12);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_MULTU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			long unsignedRS = (long) reg0.getAsIntValue() & 0x00000000ffffffffl;
			long unsignedRT = (long) reg1.getAsIntValue() & 0x00000000ffffffffl;
			long ans = unsignedRS * unsignedRT;
			hiRegister.setAsBits((int) ((ans & 0xffffffff00000000l) >>> 32));
			loRegister.setAsBits((int) ((ans & 0x00000000ffffffff)));
			info.setCycle(12);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_NOR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(~(reg0.getAsIntValue() | reg1.getAsIntValue()));
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_OR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() | reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_ORI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits(reg0.getAsIntValue() | op.getZeroExtendedImmediate());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_RFE:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_SB: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			byte data[] = new byte[1];
			data[0] = (byte) (reg1.getAsIntValue() & 0x000000ff);
			// int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate() + 3;
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			storeData(data, vaddr, data.length);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SH: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			byte data[] = new byte[2];
			data[0] = (byte) ((reg1.getAsIntValue() & 0x0000ff00) >>> 8);
			data[1] = (byte) ((reg1.getAsIntValue() & 0x000000ff) >>> 0);
			// int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate() + 2;
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			storeData(data, vaddr, data.length);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRD());
			reg1.setAsBits(reg0.getAsIntValue() << op.getZeroExtendedShamt());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLLV: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2
					.setAsBits(reg1.getAsIntValue() << (reg0.getAsIntValue() & 0x0000001F));
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLT: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits((reg0.getAsIntValue() < reg1.getAsIntValue()) ? 1 : 0);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLTI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits((reg0.getAsIntValue() < op.getSignExtendedImmediate()) ? 1
					: 0);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLTIU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			long unsignedRS = (long) reg0.getAsIntValue() & 0x00000000ffffffffl;
			reg1.setAsBits((unsignedRS < op.getSignExtendedImmediate()) ? 1 : 0);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SLTU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			long unsignedRS = (long) reg0.getAsIntValue() & 0x00000000ffffffffl;
			long unsignedRT = (long) reg1.getAsIntValue() & 0x00000000ffffffffl;
			reg2.setAsBits((unsignedRS < unsignedRT) ? 1 : 0);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SRA: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRD());
			reg1.setAsBits(reg0.getAsIntValue() >> op.getZeroExtendedShamt());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SRAV: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2
					.setAsBits(reg1.getAsIntValue() >> (reg0.getAsIntValue() & 0x0000001F));
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SRL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRD());
			reg1.setAsBits(reg0.getAsIntValue() >>> op.getZeroExtendedShamt());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SRLV: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2
					.setAsBits(reg1.getAsIntValue() >>> (reg0.getAsIntValue() & 0x0000001F));
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SUB: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() - reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SUBU: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() - reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SW: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			byte data[] = new byte[4];
			data[0] = (byte) ((reg1.getAsIntValue() & 0xff000000) >>> 24);
			data[1] = (byte) ((reg1.getAsIntValue() & 0x00ff0000) >>> 16);
			data[2] = (byte) ((reg1.getAsIntValue() & 0x0000ff00) >>> 8);
			data[3] = (byte) ((reg1.getAsIntValue() & 0x000000ff) >>> 0);
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			storeData(data, vaddr, data.length);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SWC0:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_SWC1: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			// reg1 = cop1.getRegister(op.getZeroExtendedRT());
			MipsFloatRegister freg = cop1.getRegister(op.getZeroExtendedRT());
			byte data[] = new byte[4];
			int value = freg.getAsIntValue();
			data[0] = (byte) ((value & 0xff000000) >>> 24);
			data[1] = (byte) ((value & 0x00ff0000) >>> 16);
			data[2] = (byte) ((value & 0x0000ff00) >>> 8);
			data[3] = (byte) ((value & 0x000000ff) >>> 0);
			int vaddr = op.getSignExtendedImmediate() + reg0.getAsIntValue();
			storeData(data, vaddr, data.length);
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SWC2:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_SWC3:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_SWL: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			int value = reg1.getAsIntValue();

			byte data[] = new byte[4 - (vaddr & 0x00000003)];
			switch (vaddr & 0x00000003) {
			case 0:
				data[0] = (byte) ((value & 0xff000000) >>> 24);
				data[1] = (byte) ((value & 0x00ff0000) >>> 16);
				data[2] = (byte) ((value & 0x0000ff00) >>> 8);
				data[3] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 1:
				data[0] = (byte) ((value & 0x00ff0000) >>> 16);
				data[1] = (byte) ((value & 0x0000ff00) >>> 8);
				data[2] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 2:
				data[0] = (byte) ((value & 0x0000ff00) >>> 8);
				data[1] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 3:
				data[0] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			}

			programCounter.add(CODE_LENGTH);
			break;
		}

		case MipsOperation.INST_SWR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			int vaddr = reg0.getAsIntValue() + op.getSignExtendedImmediate();
			int value = reg1.getAsIntValue();

			byte data[] = new byte[1 + (vaddr & 0x00000003)];
			switch (vaddr & 0x00000003) {
			case 0:
				data[0] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 1:
				data[0] = (byte) ((value & 0x0000ff00) >>> 8);
				data[1] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 2:
				data[0] = (byte) ((value & 0x00ff0000) >>> 16);
				data[1] = (byte) ((value & 0x0000ff00) >>> 8);
				data[2] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			case 3:
				data[0] = (byte) ((value & 0xff000000) >>> 24);
				data[1] = (byte) ((value & 0x00ff0000) >>> 16);
				data[2] = (byte) ((value & 0x0000ff00) >>> 8);
				data[3] = (byte) ((value & 0x000000ff) >>> 0);
				storeData(data, vaddr, data.length);
				break;
			}

			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_SYSCALL:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_TLBP:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_TLBR:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_TLBWI:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_TLBWR:
			System.out.printf("incomplete instruction\n0x%08x:  %s\n", programCounter
					.getVirtualCounter());
			programCounter.add(CODE_LENGTH);
			break;

		case MipsOperation.INST_XOR: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			MipsIntRegister reg2 = getRegister(op.getZeroExtendedRD());
			reg2.setAsBits(reg0.getAsIntValue() ^ reg1.getAsIntValue());
			programCounter.add(CODE_LENGTH);
		}
			break;

		case MipsOperation.INST_XORI: {
			MipsIntRegister reg0 = getRegister(op.getZeroExtendedRS());
			MipsIntRegister reg1 = getRegister(op.getZeroExtendedRT());
			reg1.setAsBits(reg0.getAsIntValue() ^ op.getZeroExtendedImmediate());
			programCounter.add(CODE_LENGTH);
		}
			break;

		default:
			cop1.exec(op);
			info.setCycle(2);
			programCounter.add(CODE_LENGTH);

		}

		switch (delayState) {
		case NORMAL_STATE:
			break;

		case DELAYING_STATE:
			delayState = DELAYSLOT_STATE;
			break;

		case DELAYSLOT_STATE:
			delayState = NORMAL_STATE;
			programCounter.setVirtualCounter(delayPC);
			break;
		}

		delayLoad();

		return info;
	}

	private boolean storeData(byte[] data, int vaddr, int length)
			throws ExecutableElementException {

		int paddr = virtualMemory.getPhysicalAddress(vaddr);

		if (paddr == 0xffffffff) {// 当座
			System.out.println("address error in storeData(data[], vaddr, length)");
			System.out
					.printf("0x%08x:  %s\n", programCounter.getVirtualCounter(), op);
			System.out.printf("vaddr = 0x%08x", vaddr);
			return false;
		}

		if (paddr < memory.size()) {
			memory.write(paddr, length, data);
		} else {
			/*
			 * System.out.println("-----------------------------------");
			 * System.out.printf("0x%08x: %s\n", programCounter.getVirtualCounter(),
			 * op); System.out.println(String.format("offset = 0x%08x", paddr -
			 * 0x07000000)); System.out.println(String.format("length = %d", length));
			 * System.out.println(String.format("data as bits = 0x%08x",
			 * zeroExtend(data))); System.out.println(String.format("data as char =
			 * %c", (char)zeroExtend(data))); System.out.println(String.format("data
			 * as int = %d", zeroExtend(data)));
			 * System.out.println(String.format("data as float = %f",
			 * Float.intBitsToFloat(zeroExtend(data))));
			 * System.out.println("-----------------------------------");
			 */
			ChannelManager.Element c = channelManager.search(paddr);
			try {
				c.getChannel().writeRequest(
						this,
						RandomAccessMemoryDataPacket.writePacket(paddr - c.offset, length,
								8, data));
			} catch (MicsException e) {
				throw new ExecutableElementException(e);
			}
		}

		return true;
	}

	private boolean loadData(MipsRegister dest, int vaddr, int length, int sign)
			throws ExecutableElementException {
		int paddr = virtualMemory.getPhysicalAddress(vaddr);

		if (paddr == 0xffffffff) {// 当座
			System.out
					.println("address error in loadData(dest, vaddr, length, sign)");
			System.out.printf("vaddr = 0x%08x", vaddr);
			return false;
		}

		if (paddr < memory.size()) {
			delayLoadSetList.add(new DelayLoadSet(dest, paddr, length, sign));
		} else {
			readRequestRegister = dest;
			readRequestDataSign = sign;
			ChannelManager.Element c = channelManager.search(paddr);
			try {
				c.getChannel().readRequest(
						this,
						RandomAccessMemoryDataPacket
								.readPacket(paddr - c.offset, length, 8));
			} catch (MicsException e) {
				throw new ExecutableElementException(e);
			}
		}

		return true;
	}

	private void loadData(DelayLoadSet dls) {
		switch (dls.sign) {
		case SIGNED:
			dls.dest.setAsBits(signExtend(memory.read(dls.paddr, dls.length)));
			break;

		case UNSIGNED:
			dls.dest.setAsBits(zeroExtend(memory.read(dls.paddr, dls.length)));
			break;
		}
	}

	private MipsOperation readProgram(MipsProgramCounter pc) throws MicsException {
		int paddr = pc.getPhysicalCounter();
		if (paddr == 0xffffffff) {
			String svpc = String.format("0x%08x", pc.getVirtualCounter());
			String sppc = String.format("0x%08x", paddr);
			throw new MicsException("pc address error:vpc = " + svpc + " ppc = "
					+ sppc);
		}
		byte[] data = memory.read(paddr + programOffset, 4);
		int val = ((data[0] << 24) & 0xff000000) | ((data[1] << 16) & 0x00ff0000)
				| ((data[2] << 8) & 0x0000ff00) | (data[3] & 0x000000ff);
		return new MipsOperation(val);
	}

	public void writeback(Channel src, MicsDataPacket data) {
		if (data instanceof RandomAccessMemoryDataPacket) {
			RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
			switch (readRequestDataSign) {
			case SIGNED:
				readRequestRegister.setAsBits(signExtend(rd.data));
				break;

			case UNSIGNED:
				readRequestRegister.setAsBits(zeroExtend(rd.data));
				break;
			}
		} else if (data instanceof InterruptDataPacket) {
			InterruptDataPacket d = (InterruptDataPacket) data;
			// intRegister[d.addr].setAsBits(d.data);
			try {
				storeData(d.data, 0x0a000002, d.data.length);
			} catch (ExecutableElementException e) {

			}
		}
	}

	private void delayLoad() {
		if (delayLoadSetList.size() == 0) {
			return;
		}

		for (int i = delayLoadSetList.size() - 1; i >= 0; --i) {
			delayLoadSetList.get(i).counter += 1;
		}

		DelayLoadSet set = delayLoadSetList.get(0);
		if (set.counter == 2) {
			if (set.type == 0) {
				loadData(set);
			} else if (set.type == 1) {
				set.dest.setAsBits(set.datai);
			} else {
				set.dest.setAsBits(set.dataf);
			}
			delayLoadSetList.remove(0);
		} else if (delayLoadSetList.get(0).counter > 2) {
			System.out.println("Delay Load Error !!!!!!!");
		}

	}

	public int getChannelOffset(Channel c) {
		return channelManager.search(c);
	}

	public String getInfo() {
		String s = "";
		s += "Mips\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  ID: " + id() + "\n";
		s += "  Local Memory ID: " + ((MicsElement) memory).id() + ",";
		s += " offset = "
				+ DataUtil.toBigEndianValueString(DataUtil.toByteArray(programOffset))
				+ "\n";
		ChannelManager.Element[] channels = channelManager.array();
		for (int i = 0; i < channels.length; i++) {
			ChannelManager.Element element = channels[i];
			s += "  Channel ID: " + element.id + ",";
			s += " offset = " + element.offset + "\n";
		}
		return s;
	}

	public String toString() {
		try {
			String str = "\n";
			try {
				str += programCounter + ":";
			} catch (FormatException e) {
				e.printStackTrace();
			}
			str += readProgram(programCounter).toString() + "\n";
			str += "=========================== R3000 Register ============================\n";
			for (int i = 0; i < REGISTER_SIZE; i++) {
				try {
					str += register[i];
					if (i % 8 == 7) {
						str += "\n";
					} else {
						str += " ";
					}
				} catch (FormatException e) {

				}
			}
			str += hiRegister + " " + loRegister + "\n";

			str += "=========================== R3010 Register ============================\n";
			for (int i = 0; i < REGISTER_SIZE; i++) {
				try {
					str += cop1.getRegister(i);
					if (i % 8 == 7) {
						str += "\n";
					} else {
						str += " ";
					}
				} catch (FormatException e) {

				}
			}
			str += cop1.getControlRegister(0) + " " + cop1.getControlRegister(31);

			return str;
		} catch (ArrayIndexOutOfBoundsException e) {
			return new String(programCounter.getVirtualCounter()
					+ ": illegal program counter");
		} catch (MicsException e) {
			return new String(programCounter.getVirtualCounter()
					+ ": illegal program counter");
		}
	}

	public String[] getConnectedElements() {
		return channelManager.getConnectedElements();
	}

	MicsTablePanel table;

	public void show() {
		if (table == null) {
			MipsRegister[] sregister = new MipsRegister[REGISTER_SIZE + 3];
			MipsRegister[] sregisterCOP1 = new MipsRegister[REGISTER_SIZE + 3];
			String[] label = { "$zero", "$at", "$v0", "$v1", "$a0", "$a1", "$a2",
					"$a3", "$t0", "$t1", "$t2", "$t3", "$t4", "$t5", "$t6", "$t7", "$s0",
					"$s1", "$s2", "$s3", "$s4", "$s5", "$s6", "$s7", "$t8", "$t9", "$k0",
					"$k1", "$gp", "$sp", "$fp", "$ra", "$hi", "$lo", "$pc" };
			String[] labelCOP1 = new String[label.length];

			for (int i = 0; i < REGISTER_SIZE; ++i) {
				label[i] = String.format("$%d(", i) + label[i] + ")";
				sregister[i] = register[i];
			}

			for (int i = 0; i < labelCOP1.length - 3; ++i) {
				labelCOP1[i] = String.format("$f%d", i);
				sregisterCOP1[i] = cop1.getRegister(i);
			}
			labelCOP1[32] = "fcr0";
			labelCOP1[33] = "fcr31";
			labelCOP1[34] = "";
			sregisterCOP1[32] = cop1.getControlRegister(0);
			sregisterCOP1[33] = cop1.getControlRegister(31);
			sregisterCOP1[34] = null;

			sregister[REGISTER_SIZE] = hiRegister;
			sregister[REGISTER_SIZE + 1] = loRegister;
			sregister[REGISTER_SIZE + 2] = programCounter;

			table = new MicsTablePanel("Mips Register",
					new MicsTableModel(new String[] { "R3000 Register", "value",
							"R3010 Register", "value" }, new Object[][] { label, sregister,
							labelCOP1, sregisterCOP1 }));
		}

		table.setVisible(true);
	}

	public String getImagePath() {
		return "combo_processor.png";
	}

	class DelayLoadSet {

		MipsRegister dest;
		int paddr, length, sign, counter, type;
		int datai;
		float dataf;

		DelayLoadSet(MipsRegister dest, int paddr, int length, int sign) {
			this.dest = dest;
			this.paddr = paddr;
			this.length = length;
			this.sign = sign;
			this.counter = 0;
		}

		DelayLoadSet(MipsRegister dest, int datai) {
			this.dest = dest;
			this.datai = datai;
			this.type = 1;
		}

		DelayLoadSet(MipsRegister dest, float dataf) {
			this.dest = dest;
			this.dataf = dataf;
			this.type = 2;
		}

	}

	public String getDescription(){
		return "TODO";
	}

}
