package net.wasamon.mics.processor.mips;

import net.wasamon.mics.memory.RandomAccessMemory;

public class ELFHeader {

    //ELFヘッダ定数

    private static final int ELFMAG0 = 0x7f;

    private static final int ELFMAG1 = 0x45;

    private static final int ELFMAG2 = 0x4c;

    private static final int ELFMAG3 = 0x46;

    private static final int ELFMAG = 0x7f454c46;

    private static final int ELFCLASS32 = 1;

    private static final int ELFDATA2MSB = 2;

    private static final int ET_EXEC = 2;

    private static final int EV_CURRENT = 1;

    private static final int EM_NONE = 0;

    private static final int EM_M32 = 1;

    private static final int EM_SPARC = 2;

    private static final int EM_386 = 3;

    private static final int EM_68K = 4;

    private static final int EM_88K = 5;

    private static final int EM_486 = 6;

    private static final int EM_860 = 7;

    private static final int EM_MIPS = 8;

    private static final int EM_MIPS_RS4_BE = 10;

    private static final int EM_SPARC64 = 11;

    private static final int EM_PARISC = 15;

    private static final int EM_SPARC32PLUS = 18;

    private static final int EM_PPC = 20;

    private static final int EM_ARM = 40;

    private static final int EM_ALPHA = 41;

    private static final int EM_SPARCV9 = 43;

    private static final int EM_ALPHA_EXP = 0x9026;

    private static final int EM_AMD64 = 62;

    private static final int EM_VAX = 75;

    private static final String eht_string_class[] = {
	"invalid", "ELF32", "ELF64"
    };

    private static final String eht_string_data[] = {
	"invalid", "Little-Endian", "Big-Endian"
    };

    private static final String eht_string_version[] = {
	"none", "current"
    };

    private static final String eht_string_osabi[] = {
	"UNIX System V ABI", "HP-UX operating system", "NetBSD", "GNU/Linux", "GNU/Hurd", 
	"86Open common IA32 ABI", "Solaris", "Monterey", "IRIX", "FreeBSD", 
	"TRU64 UNIX", "Novell Modesto", "OpenBSD", "ARM", "Standalone (embedded) application"
    };

    private static final String eht_string_type[] = {
	"NONE", "REL", "EXEC", "DYN", "CORE", "NUM"
    };

    //プログラムヘッダ定数

    private static final String pht_string[] = {
	"NULL", "LOAD", "DYNAMIC", "INTERP", "NOTE", 
	"SHLIB", "PHDR", "NUM", "REGINFO", "RTPROC",
	"OPTIONS"
    };

    //セクションヘッダ定数

    private static final String sht_string[] = {
	"NULL", "PROGBITS", "SYMTAB", "STRTAB", "RELA",
	"HASH", "DYNAMIC", "NOTE", "NOBITS", "REL",
	"SHLIB", "DYNSYM", "NUM"
    };

    private static final String sht_string_mips[] = {
	"MIPS_LIBLIST", "MIPS_MSYM", "MIPS_CONFLICT", "MIPS_GPTAB", "MIPS_UCODE", 
	"MIPS_DEBUG", "MIPS_REGINFO", "MIPS_PACKAGE", "MIPS_PACKSYM", "MIPS_RELD", 
	"0x7000000a", "MIPS_IFACE", "MIPS_CONTENT", "MIPS_OPTIONS", "0x7000000e", "0x7000000f", 
	"MIPS_SHDR", "MIPS_FDESC", "MIPS_EXTSYM", "MIPS_DENSE", "MIPS_PDESC", 
	"MIPS_LOCSYM", "MIPS_AUXSYM", "MIPS_OPTSYM", "MIPS_LOCSTR", "MIPS_LINE", 
	"MIPS_RFDESC", "MIPS_DELTASYM", "MIPS_DELTAINST", "MIPS_DELTACLASS", "MIPS_DWARF", 
	"MIPS_DELTADECL", "MIPS_SYMBOL_LIB", "MIPS_EVENTS", "MIPS_TRANSLATE", "MIPS_PIXIE", 
	"MIPS_XLATE", "MIPS_XLATE_DEBUG", "MIPS_WHIRL", "MIPS_EH_REGION", "MIPS_XLATE_OLD", 
	"MIPS_PDR_EXCEPTION"
    };

    //ELFヘッダ変数

    private byte[] e_ident;   //ELF Identification

    private int e_type;       //オブジェクトファイルタイプ

    private int e_machine;    //マシンタイプ

    private int e_version;    //オブジェクトファイルバージョン

    private int e_entry;      //エントリポイントアドレス

    private int e_phoff;      //プログラムヘッダ始点

    private int e_shoff;      //セクションヘッダ始点

    private int e_flags;      //プロセッサフラグ

    private int e_ehsize;     //ELFヘッダサイズ

    private int e_phentsize;  //プログラムヘッダサイズ

    private int e_phnum;      //プログラムヘッダ数

    private int e_shentsize;  //セクションヘッダサイズ

    private int e_shnum;      //セクションヘッダ数

    private int e_shstrndx;   //セクション名のストリングテーブルエントリ始点


    //プログラムヘッダ変数

    private int[] p_type;       //セグメントタイプ

    private int[] p_offset;     //セグメントオフセット

    private int[] p_vaddr;      //セグメントの仮想アドレス

    private int[] p_paddr;      //セグメントの物理アドレス(無効？)

    private int[] p_filesz;     //ファイル内のサイズ(バイト)

    private int[] p_memsz;      //メモリ内のサイズ(バイト)

    private int[] p_flags;      //フラグ？

    private int[] p_align;      //メモリアライメント


    //セクションヘッダ変数

    private int[] sh_name;      //name - index into section header string table section

    private int[] sh_type;      //type

    private int[] sh_flags;     //flags

    private int[] sh_addr;      //address

    private int[] sh_offset;    //file offset

    private int[] sh_size;      //section size

    private int[] sh_link;      //section header table index link

    private int[] sh_info;      //extra information

    private int[] sh_addralign; //address alignment

    private int[] sh_entsize;   //section entry size


    private RandomAccessMemory memory;

    public ELFHeader(RandomAccessMemory memory){
	this.memory = memory;
	analyzeHeaders();
    }

    public boolean isELFFormat(){
	if(memory.read(0, 1)[0] == ELFMAG0 && memory.read(1, 1)[0] == ELFMAG1 && 
	   memory.read(2, 1)[0] == ELFMAG2 && memory.read(3, 1)[0] == ELFMAG3){
	    return true;
	}

	return false;
    }

    public boolean analyzeHeaders(){

	if(isELFFormat() == false){
	    return false;
	}

	analyzeELFHeader();
	analyzeProgramHeaders();
	analyzeSectionHeaders();

	return true;
    }

    public void analyzeELFHeader(){
	e_ident = memory.read(0, 16);
	e_type = zeroExtend(memory.read(16, 2));
	e_machine = zeroExtend(memory.read(18, 2));
	e_version = zeroExtend(memory.read(20, 4));
	e_entry = zeroExtend(memory.read(24, 4));
	e_phoff = zeroExtend(memory.read(28, 4));
	e_shoff = zeroExtend(memory.read(32, 4));
	e_flags = zeroExtend(memory.read(36, 4));
	e_ehsize = zeroExtend(memory.read(40, 2));
	e_phentsize = zeroExtend(memory.read(42, 2));
	e_phnum = zeroExtend(memory.read(44, 2));
	e_shentsize = zeroExtend(memory.read(46, 2));
	e_shnum = zeroExtend(memory.read(48, 2));
	e_shstrndx = zeroExtend(memory.read(50, 2));
    }

    public void analyzeProgramHeaders(){
	p_type = new int[e_phnum];
	p_offset = new int[e_phnum];
	p_vaddr = new int[e_phnum];
	p_paddr = new int[e_phnum];
	p_filesz = new int[e_phnum];
	p_memsz = new int[e_phnum];
	p_flags = new int[e_phnum];
	p_align = new int[e_phnum];

	int offset = e_phoff;
	for(int index = 0; index < e_phnum; ++index){
	    analyzeProgramHeader(index, offset);
	    offset += e_phentsize;
	}
    }

    public void analyzeProgramHeader(int index, int offset){
	p_type[index] = zeroExtend(memory.read(offset, 4));
	p_offset[index] = zeroExtend(memory.read(offset + 4, 4));
	p_vaddr[index] = zeroExtend(memory.read(offset + 8, 4));
	p_paddr[index] = zeroExtend(memory.read(offset + 12, 4));
	p_filesz[index] = zeroExtend(memory.read(offset + 16, 4));
	p_memsz[index] = zeroExtend(memory.read(offset + 20, 4));
	p_flags[index] = zeroExtend(memory.read(offset + 24, 4));
	p_align[index] = zeroExtend(memory.read(offset + 28, 4));
    }

    public void analyzeSectionHeaders(){
	sh_name = new int[e_shnum];
	sh_type = new int[e_shnum];
	sh_flags = new int[e_shnum];
	sh_addr = new int[e_shnum];
	sh_offset = new int[e_shnum];
	sh_size = new int[e_shnum];
	sh_link = new int[e_shnum];
	sh_info = new int[e_shnum];
	sh_addralign = new int[e_shnum];
	sh_entsize = new int[e_shnum];

	int offset = e_shoff;
	for(int index = 0; index < e_shnum; ++index){
	    analyzeSectionHeader(index, offset);
	    offset += e_shentsize;
	}
    }

    public void analyzeSectionHeader(int index, int offset){
	sh_name[index] = zeroExtend(memory.read(offset, 4));
	sh_type[index] = zeroExtend(memory.read(offset + 4, 4));
	sh_flags[index] = zeroExtend(memory.read(offset + 8, 4));
	sh_addr[index] = zeroExtend(memory.read(offset + 12, 4));
	sh_offset[index] = zeroExtend(memory.read(offset + 16, 4));
	sh_size[index] = zeroExtend(memory.read(offset + 20, 4));
	sh_link[index] = zeroExtend(memory.read(offset + 24, 4));
	sh_info[index] = zeroExtend(memory.read(offset + 28, 4));
	sh_addralign[index] = zeroExtend(memory.read(offset + 32, 4));
	sh_entsize[index] = zeroExtend(memory.read(offset + 36, 4));
    }

    public  String printELFHeader(){
	String value = "ELF Header:\n  magic:                             ";


	String cls;
	if(0 <= (int)e_ident[4] && (int)e_ident[4] <= 2){
	    cls = eht_string_class[(int)e_ident[4]];
	}
	else{
	    cls = String.valueOf((int)e_ident[4]);
	}

	String data;
	if(0 <= (int)e_ident[5] && (int)e_ident[5] <= 2){
	    data = eht_string_data[(int)e_ident[5]];
	}
	else{
	    data = String.valueOf((int)e_ident[5]);
	}

	String osabi;
	if(0 <= (int)e_ident[7] && (int)e_ident[7] <= 12){
	    osabi = eht_string_osabi[(int)e_ident[7]];
	}
	else if((int)e_ident[7] == 97){
	    osabi = eht_string_osabi[13];
	}
	else if((int)e_ident[7] == 255){
	    osabi = eht_string_osabi[14];
	}
	else{
	    osabi = String.valueOf((int)e_ident[7]);
	}

	String version = String.valueOf((int)e_ident[6]);
	if(0 <= (int)e_ident[6] && (int)e_ident[6] <= 1){
	    version += " (" + eht_string_version[(int)e_ident[6]] + ")";
	}

	String type;
	if(0 <= e_type && e_type <= 5){
	    type = eht_string_type[e_type];
	}
	else{
	    type = String.valueOf(e_type);
	}

	String machine;
	switch(e_machine){
	case EM_NONE:
	    machine = "No Machine";
	    break;

	case EM_M32:
	    machine = "AT&T WE 32100";
	    break;

	case EM_SPARC:
	    machine = "SPARC";
	    break;

	case EM_386:
	    machine = "Intel 80386";
	    break;

	case EM_68K:
	    machine = "Motorola 68000";
	    break;

	case EM_88K:
	    machine = "Motorola 88000";
	    break;

	case EM_486:
	    machine = "Intel 80486";
	    break;

	case EM_860:
	    machine = "Intel 80860";
	    break;

	case EM_MIPS:
	    machine = "MIPS R3000 Big-Endian only";
	    break;

	case EM_MIPS_RS4_BE:
	    machine = "MIPS R4000 Big-Endian";
	    break;

	case EM_SPARC64:
	    machine = "SPARC v9 64-bit unoffical";
	    break;

	case EM_PARISC:
	    machine = "HPPA";
	    break;

	case EM_SPARC32PLUS:
	    machine = "Enhanced instruction set SPARCEnhanced instruction set SPARC";
	    break;

	case EM_PPC:
	    machine = "PowerPC";
	    break;

	case EM_ARM:
	    machine = "Advanced RISC Machines ARM";
	    break;

	case EM_ALPHA:
	    machine = "DEC ALPHA";
	    break;

	case EM_SPARCV9:
	    machine = "SPARC version 9";
	    break;

	case EM_ALPHA_EXP:
	    machine = "DEC ALPHADEC ALPHA";
	    break;

	case EM_AMD64:
	    machine = "AMD64 architecture";
	    break;

	case EM_VAX:
	    machine = "DEC VAX";
	    break;

	default:
	    machine = String.valueOf(e_machine);
	}

	for(int i = 0; i < 4; ++i){
	    value += String.valueOf((char)e_ident[i]);
	}
	value += "\n";

	value += "  Class:                             " + cls + "\n";
	value += "  Data:                              " + data + "\n";
	value += "  Version:                           " + version + "\n";
	value += "  OS/ABI:                            " + osabi + "\n";
	value += "  ABI Version:                       " + String.valueOf((int)e_ident[8]) + "\n";
	value += "  Type:                              " + type + "\n";
	value += "  Machine:                           " + machine + "\n";
	value += "  Version:                           " + String.format("0x%x\n", e_version);
	value += "  Entry point address:               " + String.format("0x%08x\n", e_entry);
	value += "  Start of program headers:          " + String.format("0x%08x\n", e_phoff);
	value += "  Start of section headers:          " + String.format("0x%08x\n", e_shoff);
	value += "  Flags:                             " + String.format("0x%x\n", e_flags);
	value += "  Size of this header:               " + String.valueOf(e_ehsize) + " (bytes)\n";
	value += "  Size of program headers:           " + String.valueOf(e_phentsize) + " (bytes)\n";
	value += "  Number of program headers:         " + String.valueOf(e_phnum) + "\n";
	value += "  Size of section headers:           " + String.valueOf(e_shentsize) + " (bytes)\n";
	value += "  Number of sectioin headers:        " + String.valueOf(e_shnum) + "\n";
	value += "  Section header string table index: " + String.valueOf(e_shstrndx) + "\n";

	return value + "\n";
    }

    public  String printProgramHeaders(){
	String value = "Program Headers:\n  Type           Offset   VirtAddr   PhysAddr   FileSiz MemSiz  Flg Align\n";
	for(int index = 0; index < e_phnum; ++index){
	    value += printProgramHeader(index);
	}

	return value + "\n";
    }

    public  String printProgramHeader(int index){
	String value = "  ";
	String flg = "";

	flg += ((p_flags[index] & 0x00000004) == 0x00000004) ? "R" : " ";
	flg += ((p_flags[index] & 0x00000002) == 0x00000002) ? "W" : " ";
	flg += ((p_flags[index] & 0x00000001) == 0x00000001) ? "E" : " ";

	String type;
	if(0 <= p_type[index] && p_type[index] <= 7){
	    type = pht_string[p_type[index]];
	}
	else if(0x70000000 <= p_type[index] && p_type[index] <= 0x70000002){
	    type = pht_string[p_type[index] - 0x70000000 + 8];
	}
	else{
	    type = String.format("0x%08x", p_type[index]);
	}
	for(int i = 15 - type.length(); i > 0 ; --i){
	    type += " ";
	}

	value += type;
	value += String.format("0x%06x ", p_offset[index]);
	value += String.format("0x%08x ", p_vaddr[index]);
	value += String.format("0x%08x ", p_paddr[index]);
	value += String.format("0x%05x ", p_filesz[index]);
	value += String.format("0x%05x ", p_memsz[index]);
	value += flg + " ";
	value += String.format("0x%x", p_align[index]);

	return value + "\n";
    }

    public String printSectionHeaders(){
	String value = "Section Headers:\n  [Nr] Name              Type            Addr     Off    Size   ES Flg Lk Inf Al\n";
	for(int index = 0; index < e_shnum; ++index){
	    value += printSectionHeader(index);
	}

	value += "Key to Flags:\n";
	value += "  W (write), A (alloc), X (execute), M (merge), S (strings)\n";
	value += "  I (info), L (link order), G (group), x (unknown)\n";
	value += "  O (extra OS processing required) o (OS specific), p (processor specific)\n";

	return value + "\n";
    }

    public String printSectionHeader(int index){

	String flg = "";
	flg += ((sh_flags[index] & 0x00000001) == 0x00000001) ? "W" : "";
	flg += ((sh_flags[index] & 0x00000002) == 0x00000002) ? "A" : "";
	flg += ((sh_flags[index] & 0x00000004) == 0x00000004) ? "X" : "";
	flg += ((sh_flags[index] & 0x00000008) == 0x00000008) ? "x" : "";
	flg += ((sh_flags[index] & 0x00000010) == 0x00000010) ? "M" : "";
	flg += ((sh_flags[index] & 0x00000020) == 0x00000020) ? "S" : "";
	flg += ((sh_flags[index] & 0x00000040) == 0x00000040) ? "I" : "";
	flg += ((sh_flags[index] & 0x00000080) == 0x00000080) ? "L" : "";
	flg += ((sh_flags[index] & 0x00000100) == 0x00000100) ? "O" : "";
	flg += ((sh_flags[index] & 0x00000200) == 0x00000200) ? "G" : "";
	flg += ((sh_flags[index] & 0x00000400) == 0x00000400) ? "T" : "";
	flg += ((sh_flags[index] & 0x00000800) == 0x00000800) ? "x" : "";
	flg += ((sh_flags[index] & 0x00001000) == 0x00001000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00002000) == 0x00002000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00004000) == 0x00004000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00008000) == 0x00008000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00010000) == 0x00010000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00020000) == 0x00020000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00040000) == 0x00040000) ? "x" : "";
	flg += ((sh_flags[index] & 0x00080000) == 0x00080000) ? "x" : "";
	if((sh_flags[index] & 0x00100000) == 0x00100000 || (sh_flags[index] & 0x00200000) == 0x00200000 ||
	   (sh_flags[index] & 0x00400000) == 0x00400000 || (sh_flags[index] & 0x00800000) == 0x00800000 ||
	   (sh_flags[index] & 0x01000000) == 0x01000000 || (sh_flags[index] & 0x02000000) == 0x02000000 ||
	   (sh_flags[index] & 0x04000000) == 0x04000000 || (sh_flags[index] & 0x08000000) == 0x08000000){
	    flg += "o";
	}
	if((sh_flags[index] & 0x10000000) == 0x10000000 || (sh_flags[index] & 0x20000000) == 0x20000000 ||
	   (sh_flags[index] & 0x40000000) == 0x40000000 || (sh_flags[index] & 0x80000000) == 0x80000000){
	    flg += "p";
	}
	for(int i = 3 - flg.length(); i > 0 ; --i){
	    flg = " " + flg;
	}

	String name = "";
	for(int i = sh_offset[e_shstrndx] + sh_name[index];; ++i){
	    char c = (char)(memory.read(i, 1)[0]);
	    if(c == 0x00){
		break;
	    }
	    name += String.valueOf(c);
	}
	for(int i = 17 - name.length(); i > 0 ; --i){
	    name += " ";
	}

	String type;
	if(0 <= sh_type[index] && sh_type[index] <= sht_string.length){
	    type = sht_string[sh_type[index]];
	}
	else if(0x70000000 <= sh_type[index] && sh_type[index] <= 0x70000000 + sht_string_mips.length){
	    type = sht_string_mips[sh_type[index] - 0x70000000];
	}
	else{
	    type = String.format("0x%08x", sh_type[index]);
	}
	for(int i = 15 - type.length(); i > 0 ; --i){
	    type += " ";
	}

	String value = "";
	value += String.format("  [%2d] ", index);
	value += name + " ";
	value += type + " ";
	value += String.format("%08x ", sh_addr[index]);
	value += String.format("%06x ", sh_offset[index]);
	value += String.format("%06x ", sh_size[index]);
	value += String.format("%02x ", sh_entsize[index]);
	value += flg + " ";
	value += String.format("%2d ", sh_link[index]);
	value += String.format("%2d ", sh_info[index]);
	value += String.format("%2d ", sh_addralign[index]);

	return value + "\n";
    }

    public RandomAccessMemory getPhysicalMemory(){
	return memory;
    }

    public int[] getMapping(){
	int counter = 0;
	for(int i = 0; i < sh_addr.length; ++i){
	    if(sh_addr[i] >= 0x00400000){
		++counter;
	    }
	}

	if(counter == 0){
	    return null;
	}

	int[] mapping = new int[counter * 3];
	int index = 0;
	for(int i = 0; i < sh_addr.length; ++i){
	    if(sh_addr[i] >= 0x00400000){
		mapping[index * 3] = sh_addr[i];
		mapping[index * 3 + 1] = sh_offset[i];
		mapping[index * 3 + 2] = sh_size[i];
		++index;
	    }
	}

	return mapping;
    }

    public int getEntryPoint(){
	return e_entry;
    }

    private int zeroExtend(byte[] data){
	int v = 0x00000000;
	switch(data.length){
	case 1:
	    v |= ((int)data[0] & 0x000000ff) << 0;
	    break;
	case 2:
	    v |= ((int)data[0] & 0x000000ff) << 8;
	    v |= ((int)data[1] & 0x000000ff) << 0;
	    break;
	case 4:
	    v |= ((int)data[0] & 0x000000ff) << 24;
	    v |= ((int)data[1] & 0x000000ff) << 16;
	    v |= ((int)data[2] & 0x000000ff) << 8;
	    v |= ((int)data[3] & 0x000000ff) << 0;
	    break;
	}
	return v;
    }

    private int signExtend(byte[] data){
	int v;
	if(((int)data[0] & 0x00000080) == 0x00000080){
	    v = 0xffffffff;
	}
	else{
	    v = 0x00000000;
	}
	
	switch(data.length){
	case 1:
	    v &= 0xffffff00;
	    v |= ((int)data[0] & 0x000000ff) << 0;
	    break;
	case 2:
	    v &= 0xffff0000;
	    v |= ((int)data[0] & 0x000000ff) << 8;
	    v |= ((int)data[1] & 0x000000ff) << 0;
	    break;
	case 4:
	    v &= 0x00000000;
	    v |= ((int)data[0] & 0x000000ff) << 24;
	    v |= ((int)data[1] & 0x000000ff) << 16;
	    v |= ((int)data[2] & 0x000000ff) << 8;
	    v |= ((int)data[3] & 0x000000ff) << 0;
	    break;
	}
	return v;
    }

}
