/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * GDBを用いたプロセッサシミュレーション要素
 */
package net.wasamon.mics.processor.gdb;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.wasamon.jgdb.GDBClient;
import net.wasamon.jgdb.GDBMIInterface;
import net.wasamon.jgdb.GDBXMLClient;
import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.ExecutableElementException;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mics.util.ChannelManager;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * GDBの"target sim"によるSHプロセッサシミュレータをMICSシミュレーション要素としてラップするクラス
 * @author Takefumi MIYOSHI
 */
public class SH3 extends MicsElement implements ExecutableElement,
		ChannelConnectable {

	private GDBClient gdb;

	class OperationCode {

		String op_name;

		InstInfo info;

		class Operand {
			int type;

			String value;

			public Operand(int type, String value) {
				this.type = type;
				this.value = value;
			}

			public String toString() {
				return value + "%" + InstInfo.getOpTypeString(type);
			}
		}

		Operand[] operand;

		int[] preOperandValue;

		int width;

		public int getOperandWidth(String name) {
			int len;
			if (op_name.indexOf(".W") > 0) {
				len = 2;
			} else if (op_name.indexOf(".L") > 0) {
				len = 4;
			} else if (op_name.indexOf(".B") > 0) {
				len = 1;
			} else {
				len = 2;
			}
			return len;
		}

		private Pattern codePattern = Pattern.compile("(\\S+)\\s*(.*)\\s*");

		public OperationCode(String code) throws Exception {
			Matcher m = codePattern.matcher(code);
			if (m.matches() == false) {
				throw new Exception("syntax error");
			}
			this.op_name = m.group(1).toUpperCase();
			this.width = getOperandWidth(op_name);
			this.operand = parseOperand(m.group(2));
			this.info = SHInstList.getInstance().getInstInfo(op_name, getType());
			preOperandValue = new int[operand.length];
			for (int i = 0; i < operand.length; i++) {
				if (operand[i].type == InstInfo.value) {
					preOperandValue[i] = getAccessAddress(i);
				}
			}
		}

		private int[] getType() {
			int[] t = new int[operand.length];
			for (int i = 0; i < t.length; i++) {
				t[i] = operand[i].type;
			}
			return t;
		}

		private Operand[] parseOperand(String o) {
			Operand[] operands = null;
			if (o.equals("")) {
				operands = new Operand[0];
			} else {
				String[] tmp = o.split(",");
				String[] op_value = new String[tmp.length];
				int op_index = 0;
				for (int i = 0; i < tmp.length; i++, op_index++) {
					if (tmp[i].indexOf('(') > 0) {
						op_value[op_index] = tmp[i] + "," + tmp[i + 1];
						i = i + 1;
					} else {
						op_value[op_index] = tmp[i];
					}
				}
				operands = new Operand[op_index];
				for (int i = 0; i < operands.length; i++) {
					operands[i] = new Operand(InstInfo.getOpType(op_value[i]),
							op_value[i]);
				}
			}
			return operands;
		}

		public String toString() {
			String str = "";
			str += "[OperationCode] " + op_name;
			for (int i = 0; i < operand.length; i++) {
				str += " " + operand[i];
			}
			return str;
		}

		public int getOperandSize() {
			return operand.length;
		}

		private Pattern dispRegPattern = Pattern.compile("@\\((\\d+),(.+)\\)");

		/**
		 * <dl>
		 * <dt>レジスタ直接:Rn
		 * <dt>レジスタ間接:@Rn
		 * <dt>ポストインクリメントレジスタ間接:@Rn+
		 * <dd>命令実行後にアクセス長に応じてRn = Rn+{1,2,4}と更新される．
		 * <dt>プレデクリメントレジスタ間接:@-Rn
		 * <dd>命令実行前にアクセス長に応じてRn = Rn-{1,2,4}と更新される．
		 * <dt>ディスプレースメント付きレジスタ相対:@(disp:4,Rn)
		 * <dd>4bitのdispは，ゼロ拡張後，オペランド長に応じて{1,2,4}倍され，Rnに加算される
		 * <dt>インデックス付きレジスタ間接:@(R0,Rn)
		 * <dd>RnとR0の和
		 * <dt>ディスプレースメント付きGBR間接:@(disp:8,GBR)
		 * <dd>8bitのdisp．GBR+disp*{1,2,4}
		 * <dt>インデックス付きGBR間接:@(R0,GBR)
		 * <dd>GBRとR0の和
		 * <dt>ディスプレースメント付きPC間接:@(disp:8,PC)
		 * <dd>8bitのdisp．ワードのときPC+disp*2，ロングワードのときPC&H'FFFFFFFC+disp*4
		 * <dt>PC相対:disp:8
		 * <dd>レジスタPCに8bitのdispを符号拡張後に2倍し加算した内容．PC+disp*2
		 * <dt>PC相対:disp:12
		 * <dd>レジスタPCに12bitのdispを符号拡張後に2倍し加算した内容．PC+disp*2
		 * <dt>PC相対:Rn
		 * <dd>PC+R0
		 * <dl>
		 * 即値:#imm:8(TST，AND，OR，XOR)
		 * <dd>ゼロ拡張
		 * <dl>
		 * 即値:#imm:8(MOV，ADD，CMP/EQ)
		 * <dd>符号ゼロ拡張
		 * <dl>
		 * 即値:#imm:8(TRAPA)
		 * <dd>ゼロ拡張後4倍
		 * </dl>
		 */
		public int getAccessAddress(int index) {
			String val = operand[index].value;
			Object r = null;
			if (val.startsWith("@r")) {// レジスタ間接
				if (val.endsWith("+")) { // ポストインクリメント
					r = gdb.readRegister(val.substring(1, val.length() - 1));
					// System.out.println("post increment: " + r);
					return DataUtil.parseInt((String) r);
				} else { // 普通のレジスタ間接
					r = gdb.readRegister(val.substring(1));
					return DataUtil.parseInt((String) r);
				}
			} else if (val.startsWith("@-r")) { // プレデクリメント
				r = gdb.readRegister(val.substring(2));
				return DataUtil.parseInt((String) r);
			} else if (val.startsWith("@(r0,")) {
				String tmp = val.substring(2, val.length() - 1);
				String[] regs = tmp.split(",");
				String r0 = (String) gdb.readRegister(regs[0]);
				String r1 = (String) gdb.readRegister(regs[1]);
				return DataUtil.parseInt(r0) + DataUtil.parseInt(r1);
			} else if (val.startsWith("@")) {
				Matcher m = dispRegPattern.matcher(val);
				if (m.matches()) {
					int disp = DataUtil.parseInt(m.group(1));
					r = gdb.readRegister(m.group(2));
					return DataUtil.parseInt((String) r) + disp;
				} else {
					System.out.println("unsupported access: " + val);
					return 0;
				}
			} else if (val.charAt(0) == '0') { // ���l
				String[] tmp = val.split("\\s+");
				return DataUtil.parseInt(tmp[0]);
			} else {
				System.out.println("unsupported access: " + val);
				return 0;
			}
		}

	}

	private ChannelManager channelManager;

	public SH3() {
	}

	private void memoryHook(OperationCode op) throws ExecutableElementException {
		if (op.operand[0].type == InstInfo.reg
				&& op.operand[1].type == InstInfo.value) {
			// operand[0] -> *operand[1]
			int src = DataUtil.parseInt((String) gdb
					.readRegister(op.operand[0].value));
			int destAddr = op.getAccessAddress(1);
			// System.out.println(op.op_name + "(" + op.width + ") val:" + src + "-> *" + destAddr);
			ChannelManager.Element c = channelManager.search(destAddr);
			try {
				MicsDataPacket p = null;
				byte[] b = DataUtil.toByteArray(src);
				switch (op.width) {
				case 1:
					p = RandomAccessMemoryDataPacket.writePacket(destAddr - c.offset, 1,
							8, new byte[] { b[3] });
					break;
				case 2:
					p = RandomAccessMemoryDataPacket.writePacket((destAddr - c.offset),
							1, 16, new byte[] { b[2], b[3] });
					break;
				case 4:
					p = RandomAccessMemoryDataPacket.writePacket((destAddr - c.offset),
							2, 16, b);
				}
				c.getChannel().writeRequest(this, p);
			} catch (MicsException e) {
				System.out.println("SH3 memory hook exception: " + e.getMessage());
				throw new ExecutableElementException(e);
			}
		} else if (op.operand[0].type == InstInfo.value
				&& op.operand[1].type == InstInfo.reg) {
			// *operand[0] -> operand[1]
			// String destReg = op.operand[1].value;
			destRegister = op.operand[1].value;
			destWidth = op.width;
			int srcAddr = op.preOperandValue[0];
			// System.out.println(op.op_name + "(" + op.width + ") val: *" + srcAddr + "->" + destReg);
			ChannelManager.Element c = channelManager.search(srcAddr);
			try {
				MicsDataPacket p = null;
				p = RandomAccessMemoryDataPacket.readPacket(srcAddr - c.offset,
						op.width, 8);
				// System.out.println("read request: " + p);
				c.getChannel().readRequest(this, p);
			} catch (MicsException e) {
				throw new ExecutableElementException(e);
			}
		} else {
			System.out.println("unsupported data transfer instruction:" + op);
		}
	}

	String destRegister;
	int destWidth;

	public void writeback(Channel src, MicsDataPacket d) {
		RandomAccessMemoryDataPacket r = (RandomAccessMemoryDataPacket) d;
		gdb.setValueToRegister("$" + destRegister, DataUtil.toBigEndianValueString(r.data));
	}

	public int getChannelOffset(Channel c) {
		return channelManager.search(c);
	}

	private boolean terminatedFlag;

	public ExecInfo exec_first() throws ExecutableElementException {
		ExecInfo status = new ExecInfo();
		if (terminatedFlag) {
			status.setTerminatableFlag(true);
		} else {
			try {
				String code = (String) gdb.readPCCode();
				String[] tmp = code.split(":\\s+");
				OperationCode op = new OperationCode(tmp[1]);
				gdb.stepInstruction();
				lastCode = op;
				status.setCycle(lastCode.info.getState());
				if (gdb.isStopped()) {
					if (gdb.getSignal() != null) {
						String signal = gdb.getSignal();
						if (signal.startsWith("SIGBUS") || signal.startsWith("SIGEMT")) {
							memoryHook(op);
						} else {
							System.out.println("signal:" + signal);
						}
						gdb.setValueToRegister("$pc", "$pc+2");
					} else {
						status.setTerminatableFlag(terminatedFlag = true);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
				throw new ExecutableElementException(e);
			}
		}
		return status;
	}

	public ExecInfo exec_second() throws ExecutableElementException {
		ExecInfo status = new ExecInfo();
		return status;
	}

	public void reset() {
		if (gdb != null) {
			gdb.doInit(new File(targetFilePath), "main");
		}
		terminatedFlag = false;
	}

	private String targetFilePath;

	private String targetGdbPath;

	private String targetGdbHostName;

	private int targetGdbPort;

	public void init(String type, String targetFilePath, String gdbPath)
			throws MicsException {
		this.targetGdbPath = gdbPath;
		this.targetFilePath = targetFilePath;
		try {
			if (type.equals("local")) {
				this.gdb = new GDBMIInterface(gdbPath);
			} else if (type.equals("xmlrpc")) {
				this.gdb = new GDBXMLClient(gdbPath);
			} else {
				throw new MicsException("illetal type value: " + type);
			}
		} catch (IOException e) {
			throw new MicsException(
					"cannot initialize net.wasamon.mics.processor.gdb.SH3");
		}
		gdb.doInit(new File(targetFilePath), "main");
	}

	/* generated automatically by MakeInitialize from inits/sh3.init */
	public void initialize(String base, Node n) throws MicsException {
		channelManager = new ChannelManager(composite);
		try {
			{
				Node init_var_node = n;
				String init_var_target;
				init_var_target = XMLParser.getAttribute(init_var_node, "target")
						.getNodeValue();
				String init_var_gdb;
				init_var_gdb = XMLParser.getAttribute(init_var_node, "gdb")
						.getNodeValue();
				String init_var_type;
				init_var_type = XMLParser.getAttribute(init_var_node, "type")
						.getNodeValue();

				init(init_var_type, init_var_target, init_var_gdb);

				{
					Node[] init_var__channel_obj = XMLParser.getNamedNodeArray(n,
							"channel");
					for (int i = 0; i < init_var__channel_obj.length; i++) {
						Node init_var__channel_node = init_var__channel_obj[i];
						String init_var__channel_id;
						init_var__channel_id = XMLParser.getAttribute(
								init_var__channel_node, "id").getNodeValue();
						int init_var__channel_offset;
						init_var__channel_offset = DataUtil.parseInt(XMLParser
								.getAttribute(init_var__channel_node, "offset").getNodeValue());

						channelManager.add(init_var__channel_id, init_var__channel_offset);

					}
				}
			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.gdb");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.gdb");
		}
	}

	private OperationCode lastCode;

	public void execStepCode() throws Exception {
		String code = (String) gdb.readPCCode();
		String[] tmp = code.split(":\\s+");
		OperationCode op = new OperationCode(tmp[1]);
		memoryHook(op);
		gdb.stepInstruction();
		memoryHook(op);
		lastCode = op;
	}

	public void execAll() {
		Calendar cal = Calendar.getInstance();
		System.out.println("start execution at " + cal.getTime());
		long start_time = cal.getTimeInMillis();
		int count = 0;
		try {
			while (true) {
				execStepCode();
				count++;
				if (gdb.isStopped()) {
					break;
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		cal = Calendar.getInstance();
		System.out.println("stop execution at " + cal.getTime());
		long time_span = cal.getTimeInMillis() - start_time;
		System.out.print("exec cycles: " + count);
		if (time_span > 0) {
			System.out.println(" C/S: " + ((double) count / (double) time_span)
					+ "kHz");
		} else {
			System.out.println("");
		}
	}

	public void shutdown() {
		gdb.doQuit();
	}

	public static void main(String args[]) throws Exception {
		SH3 proc = new SH3();
		proc.init("local", "l:\\a.out", "sh-hitachi-elf-gdb");
		proc.execAll();
		proc.shutdown();
	}

	public String[] getConnectedElements() {
		return channelManager.getConnectedElements();
	}

	public String getInfo() {
		String str = "";
		str += "id: " + super.id() + "\n";
		str += "class: " + this.getClass().getName() + "\n";
		str += "gdb: " + targetGdbPath + "\n";
		str += "host: " + targetGdbHostName + "\n";
		str += "port: " + targetGdbPort + "\n";
		str += "target: " + targetFilePath + "\n";
		ChannelManager.Element[] channels = channelManager.array();
		for (int i = 0; i < channels.length; i++) {
			ChannelManager.Element element = channels[i];
			str += "  Channel ID: " + element.id + ",";
			str += " offset = " + element.offset + "\n";
		}
		return str;
	}

	public String toString() {
		return getInfo() + lastCode.toString();

	}

	public String getImagePath() {
		return "superh.png";
	}

	public String getDescription(){
		return "TODO";
	}

}

class InstInfo {

	public static final int no_op = -1;

	public static final int imm = 0;

	public static final int reg = 1;

	public static final int value = 2;

	public static final int label = 2;

	public static final int pr = 4;

	String name;

	int[] operand;

	int state;

	int stage;

	int state_dsp;

	int stage_dsp;

	boolean dsp_modifier;

	InstInfo(String name, int[] operand, int state, int stage) {
		this.dsp_modifier = false;
		this.name = name;
		this.operand = operand;
		this.state = state;
		this.stage = stage;
	}

	InstInfo(String name, int[] operand, int state, int state_dsp, int stage,
			int stage_dsp) {
		this(name, operand, state, stage);
		this.dsp_modifier = true;
		this.state_dsp = state_dsp;
		this.stage_dsp = stage_dsp;
	}

	public int getState() {
		return state;
	}

	public int getStage() {
		return stage;
	}

	public boolean equals(String name, int[] operand) {
		if (this.name.equals(name) == false)
			return false;
		if (operand.length != this.operand.length)
			return false;
		for (int i = 0; i < operand.length; i++) {
			if (operand[i] != this.operand[i])
				return false;
		}
		return true;
	}

	public String toString() {
		String str = "";
		str += name + ":";
		String tmp = "";
		for (int i = 0; i < operand.length; i++) {
			tmp += "," + getOpTypeString(operand[i]);
		}
		if (tmp.length() > 1)
			str += tmp.substring(1);
		str += "  [";
		str += "states = " + state;
		if (state_dsp > 0)
			str += "(" + state_dsp + ")";
		str += ",stages = " + stage;
		if (state_dsp > 0)
			str += "(" + stage_dsp + ")";
		str += "]";
		return str;
	}

	public static String getOpTypeString(int type) {
		switch (type) {
		case no_op:
			return "none";
		case imm:
			return "imm";
		case reg:
			return "reg";
		case value:
			return "value";
		// case label:
		// return "label";
		case pr:
			return "pr";
		}
		return "unknown";
	}

	public static int getOpType(String op) {
		if (op == null || op.equals(""))
			return no_op;
		else if (op.charAt(0) == '#')
			return imm;
		else if (op.charAt(0) == 'r')
			return reg;
		else if (op.charAt(0) == '@')
			return value;
		else if (op.charAt(0) == '0')
			return label;
		else if (op.toLowerCase().equals("pr"))
			return pr;
		else
			return no_op;
	}

	public String getDescription(){
		return "TODO";
	}

}

class SHInstList {

	ArrayList list;

	private static SHInstList instance = new SHInstList();

	private SHInstList() {
		this.list = initInstInfo();
	}

	public static SHInstList getInstance() {
		return instance;
	}

	public InstInfo getInstInfo(String name, int[] operand) throws Exception {
		Iterator it = list.iterator();
		while (it.hasNext()) {
			InstInfo info = (InstInfo) (it.next());
			if (info.equals(name, operand)) {
				return info;
			}
		}
		String s = name;
		for (int i = 0; i < operand.length; i++) {
			s += " " + InstInfo.getOpTypeString(operand[i]);
		}
		throw new Exception("No Such Instruction : " + s);
	}

	private ArrayList initInstInfo() {
		ArrayList list = new ArrayList();
		/*
		 * レジスタ-レジスタ間転送命令 @実行ステート 1 @ステージ段数 5
		 */
		// MOV #imm,Rn
		list
				.add(new InstInfo("MOV", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		// MOV Rm,Rn
		list
				.add(new InstInfo("MOV", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// MOVA @(disp,PC),R0
		list.add(new InstInfo("MOVA", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVT Rn
		list.add(new InstInfo("MOVT", new int[] { InstInfo.reg }, 1, 5));
		// SWAP.B Rm,Rn
		list.add(new InstInfo("SWAP.B", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// SWAP.W Rm,Rn
		list.add(new InstInfo("SWAP.W", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// XTRCT Rm,Rn
		list.add(new InstInfo("XTRCT", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		/*
		 * メモリロード命令 @実行ステート 1 @ステージ段数 5 @競合
		 * この命令の直後にこの命令のデスティネーションレジスタを使う命令をおくと競合します MAはIFと競合します
		 */
		// MOV.W @(disp,PC),Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @(disp,PC),Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.B @Rm,Rn
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.W @Rm,Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @Rm,Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.B @Rm+,Rn
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.W @Rm+,Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @Rm+,Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.B @(disp,Rm),R0
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.W @(disp,Rm),R0
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @(disp,Rm),Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.B @(R0,Rm),Rn
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.W @(R0,Rm),Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @(R0,Rm),Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.B @(disp,GBR),R0
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.W @(disp,GBR),R0
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOV.L @(disp,GBR),R0
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		/*
		 * メモリストア命令 @実行ステート 1 @ステージ段数 4 @競合 MAはIFと競合します
		 */
		// MOV.B Rm,@Rn
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.W Rm,@Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.L Rm,@Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.B R0,@(disp,Rn)
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.W R0,@(disp,Rn)
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.L Rm,@(disp,Rn)
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.B Rm,@(R0,Rn)
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.W Rm,@(R0,Rn)
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.L Rm,@(R0,Rn)
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.B R0,@(disp,GBR)
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.W R0,@(disp,GBR)
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOV.L R0,@(disp,GBR)
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		/*
		 * メモリストア命令(プリデクリメント) @実行ステート 1 @ステージ段数 5 @競合 MAはIFと競合します
		 */
		// MOV.B Rm,@-Rn
		list.add(new InstInfo("MOV.B", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOV.W Rm,@-Rn
		list.add(new InstInfo("MOV.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOV.L Rm,@-Rn
		list.add(new InstInfo("MOV.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		/*
		 * キャッシュ命令 @実行ステート 1/2 @ステージ段数 4 @競合 MAはIFと競合します
		 */
		// PREF @Rn
		list.add(new InstInfo("PREF", new int[] { InstInfo.value }, 1, 2, 4, 4));
		/*
		 * レジスタ間算術演算命令(除算系命令を除く) @実行ステート 1 @ステージ段数 5
		 */
		// ADD Rm,Rn
		list
				.add(new InstInfo("ADD", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// ADD #InstInfo.imm,Rn
		list
				.add(new InstInfo("ADD", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		// EXTS.B Rm,Rn
		list.add(new InstInfo("EXTS.B", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// EXTS.W Rm,Rn
		list.add(new InstInfo("EXTS.W", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// EXTU.B Rm,Rn
		list.add(new InstInfo("EXTU.B", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// EXTU.W Rm,Rn
		list.add(new InstInfo("EXTU.W", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// NEG Rm,Rn
		list
				.add(new InstInfo("NEG", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// SUB Rm,Rn
		list
				.add(new InstInfo("SUB", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		/*
		 * SR更新レジスタ間算術演算命令(除算系命令を除く) @実行ステート 1 @ステージ段数 5 @競合
		 * この命令の直後あるいはその次にSRを読み出す命令をおくと競合します
		 */
		// ADDC Rm,Rn
		list.add(new InstInfo("ADDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// ADDV Rm,Rn
		list.add(new InstInfo("ADDV", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// CMP/EQ #InstInfo.imm,R0
		list.add(new InstInfo("CMP/EQ", new int[] { InstInfo.imm, InstInfo.reg },
				1, 5));
		// CMP/EQ Rm,Rn
		list.add(new InstInfo("CMP/EQ", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// CMP/HS Rm,Rn
		list.add(new InstInfo("CMP/HS", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// CMP/GE Rm,Rn
		list.add(new InstInfo("CMP/GE", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// CMP/HI Rm,Rn
		list.add(new InstInfo("CMP/HI", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// CMP/GT Rm,Rn
		list.add(new InstInfo("CMP/GT", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// CMP/PL Rn
		list.add(new InstInfo("CMP/PL", new int[] { InstInfo.reg }, 1, 5));
		// CMP/PZ Rn
		list.add(new InstInfo("CMP/PZ", new int[] { InstInfo.reg }, 1, 5));
		// CMP/STR Rm,Rn
		list.add(new InstInfo("CMP_STR", new int[] { InstInfo.reg, InstInfo.reg },
				1, 5));
		// DIV1 Rm,Rn
		list.add(new InstInfo("DIV1", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DIV0S Rm,Rn
		list.add(new InstInfo("DIVOS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DIV0U
		list.add(new InstInfo("DIVOU", new int[] {}, 1, 5));
		// DT Rn
		list.add(new InstInfo("DT", new int[] { InstInfo.reg }, 1, 5));
		// NEGC Rm,Rn
		list.add(new InstInfo("NEGC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// SUBC Rm,Rn
		list.add(new InstInfo("SUBC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// SUBV Rm,Rn
		list.add(new InstInfo("SUBV", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		/*
		 * 積和命令 @実行ステート 2(*5) @ステージ段数 8 @競合 乗算器との競合をおこします MAはIFと競合します
		 */
		// MAC.W @Rm+,@Rn+
		list.add(new InstInfo("MAC.W",
				new int[] { InstInfo.value, InstInfo.value }, 2, 5, 8, 8));
		/*
		 * 倍精度積和命令 @実行ステート 2(*5) @ステージ段数 8 @競合 乗算器との競合をおこします MAはIFと競合します
		 */
		// MAC.L @Rm+,@Rn+
		list.add(new InstInfo("MAC.L",
				new int[] { InstInfo.value, InstInfo.value }, 2, 5, 8, 8));
		/*
		 * 乗算命令 @実行ステート 1(*3) @ステージ段数 6 @競合 乗算器との競合をおこします MAはIFと競合します
		 */
		// MULS.W Rm,Rn
		list.add(new InstInfo("MULS.W", new int[] { InstInfo.reg, InstInfo.reg },
				1, 3, 6, 6));
		// MULU.W Rm,Rn
		list.add(new InstInfo("MULU.W", new int[] { InstInfo.reg, InstInfo.reg },
				1, 3, 6, 6));
		list.add(new InstInfo("MUL.L", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 6, 6));
		/*
		 * 倍精度乗算命令 @実行ステート 2(*5) @ステージ段数 8 @競合 乗算器との競合をおこします MAはIFと競合します
		 */
		// DMULS.L Rm,Rn
		list.add(new InstInfo("DMULS.L", new int[] { InstInfo.reg, InstInfo.reg },
				2, 5, 8, 8));
		// DMULU.L Rm,Rn
		list.add(new InstInfo("DMULU.L", new int[] { InstInfo.reg, InstInfo.reg },
				2, 5, 8, 8));
		/*
		 * レジスタ間論理演算命令 @実行ステート 1 @ステージ段数 5
		 */
		// AND Rm,Rn
		list
				.add(new InstInfo("AND", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// AND #InstInfo.imm,R0
		list
				.add(new InstInfo("AND", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		// NOT Rm,Rn
		list
				.add(new InstInfo("NOT", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// OR Rm,Rn
		list
				.add(new InstInfo("OR", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// OR #InstInfo.imm,R0
		list
				.add(new InstInfo("OR", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		// XOR Rm,Rn
		list
				.add(new InstInfo("XOR", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// XOR #InstInfo.imm,R0
		list
				.add(new InstInfo("XOR", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		/*
		 * SR更新レジスタ間論理演算命令 @実行ステート 1 @ステージ段数 5 @競合
		 * この命令の直後あるいはその次にSRを読み出す命令をおくと競合します
		 */
		// TST Rm,Rn
		list
				.add(new InstInfo("TST", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// TST #InstInfo.imm,R0
		list
				.add(new InstInfo("TST", new int[] { InstInfo.imm, InstInfo.reg }, 1, 5));
		/*
		 * メモリ論理演算命令 @実行ステート 3 @ステージ段数 6 @競合 MAはIFと競合します
		 */
		// AND.B #InstInfo.imm,@(R0,GBR)
		list.add(new InstInfo("AND.B", new int[] { InstInfo.imm, InstInfo.value },
				3, 6));
		// OR.B #InstInfo.imm,@(R0,GBR)
		list.add(new InstInfo("OR.B", new int[] { InstInfo.imm, InstInfo.value },
				3, 6));
		// XOR.B #InstInfo.imm,@(R0,GBR)
		list.add(new InstInfo("XOR.B", new int[] { InstInfo.imm, InstInfo.value },
				3, 6));
		/*
		 * SR更新メモリ論理演算命令 @実行ステート 3 @ステージ段数 7 @競合
		 * この命令の直後あるいはその次にSRを読み出す命令を置くと競合します。
		 */
		// TST.B #InstInfo.imm,@(R0,GBR)
		list.add(new InstInfo("TST.B", new int[] { InstInfo.imm, InstInfo.value },
				3, 7));
		/*
		 * TAS 命令 @実行ステート 3/4(*3) @ステージ段数 7 @競合 MAはIFと競合します。
		 */
		// TAS.B @Rn
		list.add(new InstInfo("TAS.B", new int[] { InstInfo.value }, 3, 4, 7, 7));
		/*
		 * シフト命令 @実行ステート 1 @ステージ段数 5
		 */
		// SHLL2 Rn
		list.add(new InstInfo("SHLL2", new int[] { InstInfo.reg }, 1, 5));
		// SHLR2 Rn
		list.add(new InstInfo("SHLR2", new int[] { InstInfo.reg }, 1, 5));
		// SHLL8 Rn
		list.add(new InstInfo("SHLL8", new int[] { InstInfo.reg }, 1, 5));
		// SHLR8 Rn
		list.add(new InstInfo("SHLR8", new int[] { InstInfo.reg }, 1, 5));
		// SHLL16 Rn
		list.add(new InstInfo("SHLL16", new int[] { InstInfo.reg }, 1, 5));
		// SHLR16 Rn
		list.add(new InstInfo("SHLR16", new int[] { InstInfo.reg }, 1, 5));
		// SHAD Rm,Rn
		list.add(new InstInfo("SHAD", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// SHLD Rm,Rn
		list.add(new InstInfo("SHLD", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		/*
		 * SR更新シフト命令 @実行ステート 1 @ステージ段数 5 @競合 この命令の直後あるいはその次にSRを読み出す命令を置くと競合します。
		 */
		// ROTL Rn
		list.add(new InstInfo("ROTL", new int[] { InstInfo.reg }, 1, 5));
		// ROTR Rn
		list.add(new InstInfo("ROTR", new int[] { InstInfo.reg }, 1, 5));
		// ROTCL Rn
		list.add(new InstInfo("ROTCL", new int[] { InstInfo.reg }, 1, 5));
		// ROTCR Rn
		list.add(new InstInfo("ROTCR", new int[] { InstInfo.reg }, 1, 5));
		// SHAL Rn
		list.add(new InstInfo("SHAL", new int[] { InstInfo.reg }, 1, 5));
		// SHAR Rn
		list.add(new InstInfo("SHAR", new int[] { InstInfo.reg }, 1, 5));
		// SHLL Rn
		list.add(new InstInfo("SHLL", new int[] { InstInfo.reg }, 1, 5));
		// SHLR Rn
		list.add(new InstInfo("SHLR", new int[] { InstInfo.reg }, 1, 5));
		/*
		 * 条件分岐命令 @実行ステート 3/1(*4) @ステージ段数 3 @競合
		 */
		// BF lavel
		list.add(new InstInfo("BF", new int[] { InstInfo.label }, 3, 1, 3, 3));
		// BT lavel
		list.add(new InstInfo("BT", new int[] { InstInfo.label }, 3, 1, 3, 3));
		/*
		 * 遅延付き条件分岐命令 @実行ステート 2/1(*4) @ステージ段数 3 @競合
		 */
		// BF/S lavel
		list.add(new InstInfo("BF/S", new int[] { InstInfo.label }, 2, 1, 3, 3));
		list.add(new InstInfo("BF.S", new int[] { InstInfo.label }, 2, 1, 3, 3));
		// BT/S lavel
		list.add(new InstInfo("BT/S", new int[] { InstInfo.label }, 2, 1, 3, 3));
		list.add(new InstInfo("BT.S", new int[] { InstInfo.label }, 2, 1, 3, 3));
		/*
		 * 無条件分岐命令 @実行ステート 2 @ステージ段数 3 @競合
		 */
		// BRA InstInfo.label
		list.add(new InstInfo("BRA", new int[] { InstInfo.label }, 2, 3));
		// BRAF Rm
		list.add(new InstInfo("BRAF", new int[] { InstInfo.reg }, 2, 3));
		// JMP @Rm
		list.add(new InstInfo("JMP", new int[] { InstInfo.value }, 2, 3));
		// RTS
		list.add(new InstInfo("RTS", new int[] {}, 2, 3));
		/*
		 * 無条件分岐命令(PR) @実行ステート 2 @ステージ段数 5
		 */
		// BSR lavel
		list.add(new InstInfo("BSR", new int[] { InstInfo.label }, 2, 5));
		// BSRF Rm
		list.add(new InstInfo("BSRF", new int[] { InstInfo.reg }, 2, 5));
		// JSR @Rm
		list.add(new InstInfo("JSR", new int[] { InstInfo.value }, 2, 5));
		/*
		 * システム制御 @実行ステート 1/3(*5) @ステージ段数 5
		 */
		// LDC Rm,GBR
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,VBR
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,SSR
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,SPC
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		/*
		 * システム制御 @実行ステート 3 @ステージ段数 5
		 */
		// LDC Rm,MOD
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 3, 5));
		// LDC Rm,RE
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 3, 5));
		// LDC Rm,RS
		list
				.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 3, 5));
		/*
		 * システム制御 @実行ステート 1/3(*5) @ステージ段数 5
		 */
		// LDC Rm,R0_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R1_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R2_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R3_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R4_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R5_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R6_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		// LDC Rm,R7_BANK
		list.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				3, 5, 5));
		/*
		 * システム制御 @実行ステート 3 @ステージ段数 5
		 */
		// SETRC Rm
		list.add(new InstInfo("SETRC", new int[] { InstInfo.reg }, 3, 5));
		// SETRC #InstInfo.imm
		list.add(new InstInfo("SETRC", new int[] { InstInfo.imm }, 3, 5));
		// LDRE @(disp, PC)
		list.add(new InstInfo("SETRC", new int[] { InstInfo.value }, 3, 5));
		// LDRS @(disp, PC)
		list.add(new InstInfo("SETRC", new int[] { InstInfo.value }, 3, 5));
		/*
		 * システム制御 @実行ステート 1 @ステージ段数 5
		 */
		// LDS Rm,PR
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.pr }, 1, 5));
		// STC SR,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC GBR,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC VBR,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC SSR, Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC SPC,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC MOD,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC RE,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC RS,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R0_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R1_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R2_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R3_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R4_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R5_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R6_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STC R7_BANK,Rn
		list
				.add(new InstInfo("STC", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS PR,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.pr, InstInfo.reg }, 1, 5));
		// STS PR,Rn
		list.add(new InstInfo("STS", new int[] { InstInfo.no_op, InstInfo.reg }, 1,
				5));
		/*
		 * SR更新システム制御 @実行ステート 1 @ステージ段数 5 @競合 この命令の直後あるいはその次にSRを読み出す命令を置くと競合します。
		 */
		// CLRS
		list.add(new InstInfo("CLRS", new int[] {}, 1, 5));
		// CLRT
		list.add(new InstInfo("CLRT", new int[] {}, 1, 5));
		// SETS
		list.add(new InstInfo("SETS", new int[] {}, 1, 5));
		// SETT
		list.add(new InstInfo("SETT", new int[] {}, 1, 5));
		/*
		 * LDTLB命令 @実行ステート 1 @ステージ段数 4 @競合 MA はIF と競合します。
		 */
		// LDTLB
		list.add(new InstInfo("LDTLB", new int[] {}, 1, 4));
		/*
		 * NOP命令 @実行ステート 1 @ステージ段数 3
		 */
		// NOP
		list.add(new InstInfo("NOP", new int[] {}, 1, 3));
		/*
		 * LDC命令(SR) @実行ステート 5 @ステージ段数 5
		 */
		// LDC Rm,SR
		list
				.add(new InstInfo("LDC", new int[] { InstInfo.reg, InstInfo.reg }, 7, 7));
		/*
		 * LDC.L命令(SR) @実行ステート 7 @ステージ段数 7
		 */
		// LDC.L @Rm+,SR
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				7, 7));
		/*
		 * LDC.L命令(PR) @実行ステート 1 @ステージ段数 5 @競合
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 */
		// LDS.L @Rm+,PR
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.pr },
				1, 5));
		/*
		 * STS.L 命令(PR) @実行ステート 1 @ステージ段数 5 @競合 MA はIF と競合します。
		 */
		// STS.L PR,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.pr, InstInfo.value },
				1, 5));
		/*
		 * LDC.L 命令 @実行ステート 1/5(*6) @ステージ段数 5 @競合
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 */
		// LDC.L @Rm+,GBR
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,VBR
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,SSR
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,SPC
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		/*
		 * LDC.L 命令 @実行ステート 5 @ステージ段数 5 @競合
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 */
		// LDC.L @Rm+,MOD
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				5, 5));
		// LDC.L @Rm+,RE
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				5, 5));
		// LDC.L @Rm+,RS
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				5, 5));
		/*
		 * LDC.L 命令 @実行ステート 1/5(*6) @ステージ段数 5 @競合
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 */
		// LDC.L @Rm+,R0_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R1_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R2_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R3_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R4_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R5_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R6_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		// LDC.L @Rm+,R7_BANK
		list.add(new InstInfo("LDC.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 5, 5, 5));
		/*
		 * STC.L 命令（除くバンクレジスタ） @実行ステート 1/2(*7) @ステージ段数 5 @競合 MA はIF と競合します。
		 */
		// STC.L SR,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 2, 5, 5));
		// STC.L GBR,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 2, 5, 5));
		// STC.L VBR,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 2, 5, 5));
		// STC.L SSR,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 2, 5, 5));
		// STC.L SPC,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 2, 5, 5));
		/*
		 * STC.L 命令（除くバンクレジスタ） @実行ステート 2 @ステージ段数 5 @競合 MA はIF と競合します。
		 */
		// STC.L MOD,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 5));
		// STC.L RE,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 5));
		// STC.L RS,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 5));
		/*
		 * STC.L 命令（バンクレジスタ） @実行ステート 2 @ステージ段数 6 @競合 MA はIF と競合します。
		 */
		// STC.L R0_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R1_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R2_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R3_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R4_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R5_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R6_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		// STC.L R7_BANK,@-Rn
		list.add(new InstInfo("STC.L", new int[] { InstInfo.reg, InstInfo.value },
				2, 6));
		/*
		 * レジスタ→MAC/DSP転送命令 @実行ステート 1 @ステージ段数 6 @競合 乗算器との競合を起こします。 MA はIFと競合します。
		 */
		// CLRMAC
		list.add(new InstInfo("CLRMAC", new int[] {}, 1, 6));
		// LDS Rm,MACH
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,MACL
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,DSR
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,A0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,X0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,X1
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,Y0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		// LDS Rm,Y1
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 6));
		/*
		 * メモリ→MAC/DSP転送命令 @実行ステート 1 @ステージ段数 4 @競合 乗算器との競合を起こします。 MA はIF と競合します。
		 */
		// LDS.L @Rm+,MACH
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,MACL
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,DSR
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,A0
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,X0
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,X1
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,Y0
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,Y1
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		/*
		 * MAC/DSP→レジスタ転送命令 @実行ステート 1 @ステージ段数 5 @競合 乗算器との競合を起こします。
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 */
		// STS MACH,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS MACL,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS DSR, Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS A0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS X0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS X1,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS Y0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS Y1,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		/*
		 * MAC/DSP→メモリ転送命令 @実行ステート 1 @ステージ段数 5 @競合 乗算器との競合を起こします。 MAはIFと競合します。
		 */
		// STS.L MACH,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L MACL,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L DSR,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L A0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L X0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L X1,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L Y0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// STS.L Y1,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		/*
		 * RTE命令 @実行ステート 4 @ステージ段数 5
		 */
		// RTE
		list.add(new InstInfo("RTE", new int[] {}, 4, 5));
		/*
		 * TRAP 命令 @実行ステート 6/8(*8) @ステージ段数 6/8(*8)
		 */
		// TRAPA #InstInfo.imm
		list.add(new InstInfo("TRAPA", new int[] { InstInfo.imm }, 6, 8, 6, 8));
		/*
		 * SLEEP 命令 @実行ステート 4 @ステージ段数 6
		 */
		// SLEEP
		list.add(new InstInfo("SLEEP", new int[] {}, 4, 6));
		/*
		 * レジスタ→DSP転送命令 @実行ステート 1 @ステージ段数 4 @競合 乗算器との競合を起こします。 MA はIF と競合します。
		 */
		// CLRMAC
		list.add(new InstInfo("CLRMAC", new int[] {}, 1, 4));
		// LDS Rm,MACH
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,MACL
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		/*
		 * レジスタ→DSP転送命令 @実行ステート 1 @ステージ段数 4
		 */
		// LDS Rm,DSR
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,A0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,X0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,X1
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,Y0
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		// LDS Rm,Y1
		list
				.add(new InstInfo("LDS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 4));
		/*
		 * メモリ→DSP転送命令 @実行ステート 1 @ステージ段数 4 @競合 乗算器との競合を起こします。 MA はIF と競合します。
		 */
		// LDS.L @Rm+,MACH
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// LDS.L @Rm+,MACL
		list.add(new InstInfo("LDS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		/*
		 * メモリ→DSP転送命令 @実行ステート 1 @ステージ段数 4
		 */
		// DS.L @Rm+,DSR
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// DS.L @Rm+,A0
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// DS.L @Rm+,X0
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// DS.L @Rm+,X1
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// DS.L @Rm+,Y0
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		// DS.L @Rm+,Y1
		list.add(new InstInfo("DS.L", new int[] { InstInfo.value, InstInfo.reg },
				1, 4));
		/*
		 * DSP→レジスタ転送命令 @実行ステート 1 @ステージ段数 5 @競合 乗算器との競合を起こします。
		 * この命令の直後に、この命令のデスティネーションレジスタを使う命令を置くと、競合します。 MA はIF と競合します。
		 * DSP演算との競合を起こします。
		 */
		// STS MACH,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS MACL,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS DSR,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS A0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS X0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS X1,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS Y0,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		// STS Y1,Rn
		list
				.add(new InstInfo("STS", new int[] { InstInfo.reg, InstInfo.reg }, 1, 5));
		/*
		 * DSP→メモリ転送命令 1 4 乗算器との競合を起こします。 MA はIF と競合します。
		 */
		// STS.L MACH,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L MACL,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		/*
		 * DSP→メモリ転送命令 1 4
		 */
		// STS.L DSR,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L A0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L X0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L X1,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L Y0,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// STS.L Y1,@-Rn
		list.add(new InstInfo("STS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		/*
		 * RTE 命令 4 5
		 */
		// RTE
		list.add(new InstInfo("RTE", new int[] {}, 4, 5));
		/*
		 * TRAP 命令 8 9
		 */
		// TRAPA #InstInfo.imm
		list.add(new InstInfo("TRAPA", new int[] { InstInfo.imm }, 8, 9));
		/*
		 * SLEEP命令 3 3
		 */
		// SLEEP
		list.add(new InstInfo("SLEEP", new int[] {}, 3, 3));
		/*
		 * Xメモリロード命令 1 5
		 */
		// NOPX
		list.add(new InstInfo("NOPX", new int[] {}, 1, 5));
		// MOVX.W @Ax,Dx
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVX.W @Ax+,Dx
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVX.W @Ax+Ix,Dx
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		/*
		 * Xメモリストア命令 1 4 DSP演算との競合を起こします。
		 */
		// MOVX.W Da,@Ax
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOVX.W Da,@Ax+
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOVX.W Da,@Ax+Ix
		list.add(new InstInfo("MOVX.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		/*
		 * Yメモリロード命令 1 5
		 */
		// NOPY
		list.add(new InstInfo("NOPY", new int[] {}, 1, 5));
		// MOYV.W @Ay,Dy
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVY.W @Ay+,Dy
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVY.W @Ay+Ix,Dy
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		/*
		 * Yメモリストア命令 1 4 DSP演算との競合を起こします。
		 */
		// MOVY.W Da,@Ay
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOVY.W Da,@Ay+
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		// MOVY.W Da@Ay+Iy
		list.add(new InstInfo("MOVY.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 4));
		/*
		 * シングルロード命令 1 5 MAはIFと競合します。
		 */
		// MOVS.W @-As,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.W @As,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.W @As+,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.W @As+Is,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.L @-As,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.L @As,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.L @As+,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		// MOVS.L @As+Is,Ds
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.value, InstInfo.reg },
				1, 5));
		/*
		 * シングルストア命令 1 5 MA はIF と競合します。 DSP演算との競合を起こします。
		 */
		// MOVS.W Ds,@-As
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.W Ds,@As
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.W Ds,@As+
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.W Ds,@As+Is
		list.add(new InstInfo("MOVS.W", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.L Ds,@-As
		list.add(new InstInfo("MOVS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.L Ds,@As
		list.add(new InstInfo("MOVS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.L Ds,@As+
		list.add(new InstInfo("MOVS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		// MOVS.L Ds,@As+Is
		list.add(new InstInfo("MOVS.L", new int[] { InstInfo.reg, InstInfo.value },
				1, 5));
		/*
		 * DSP演算命令 ALU算術演算命令 1 5
		 */
		// PADD Sx, Sy,Dz(Du)
		list.add(new InstInfo("PADD", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PADD Sx, Sy,Dz
		list.add(new InstInfo("PADD", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PADD Sx, Sy,Dz
		list.add(new InstInfo("PADD", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PSUB Sx, Sy,Dz(Du)
		list.add(new InstInfo("PSUB", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PSUB Sx, Sy,Dz
		list.add(new InstInfo("PSUB", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PSUB Sx, Sy,Dz
		list.add(new InstInfo("PSUB", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PCOPY Sx, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PCOPY Sx, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PCOPY Sx, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PCOPY Sy, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PCOPY Sy, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PCOPY Sy, Dz
		list.add(new InstInfo("PCOPY", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PDMSB Sx,Dz
		list.add(new InstInfo("PDMSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DTC PDMSB Sx,Dz
		list.add(new InstInfo("PDMSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PDMSB Sx,Dz
		list.add(new InstInfo("PDMSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PDMSB Sy,Dz
		list.add(new InstInfo("PDMSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PDMSB Sy,Dz
		list.add(new InstInfo("PDMSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PDMSB Sy,Dz
		list.add(new InstInfo("PMBSB", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PINC Sx,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PINC Sx,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PINC Sx,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PINC Sy,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PINC Sy,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PINC Sy,Dz
		list.add(new InstInfo("PINC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PNEG Sx,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PNEG Sx,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PNEG Sx,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PNEG Sy,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PNEG Sy,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PNEG Sy,Dz
		list.add(new InstInfo("PNEG", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PDEC Sx,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DTC PDEC Sx,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PDEC Sx,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PDEC Sy,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DTC PDEC Sy,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PDEC Sy,Dz
		list.add(new InstInfo("PDEC", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PCLR Dz
		list.add(new InstInfo("PCLR", new int[] { InstInfo.reg }, 1, 5));
		// DCT PCLR Dz
		list.add(new InstInfo("PCLR", new int[] { InstInfo.reg }, 1, 5));
		// DCF PCLR Dz
		list.add(new InstInfo("PCLR", new int[] { InstInfo.reg }, 1, 5));
		// PADDC Sx,Sy,Dz
		list.add(new InstInfo("PADDC", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PSUBC Sx,Sy,Dz
		list.add(new InstInfo("PSUBC", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PCMP Sx,Sy,
		list.add(new InstInfo("PCMP", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PABS Sx,Dz
		list.add(new InstInfo("PABS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PABS Sy,Dz
		list.add(new InstInfo("PABS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PRNDSx,Dz
		list.add(new InstInfo("PRNDSx", new int[] { InstInfo.reg }, 1, 5));
		// PRNDSy,Dz
		list.add(new InstInfo("PRNDSy", new int[] { InstInfo.reg }, 1, 5));
		/*
		 * DSP演算命令 ALU論理演算命令 1 5
		 */
		// POR Sx,Sy,Dz
		list.add(new InstInfo("POR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT POR Sx,Sy,Dz
		list.add(new InstInfo("POR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF POR Sx,Sy,Dz
		list.add(new InstInfo("POR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PAND Sx,Sy,Dz
		list.add(new InstInfo("PAND", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PAND Sx,Sy,Dz
		list.add(new InstInfo("PAND", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PAND Sx,Sy,Dz
		list.add(new InstInfo("PAND", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PXOR Sx,Sy,Dz
		list.add(new InstInfo("PXOR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PXOR Sx,Sy,Dz
		list.add(new InstInfo("PXOR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PXOR Sx,Sy,Dz
		list.add(new InstInfo("PXOR", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		/*
		 * DSP演算命令 シフト命令 1 5
		 */
		// PSHA Sx,Sy,Dz
		list.add(new InstInfo("PSHA", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PSHA Sx,Sy,Dz
		list.add(new InstInfo("PSHA", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PSHA Sx,Sy,Dz
		list.add(new InstInfo("PSHA", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PSHA #InstInfo.imm,Dz
		list.add(new InstInfo("PSHA", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PSHL Sx,Sy,Dz
		list.add(new InstInfo("PSHL", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCT PSHL Sx,Sy,Dz
		list.add(new InstInfo("PSHL", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// DCF PSHL Sx,Sy,Dz
		list.add(new InstInfo("PSHL", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		// PSHL #InstInfo.imm,Dz
		list.add(new InstInfo("PSHL", new int[] { InstInfo.imm, InstInfo.reg }, 1,
				5));
		/*
		 * DSP演算命令 乗算命令 1 5
		 * 
		 */
		// PMULS Se,Sf,Dg
		list.add(new InstInfo("PMULS", new int[] { InstInfo.reg, InstInfo.reg,
				InstInfo.reg }, 1, 5));
		/*
		 * DSP演算命令 レジスタ間転送命令 1 5 MOVX.W,MOVS.W,MOVS.L との競合
		 */
		// PSTS MACH,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DTC PSTS MACH,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PSTS MACH,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PSTS MACL,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PSTS MACL,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PSTS MACL,Dz
		list.add(new InstInfo("PSTS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PLDS Dz,MACH
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PLDS Dz,MACH
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PLDS Dz,MACH
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// PLDS Dz,MACL
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCT PLDS Dz,MACL
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));
		// DCF PLDS Dz,MACL
		list.add(new InstInfo("PLDS", new int[] { InstInfo.reg, InstInfo.reg }, 1,
				5));

		return list;
	}

}
