/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * プロセッサをシミュレーションする要素のパッケージ
 */
package net.wasamon.mics.processor;

import net.wasamon.mjlib.util.DataUtil;

/**
 * 整数レジスタ
 * @author Takefumi MIYOSHI
 *
 */
public class IntRegister implements Register {
	int data;

	public IntRegister() {
		data = 0;
	}

	public int intValue() {
		return data;
	}

	public byte[] byteArray() {
		return DataUtil.toByteArray(data);
	}

	public byte[] byteArray(int offset, int len) {
		return DataUtil.subArray(DataUtil.toByteArray(data), offset, len);
	}

	public void reset() {
		data = 0;
	}

	/**
	 * ���W�X�^�̃f�[�^�ⱂ̃��W�X�^�ɃR�s�[����
	 * 
	 * @param r
	 */
	public void set(Register r) {
		data = r.intValue();
	}

	/**
	 * int�^�̐����㌃W�X�^�f�[�^�Ƃ��ăZ�b�g����
	 * 
	 * @param v
	 */
	public void set(int v) {
		data = v;
	}

	/**
	 * byte�^�̐����㌃W�X�^�f�[�^�Ƃ��ăZ�b�g����D�f�[�^�͕����g�������
	 * 
	 * @param v
	 */
	public void set(byte b) {
		data = (int) b;
	}

	/**
	 * char�^�̐����㌃W�X�^�f�[�^�Ƃ��ăZ�b�g����D�f�[�^�͕����g�������
	 * 
	 * @param v
	 */
	public void set(char v) {
		data = DataUtil.toSiginedInteger(v);
	}

	public void setL(char v) {
		data &= 0xffff0000;
		data += ((int) v) & 0x0000ffff;
	}

	public void setH(char v) {
		data &= 0x0000ffff;
		data += (((int) v) << 16) & 0xffff0000;
	}

	public void set(byte[] d) {
		data = DataUtil.toInteger(d);
	}

	public void set(byte[] d, int offset, int len) {
		data = DataUtil.toInteger(d, offset, len);
	}

	public void add(int v) {
		data += v;
	}

	public Register add(Register r0, Register r1) {
		data = r0.intValue() + r1.intValue();
		return this;
	}

	public Register mult(Register r0, Register r1) {
		data = r0.intValue() * r1.intValue();
		return this;
	}

	public Register div(Register r0, Register r1) {
		data = r0.intValue() / r1.intValue();
		return this;
	}

	public Register mod(Register r0, Register r1) {
		data = r0.intValue() % r1.intValue();
		return this;
	}

	public Register sub(Register r0, Register r1) {
		data = r0.intValue() - r1.intValue();
		return this;
	}

	public void neg(Register r0) {
		data = -r0.intValue();
	}

	public Register shiftL(Register r0, Register r1) {
		data = r0.intValue() << r1.intValue();
		return this;
	}

	public Register shiftR(Register r0, Register r1) {
		data = r0.intValue() >> r1.intValue();
		return this;
	}

	public Register and(Register r0, Register r1) {
		data = r0.intValue() & r1.intValue();
		return this;
	}

	public Register or(Register r0, Register r1) {
		data = r0.intValue() | r1.intValue();
		return this;
	}

	public boolean eqeq(Register r0) {
		return this.intValue() == r0.intValue();
	}

	public boolean gt(Register r0) {
		return this.intValue() > r0.intValue();
	}

	public String toString() {
		return DataUtil.toBigEndianValueString(byteArray()).substring(2);
	}

	public float floatValue() {
		return (float) data;
	}
}
