package net.wasamon.mics.freehdl.data;

import java.util.ArrayList;
import java.util.Iterator;

import net.wasamon.mjlib.util.DataUtil;

public class StdLogicVector extends Signal {

	public class VectorData {
		LogicData[] value;

		VectorData(int size) {
			value = new LogicData[size];
		}

		public int size() {
			return value.length;
		}

		public boolean equals(VectorData data) {
			if (data.size() != size())
				return false;
			for (int i = 0; i < size(); i++) {
				if (!(value[i] != null && data.value[i] != null && value[i]
						.equals(data.value[i]))) {
					return false;
				}
			}
			return true;
		}

		private String getValueAsFormat(String format, LogicData[] data,
				int offset, int length, boolean sign) {
			int v = 0;
			int pad = sign ? 1 : 0;
			for (int i = 0; i < 8 - length; i++) {
				v += (byte) (pad << (7 - i));
			}
			for (int i = 0, j = length - 1; i < length; i++, j--) {
				switch (data[i + offset].value) {
				case LogicData.POSITIVE:
					v += (byte) (1 << j);
					break;
				case LogicData.NEGATIVE:
					v += (byte) (0 << j);
					break;
				case LogicData.UNKNOWN:
					return "U";
				case LogicData.DONTCARE:
					return "X";
				}
			}
			v &= 0x00ff;
			return String.format(format, v);
		}

		private String getValueAsBinaryString() {
			StringBuffer s = new StringBuffer();
			for (LogicData d : value) {
				s.append(d.getValueAsString());
			}
			return s.toString();
		}

		private String getValueAsHEX(String format) {
			String s = "";
			int len = value.length / 8 + ((value.length % 8 > 0) ? 1 : 0);
			int pad = (8 - value.length % 8) % 8;
			if (pad > 0) {
				s += getValueAsFormat(format, value, 0, value.length % 8, false);
			}
			for (int i = 0; i < value.length / 8; i++) {
				s += getValueAsFormat(format, value, i * 8 + value.length % 8, 8, false);
			}
			return s;
		}

		private String getValueAsHEXString() {
			return getValueAsHEX("%02x");
		}

		private String getValueAsASCIIString() {
			return getValueAsHEX("%c");
		}

		public String getValueAsString(int type) {
			switch (type) {
			case BINARY:
				return getValueAsBinaryString();
			case HEX:
				return getValueAsHEXString();
			case ASCII:
				return getValueAsASCIIString();
			default:
				return getValueAsBinaryString();
			}
		}

	}

	ArrayList<VectorData> values;

	public StdLogicVector(String longname, String name, String v) {
		super(longname, name);
		values = new ArrayList<VectorData>();
		setValue(v);
	}

	public String getValueAsString() {
		return getValueAsString(values.size() - 1);
	}

	public String getValueAsString(int index) {
		if (index >= 0 && index < values.size()) {
			return values.get(index).getValueAsString(stringType);
		} else {
			return "-";
		}
	}

	private VectorData parse(String str) {
		String args[] = str.substring(1, str.length() - 1).split(",");
		VectorData v = new VectorData(args.length);
		for (int i = 0; i < args.length; i++) {
			v.value[i] = new LogicData(args[i].charAt(1));
		}
		return v;
	}

	public void setValue(String v) {
		VectorData data = parse(v);
		values.add(data);
	}

	public Iterator<VectorData> iterator() {
		return values.iterator();
	}

	public int size() {
		return values.size();
	}

	public byte[] getValue(){
		return DataUtil.toByteArray(values.get(values.size()-1).getValueAsHEXString(), 16);
	}
	
	public static void main(String[] args){
		String data = "('1','1','1','1','1','1','1','1')";
		StdLogicVector v = new StdLogicVector("test", "test", data);
		System.out.println(v.getValueAsString());
		v.stringType = Signal.HEX;
		System.out.println(v.getValueAsString());
		for(byte b: v.getValue()){
			System.out.println(b);
		}
		System.out.println(DataUtil.toBigEndianValueString(v.getValue()));
	}
	
}
