package net.wasamon.mics.freehdl;

import java.io.*;
import java.util.regex.*;

import net.wasamon.mics.freehdl.data.*;

public class FreeHDLProcess {

	private String name;
	private Process process;

	/** プロセスの標準出力 */
	private InputStream in;

	/** プロセスの標準入力 */
	private PrintStream out;

	private ProcessWaitThread thread;

	private Module top;

	public FreeHDLProcess() {
	}

	public FreeHDLProcess(String name) {
		this.name = name;
	}

	public void reset(String name) throws IOException {
		if (thread != null && thread.isRunning()) {
			stop();
		}
		this.name = name;
	}

	public Module getTopModule() {
		return top;
	}

	private boolean isPrompt(String s) {
		// 第二項は，
		// s.substring(s.length()-"\n> ".length()).equals("\n> "));
		// の定数項を計算した結果
		return (s.length() > 3 && s.substring(s.length() - 3).equals("\n> "));
	}

	private String read() throws IOException {
		byte b[];
		PrintStream pout = new PrintStream(out, true);
		StringBuffer buf = new StringBuffer();
		int len = 0;
		String s = "";
		while (!isPrompt(s)) {
			while ((len = in.available()) > 0) {
				b = new byte[len];
				in.read(b);
				s = new String(b, 0, len);
				//System.out.println(s);
			}
		}
		//
		// return s.substring(0, s.length()-" >".length());
		// の定数項を計算した結果
		return s.substring(0, s.length() - 2);
	}

	private void write(String command) {
		out.println(command);
	}

	private Module getModule(String longname, String s[]) {
		if (top == null) {
			top = new Module(null, ":" + s[1] + ":", s[1]);
		}
		Module m = top;
		for (int i = 2; i < s.length - 1; i++) {
			if (m.containsModule(s[i])) {
				m = m.getModule(s[i]);
			} else {
				m = m.newModule(longname, s[i]);
			}
		}
		return m;
	}

	private void init() throws IOException {
		write("s");
		String s = read();
		for (String var : s.split("\n")) {
			String[] args = var.split(" = ");
			String[] v = args[0].split(":");
			Module m = getModule(args[0], v);
			String name = v[v.length - 1];
			Signal signal = null;
			if (args[1].charAt(0) == '(') {
				signal = new StdLogicVector(args[0], name, args[1]);
			} else if (args[1].charAt(0) == '\'') {
				signal = new StdLogic(args[0], name, args[1]);
			} else {
				signal = new UserType(args[0], name, args[1]);
			}
			if (signal != null) {
				m.add(signal);
			}
		}
	}

	public void run() throws IOException {
		run(true);
	}

	public void run(boolean quiet) throws IOException {
		process = Runtime.getRuntime().exec(name);
		in = process.getInputStream();
		out = new PrintStream(process.getOutputStream(), true);
		thread = new ProcessWaitThread(process);
		thread.start();
		String s = read(); // 起動メッセージの読み捨て
		if (!quiet) {
			System.out.println(s);
		}
		init();
	}

	public void info() throws IOException {
		write("s");
		System.out.print(read());
	}

	private void updateValues(String result) {
		int len = top.getLongName().length();
		for (String var : result.split("\n")) {
			String[] args = var.substring(len).split(" = ");
			top.getSignal(args[0]).setValue(args[1]);
		}
	}

	private Pattern nextResultPattern = Pattern
			.compile("Simulation time = (\\d+) \\w+ \\+ (\\d+)d");

	public void step() throws IOException {
		String retval;
		int value = -1;
		do {
			write("next");
			retval = read().trim();
			Matcher m = nextResultPattern.matcher(retval);
			if (m.matches()) {
				value = Integer.parseInt(m.group(2));
			} else {
				System.err.println("Unexpected return value: " + retval);
				throw new IOException("Unexpected return value: " + retval);
			}
		} while (value != 0);
		write("show");
		updateValues(read());
	}

	public void stop() {
		write("quit");
		while (thread.isRunning()) {
			// System.out.println("wait...");
		}
	}

	public static void main(String args[]) throws Exception {
		FreeHDLProcess p = new FreeHDLProcess(args[0]);
		p.run(false);
		for (int i = 0; i < 100; i++) {
			p.step();
		}
		p.stop();
	}

	class ProcessWaitThread extends Thread {

		private Process process;
		private boolean flag;

		public ProcessWaitThread(Process process) {
			this.process = process;
			this.flag = true;
		}

		public void run() {
			flag = true;
			try {
				process.waitFor();
			} catch (InterruptedException e) {
				flag = false;
			}
			flag = false;
		}

		/**
		 * @return プロセスが現在動作中かどうか返す． 真ならプロセスが走っている．
		 */
		public boolean isRunning() {
			return flag;
		}
	}

}
