/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * 通信路に相当するシミュレーション要素のパッケージ
 */
package net.wasamon.mics.bus;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * 複数のChannelConnectableからのアクセスを排他的に処理する単一のバス。
 * 同じサイクルで複数のリクエストを受けた場合、どれかひとつのリクエストのみが実行される。
 * 
 * @author Takefumi MIYOSHI
 *
 */
public class SingleDataBus extends MicsElement implements Channel,
		ExecutableElement {

	private String dest;

	private ChannelConnectable src;
	private MicsDataPacket reqData;

	private int mode;

	private int number;

	private static final int NONE = 0;

	private static final int READ = 1;

	private static final int WRITE = 2;

	public SingleDataBus() {
	}

	public String getImagePath() {
		return "combo_bus3.png";
	}

	public void initialize(String base, Node n) throws MicsException {
		try {
			{
				Node init_var_node = n;
				String init_var_ram;
				init_var_ram = XMLParser.getAttribute(init_var_node, "ram")
						.getNodeValue();

				this.dest = init_var_ram;

			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.bus.SingleDataBus16");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.bus.SingleDataBus16");
		}
	}

	public void reset() {
		this.number = 0;
		this.mode = NONE;
	}

	public void readRequest(ChannelConnectable src, MicsDataPacket data) {
		this.src = src;
		this.reqData = data;
		mode = READ;
		number++;
	}

	public void writeRequest(ChannelConnectable src, MicsDataPacket data) {
		this.src = src;
		this.reqData = data;
		mode = WRITE;
		number++;
	}

	public ExecInfo exec_first() throws MicsException {
		return null;
	}

	private DataBuffer destDataBufferInstance;

	private DataBuffer destDataBuffer() throws MicsException {
		if (destDataBufferInstance == null) {
			destDataBufferInstance = composite.getDataBuffer(dest);
		}
		return destDataBufferInstance;
	}

	public ExecInfo exec_second() throws MicsException {
		if (number > 1)
			System.out.println("bus conflict: transfer " + dest + "<->" + src.id());
		switch (mode) {
		case WRITE:
			// System.out.println("write: " + reqData);
			destDataBuffer().write(reqData);
			break;
		case READ:
			src.writeback(this, destDataBuffer().read(reqData));
			break;
		case NONE:
			break;
		default:
			break;
		}
		number = 0;
		this.src = null;
		mode = NONE;
		return new ExecInfo();
	}

	public String getInfo() {
		String s = "";
		s += "SingleDataBus16\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  Dest Memory ID: " + dest;
		return s;
	}

	public String[] getConnectedElements() {
		return new String[] { dest };
	}


	public String getDescription(){
		return "TODO";
	}
}
