/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.net;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;
import java.net.UnknownHostException;

import net.wasamon.mjlib.util.GetOpt;
import net.wasamon.mjlib.util.GetOptException;

/**
 * シンプルなTelnetクライアントの実装
 * @author Takefumi MIYOSHI
 */
public class TelnetClient{

  public static final String version = "TelnetClient 0.1";
  public static final String author = "Takefumi MIYOSHI (miyo@wasamon.net)";
  public static final String copyright = "(c) 2003/11/04 All right reserved.";
  
  private static final byte[] cr_b = {(byte)0x0d};
  private static final byte[] lf_b = {(byte)0x0a};

  private static final String CR = new String(cr_b);
  private static final String LF = new String(lf_b);

  private static final String CRLF = new String(CR+LF);
  
  /** デフォルトのポート番号 */
  private static final int DEFALTPORT = 23;
  
  /** このTelnetClientの使用するソケット */
  private Socket socket;

  private BufferedOutputStream out;

  private BufferedInputStream in;

  /** ホスト名 */
  private String host;

  /** ポート番号 */
  private int port;

  /**
   * コンストラクタ
   * @param host ホスト名
   * @param port 接続するポート番号
   * @exception NetworkException ネットワークに関する何らかの例外
   */
  public TelnetClient(String host, int port) throws NetworkException{
    this.host = host;
    this.port = port;
    try{
      socket = new Socket(host, port);
      in = new BufferedInputStream(socket.getInputStream());
      out = new BufferedOutputStream(socket.getOutputStream());
    }catch(UnknownHostException e){
      throw new NetworkException(e);
    }catch(IOException e){
      throw new NetworkException(e);
    }
  }

  public boolean isConnected(){
    return socket.isConnected();
  }

  public void close(){
    try{
      socket.close();
    }catch (IOException e){
      System.out.println("cannot close socket.");
    }
    return;
  }

  /** 
   * サーバへメッセージを送信し、受信したメッセージを返す。
   * ただしメッセージには、改行コード(CR+LF)が一つ付加される。
   * @param smesg 送るメッセージ
   * @return 帰ってきたメッセージ
   * @exception タイムアウトまでサーバから返事がない場合
   */
  public String sendln(String smesg) throws NetworkException{
    return send(smesg+CR+LF);
  }
  /** 
   * サーバへメッセージを送信し、受信したメッセージを返す
   * @param smesg 送るメッセージ
   * @return 帰ってきたメッセージ
   * @exception タイムアウトまでサーバから返事がない場合
   */
  public String send(String smesg) throws NetworkException{
    StringBuffer str = new StringBuffer();

    PrintWriter pw = new PrintWriter(out, true);
    pw.print(smesg);
    if(pw.checkError()){
      throw new NetworkException("cannot send message");
    }
    BufferedReader reader = new BufferedReader(new InputStreamReader(in));
    try{
      String s;
      while((s = reader.readLine()) != null){
	str.append(s+"\n");
      }
    }catch(IOException e){
      throw new NetworkException(e);
    }
    return str.toString();
  }

  private static void usage(){
    System.out.println(version);
    System.out.println(author);
    System.out.println(copyright);
    System.exit(0);
  }

  public static void main(String args[]) throws NetworkException, GetOptException, IOException{
    GetOpt opt = new GetOpt("p:v", "port:,version", args);
    int port = DEFALTPORT;
    String host;

    if(opt.flag("p")){
      port = Integer.decode(opt.getValue("p")).intValue();
    }
    if(opt.flag("port")){
      port = Integer.decode(opt.getValue("port")).intValue();
    }
    if(opt.flag("v")){
      usage();
    }
      
    String[] rarg = opt.getArgs();
    if(rarg.length == 0){
      usage();
    }

    host = opt.getArgs()[0];

    TelnetClient tc = new TelnetClient(host, port);

    BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
    String s;

    while(tc.isConnected()){
      while(true){
	System.out.print(">");
	s  = in.readLine();
	if(s == null){ tc.close(); break; }
	System.out.println(tc.sendln(s));
      }
    }

    System.out.println("connection closed.");
    System.exit(0);
  }

}

