/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.net;

/**
 * パケットデータを扱うためのユーティリティ.
 * バイト列を扱うためのユーティリティも含む.
 * 
 * @author Takefumi MIYOSHI
 */
public class PacketUtil{

  public static final String version = "PacketUtil 0.1";
  public static final String author = "Takefumi MIYOSHI (miyoshi@ae.titech.ac.jp)";
  public static final String copyright = "(c) 2003/06/27 All right reserved.";

  /**
   * 二つのバイト配列が等しいかどうか判定する
   * @param d1 データ
   * @param d2 データ
   * @return 等しいかどうかの真偽
   */
  public static boolean equals(byte[] d1, byte[] d2){
    if(d1.length != d2.length ){
      return false;
    }
    for(int i = 0; i < d1.length; i++){
      if(d1[i] != d2[i]) return false;
    }
    return true;
  }


  /**
   * バイト列の指定するoffsetから指定する長さの部分バイト列を新しく生成し返す.
   * ただし、ArrayIndexBoundExceptionが発生しない範囲でデータを生成する
   * @param data 元のバイト列
   * @param offset オフセット
   * @param len 長さ
   * @return 新たに生成した部分バイト列.
   */
  public static byte[] subArray(byte data[], int offset, int len){

    if(data.length < offset+len){
      len = data.length - offset;
    }

    byte[] r = new byte[len];

    for(int i=0 ; i<len ; ++i){
      r[i] = data[offset+i];
    }
    return r;
  }


  /**
   * ネットワークオーダの値をlong型の値に変換する
   * @param data もとのバイト列
   * @param offset オフセット
   * @return longの値
   */
  public static long ntoh_long(byte data[], int offset){
    return  ((((long)data[offset+0]) << 56 ) & 0x00ff00000000000000L)
           +((((long)data[offset+1]) << 48 ) & 0x0000ff000000000000L)
           +((((long)data[offset+2]) << 40 ) & 0x000000ff0000000000L)
           +((((long)data[offset+3]) << 32 ) & 0x00000000ff00000000L)
           +((((long)data[offset+4]) << 24 ) & 0x0000000000ff000000L)
           +((((long)data[offset+5]) << 16 ) & 0x000000000000ff0000L)
           +((((long)data[offset+6]) << 8  ) & 0x00000000000000ff00L)
           +((((long)data[offset+7])       ) & 0x0000000000000000ffL);
  }

  /**
   * ネットワークオーダの値をlong型の値に変換する
   * @param data もとのバイト列
   * @return longの値
   */
  public static long ntoh_long(byte data[]){
    return ntoh_long(data, 0);
  }

  /**
   * ネットワークオーダの値をint型の値に変換する
   * @param data もとのバイト列
   * @param offset オフセット
   * @return intの値
   */
  public static int ntoh_int(byte data[], int offset){
    return (int)((int)(data[offset  ] << 24 )
		 +((int)(data[offset+1] << 16 ) & 0x00ff0000 )
		 +((int)(data[offset+2] << 8  ) & 0x0000ff00 )
		 +((int)(data[offset+3]       ) & 0x000000ff ) );
  }

  /**
   * ネットワークオーダの値をint型の値に変換する
   * @param data もとのバイト列
   * @return intの値
   */
  public static int ntoh_int(byte data[]){
    return ntoh_int(data, 0);
  }


  /**
   * ネットワークオーダの値をint型の値に変換する
   * @param data もとのバイト列
   * @param offset オフセット
   * @return intの値
   */
  public static short ntoh_short(byte data[], int offset){
    return (short)(((int)(data[offset] << 8 ) & 0xff00)
		   +((int)(data[offset+1])    & 0x00ff));
  }

  /**
   * ネットワークオーダの値をint型の値に変換する
   * @param data もとのバイト列
   * @return intの値
   */
  public static short ntoh_short(byte data[]){
    return ntoh_short(data, 0);
  }


  /**
   * long型の値をネットワークオーダに変換する
   * @param data 生成したバイト列を格納する配列
   * @param offset オフセット
   * @param value 変換したい値
   * @return intの値
   */
  public static byte[] hton_long(byte data[], int offset, long value){
    data[offset+0] = (byte)((value & 0xff00000000000000L) >> 56);
    data[offset+1] = (byte)((value & 0x00ff000000000000L) >> 48);
    data[offset+2] = (byte)((value & 0x0000ff0000000000L) >> 40);
    data[offset+3] = (byte)((value & 0x000000ff00000000L) >> 32);
    data[offset+4] = (byte)((value & 0x00000000ff000000L) >> 24);
    data[offset+5] = (byte)((value & 0x0000000000ff0000L) >> 16);
    data[offset+6] = (byte)((value & 0x000000000000ff00L) >> 8);
    data[offset+7] = (byte)((value & 0x00000000000000ffL));
    return data;
  }

  /**
   * int型の値をネットワークオーダに変換する
   * @param data 生成したバイト列を格納する配列
   * @param offset オフセット
   * @param value 変換したい値
   * @return intの値
   */
  public static byte[] hton_int(byte data[], int offset, int value){
    data[offset+0] = (byte)((value & 0xff000000) >> 24);
    data[offset+1] = (byte)((value & 0x00ff0000) >> 16);
    data[offset+2] = (byte)((value & 0x0000ff00) >> 8);
    data[offset+3] = (byte)((value & 0x000000ff));
    return data;
  }

  /**
   * short型の値をネットワークオーダに変換する
   * @param data 生成したバイト列を格納する配列
   * @param offset オフセット
   * @param value 変換したい値
   * @return intの値
   */
  public static byte[] hton_short(byte data[], int offset, short value){
    data[offset+0] = (byte)((value & 0xff00) >> 8);
    data[offset+1] = (byte)((value & 0x00ff));
    return data;
  }

  /**
   * 二つのバイト列をマージし、返す
   * @param dest マージしたデータを格納する配列
   * @param offset 格納するオフセット
   * @param src1 ソースバイト列
   * @param src2 ソースバイト列
   * @return マージ後の配列への参照
   */
  public static byte[] merge(byte dest[], int offset, byte src1[], byte src2[]){
    for(int i = 0; i< src1.length; i++){
      dest[i+offset] = src1[i];
    }
    for(int i = 0; i< src2.length; i++){
      dest[i+offset+src1.length] = src2[i];
    }
    return dest;
  }

  /**
   * 二つのバイト列をマージし、返す
   * @param dest マージしたデータを格納する配列
   * @param src1 ソースバイト列
   * @param src2 ソースバイト列
   * @return マージ後の配列への参照
   */
  public static byte[] merge(byte dest[], byte src1[], byte src2[]){
    return merge(dest, 0, src1, src2);
  }

  /**
   * 二つのバイト列をマージし、新しいバイト列として返す
   * @param src1 ソースバイト列
   * @param src2 ソースバイト列
   * @return マージ後の新しい配列
   */
  public static byte[] concat(byte src1[], byte src2[]){
    byte[] dest = new byte[src1.length + src2.length];
    for(int i = 0; i < src1.length; i++){
      dest[i] = src1[i];
    }
    for(int i = 0; i < src2.length; i++){
      dest[i+src1.length] = src2[i];
    }
    return dest;
  }

  public static void main(String args[]){
    System.out.println(version);
    System.out.println(author);
    System.out.println(copyright);
  }

}
