/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.net;

import java.util.Calendar;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.TimeZone;

/**
 * HTTPを扱う上でのユーティリティクラス.
 *
 * @author Takefumi MIYOSHI
 */
public class HttpUtil{

  public static final String version = "HttpUtil 0.1";
  public static final String author = "Takefumi MIYOSHI (miyoshi@ae.titech.ac.jp)";
  public static final String copyright = "(c) 2003/06/27 All right reserved.";

  /**
   * HTTPヘッダにおけるコンテンツ依存部分を生成し返す<br>
   * (例)<br>
   * Last-Modified: Tue, 19 Mar 2002 02:34:24 GMT<br>
   * Content-Length: 1892<br>
   * Content-Type: text/html<br>
   * @param time 時刻
   * @param length ファイルの長さ
   * @param type ファイルの種類
   * @return ContentHeaderの文字列
   */
  public static String getHTTPContentHeader(long time, long length, String type){
    /*
      HTTP/1.0 200 OK
      Date: Tue, 26 Nov 2002 08:08:19 GMT
      Server: Apache/1.3.26 (Unix)
      ETag: "21a514-764-3c96a3b0"
      Accept-Ranges: bytes
    */
    String header = new String();
    byte cr[] = new byte[1]; cr[0] = (byte)0x0d;
    byte lf[] = new byte[1]; lf[0] = (byte)0x0a;
    
    String CR = new String(cr);
    String LF = new String(lf);

    header = header.concat("Last-Modified: ");
    header = header.concat(getDateString(time));
    header = header.concat(CR);
    header = header.concat(LF);
    header = header.concat("Content-Type: ");
    header = header.concat(type);
    header = header.concat(CR);
    header = header.concat(LF);
    header = header.concat("Content-Length: ");
    header = header.concat(Long.toString(length));
    return header;
  }

  /**
   * 1970からの秒で指定された時間からRFC規定のHTTP用時刻文字列を生成する<br>
   * (例)<br>
   * Tue, 19 Mar 2002 02:34:24 GMT
   * @param time 時刻
   *
   */
  public static String getDateString(long time){
  
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    cal.setTimeInMillis(time);
    String year = Integer.toString(cal.get(Calendar.YEAR));
    String date = Integer.toString(cal.get(Calendar.DATE));
    String hour = Integer.toString(cal.get(Calendar.HOUR_OF_DAY));
    String min  = Integer.toString(cal.get(Calendar.MINUTE));
    String sec  = Integer.toString(cal.get(Calendar.SECOND));
    int month   = cal.get(Calendar.MONTH);
    int day     = cal.get(Calendar.DAY_OF_WEEK);
    
    if( hour.length() == 1 ){
      hour = "0"+hour;
    }
    if( min.length() == 1 ){
      min = "0"+min;
    }
    if( sec.length() == 1 ){
      sec = "0"+sec;
    }

    String str = getDayString(day) + ", " + date + " " + getMonthString(month) + " " + year + " " + hour + ":" + min + ":" + sec + " " + "GMT";

    return str;
  }

  /**
   * 曜日の文字列表記を返す
   */
  private static String getDayString(int day){
    switch (day){
    case Calendar.SUNDAY    : return "Sun";
    case Calendar.MONDAY    : return "Mon";
    case Calendar.TUESDAY   : return "Tue";
    case Calendar.WEDNESDAY : return "Wed";
    case Calendar.THURSDAY  : return "Thu";
    case Calendar.FRIDAY    : return "Fri";
    case Calendar.SATURDAY  : return "Sat";
    }
    return "";
  }


  /**
   * 指定した時刻が今年であるかどうかを判定する
   * @param time 1970/1/1からの秒
   * @return 今年かどうか
   */
  public static boolean isThisYear(long time){
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    int current = cal.get(Calendar.YEAR); // 現在の年を取得
    cal.setTimeInMillis(time); // カレンダーを指定した時刻にあわせる
    int target = cal.get(Calendar.YEAR); // 指定した時刻の年

    if (current == target){
      return true;
    }
    return false;
  }


  /**
   * 1970/1/1からの秒で指定された時間から月名を得る
   * @param  time 1970/1/1からの秒
   * @return 月の名前を示す文字列
   */
  public static String getYear(long time){
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    cal.setTimeInMillis(time);
    int year = cal.get(Calendar.YEAR);
    return Integer.toString(year);
  }

  /**
   * 1970/1/1からの秒で指定された時間から月名を得る
   * @param  time 1970/1/1からの秒
   * @return 月の名前を示す文字列
   */
  public static String getMonth(long time){
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    cal.setTimeInMillis(time);
    int month = cal.get(Calendar.MONTH);
    return getMonthString(month);
  }

  /**
   * 1970/1/1からの秒で指定された時間から日付を得る
   * @param  time 1970/1/1からの秒
   * @return 日付を示す文字列
   */
  public static String getDate(long time){
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    cal.setTimeInMillis(time);
    String date = Integer.toString(cal.get(Calendar.DATE));
    if(date.length() == 1){
      date = " " + date;
    }
    return date;
  }

  /**
   * 1970/1/1からの秒で指定された時間から時刻を得る
   * @param  time 1970/1/1からの秒
   * @return 時刻を示す文字列
   */
  public static String getTime(long time){
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    cal.setTimeInMillis(time);
    String hour = Integer.toString(cal.get(Calendar.HOUR_OF_DAY));
    String min  = Integer.toString(cal.get(Calendar.MINUTE));
    if( hour.length() == 1 ){
      hour = "0"+hour;
    }
    if( min.length() == 1 ){
      min = "0"+min;
    }
    return hour + ":" + min;
  }

  /**
   * 月名の文字列表記を返す
   */
  private static String getMonthString(int month){
    switch(month){
    case Calendar.JANUARY: return "Jan";
    case Calendar.FEBRUARY:return "Feb";
    case Calendar.MARCH: return "Mar";
    case Calendar.APRIL: return "Apr";
    case Calendar.MAY: return "May";
    case Calendar.JUNE: return "Jun";
    case Calendar.JULY: return "Jul";
    case Calendar.AUGUST: return "Aug";
    case Calendar.SEPTEMBER: return "Sep";
    case Calendar.OCTOBER: return "Oct";
    case Calendar.NOVEMBER: return "Nov";
    case Calendar.DECEMBER: return "Dec";
    }
    return "";
  }

  /**
   * ヘッダ文字列から時刻を切りだす.
   * @param header HTTPヘッダ
   * @return 1970/1/1からの秒で示される時間.long値
   */
  public static long getTime(String header) throws HttpHeaderException{
    String lm = "Last-Modified: ";
    StringTokenizer st = new StringTokenizer(header, "\n");
    String[] inputs = new String[st.countTokens()];
    for(int i = 0; st.hasMoreElements(); i++){
      inputs[i] = st.nextToken();
    }
    for(int i = 0; i < inputs.length; i++){
      if(inputs[i].startsWith(lm)){
	return Long.parseLong(inputs[i].substring(lm.length(), inputs[i].length()-1));
      }
    }
    throw new HttpHeaderException("not found Last-Modified elemnt.");
  }

  /**
   * ヘッダ文字列からファイルの種類を切りだす.
   * @param header HTTPヘッダ
   * @return content-typeを表わす文字列
   */
  public static String getType(String header) throws HttpHeaderException{
    String ct = "Content-Type: ";
    StringTokenizer st = new StringTokenizer(header, ".");
    String[] inputs = new String[st.countTokens()];
    for(int i = 0; st.hasMoreElements(); i++){
      inputs[i] = st.nextToken();
    }
    for(int i = 0; i < inputs.length; i++){
      if(inputs[i].startsWith(ct)){
	return inputs[i].substring(ct.length(), inputs[i].length()-1);
      }
    }
    throw new HttpHeaderException("not found Content-Type elemnt.");
  }

  /**
   * ヘッダ文字列からファイルの長さを切りだす.
   * @param header HTTPヘッダ
   * @return ファイル長を示すlong値
   */
  public static long getLength(String header) throws HttpHeaderException{
    String ct = "Content-Length: ";
    StringTokenizer st = new StringTokenizer(header, ".");
    String[] inputs = new String[st.countTokens()];
    for(int i = 0; st.hasMoreElements(); i++){
      inputs[i] = st.nextToken();
    }
    for(int i = 0; i < inputs.length; i++){
      if(inputs[i].startsWith(ct)){
	return Long.parseLong(inputs[i].substring(ct.length(), inputs[i].length()-1));
      }
    }
    throw new HttpHeaderException("not found Content-Length elemnt.");
  }

  public static void main(String args[]){
    System.out.println(version);
    System.out.println(author);
    System.out.println(copyright);
    System.out.println(getDateString((new Date()).getTime()));
  }


}
