/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * AWTを用いてGUIを実装したパッケージ
 */
package net.wasamon.mics.gui.awt;

import java.awt.BorderLayout;
import java.awt.Button;
import java.awt.Color;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.MenuShortcut;
import java.awt.Panel;
import java.awt.TextArea;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintStream;

import net.wasamon.mics.Mics;
import net.wasamon.mics.MicsCompositeElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsPrompt;
import net.wasamon.mics.MicsShell;
import net.wasamon.mics.gui.MicsArchitectureViewer;
import net.wasamon.mics.tools.AsciiToBinary;
import net.wasamon.mjlib.ui.AWTOptionPane;
import net.wasamon.mjlib.ui.TextAreaOutputStream;

public class MicsAWTWindow implements MicsShell, ActionListener {

  private static MicsAWTWindow instance = new MicsAWTWindow();

  private Frame frame;

  private TextArea log;

  private TextField input;

  public static MicsAWTWindow getInstance() {
    return instance;
  }

  private void exit(int stat) {
    Mics.getInstance().shutdown(stat);
  }

  private MicsAWTWindow() {
    frame = new Frame("Mics " + Mics.printVersion());
    frame.setSize(640, 540);
    frame.addWindowListener(new WindowAdapter() {
	public void windowClosing(WindowEvent e) {
	  exit(0);
	}
      });
    frame.setMenuBar(getMenuBar());
    frame.add(getToolBar(), BorderLayout.NORTH);
    log = new TextArea();
    log.setFont(new Font("MonoSpaced", Font.PLAIN, 12));
    log.setEditable(false);
    log.setBackground(Color.WHITE);
    frame.add(log, BorderLayout.CENTER);

    PrintStream stream = new PrintStream(new TextAreaOutputStream(log), true);
    System.setOut(stream);
    System.setErr(stream);

    TextField prompt = new TextField("[command]");
    prompt.setEditable(false);
    input = new TextField();
    input.addActionListener(this);
    Panel p0 = new Panel(new BorderLayout());
    p0.add(prompt, BorderLayout.WEST);
    p0.add(input);
    frame.add(p0, BorderLayout.SOUTH);
    frame.setVisible(false);
  }

  private Button btnStep;

  private Button btnRun;

  private Button btnStop;

  private Button btnReset;

  private Button btnLoad;

  private TextField stepNumber;

  private Panel getToolBar() {
    Panel bar = new Panel();
    {
      Panel p = new Panel();
      p.setLayout(new GridLayout(1, 5));
      btnReset = new Button("Reset");
      btnReset.addActionListener(this);
      btnStep = new Button("Step");
      btnStep.addActionListener(this);
      stepNumber = new TextField("1", 3);
      stepNumber.addActionListener(this);
      btnRun = new Button("Run");
      btnRun.addActionListener(this);
      btnStop = new Button("Stop");
      btnStop.addActionListener(this);
      p.add(btnReset);
      p.add(btnRun);
      p.add(btnStop);
      p.add(btnStep);
      p.add(stepNumber);
      //			p.setBorder(new TitledBorder(new LineBorder(Color.gray, 1), "processor",
      //					TitledBorder.CENTER, TitledBorder.TOP));
      setMicsEngineEnableState(true);
      bar.add(p);
    }
    return bar;
  }

  private void setLogField(TextArea a) {
    log = a;
  }

  private MenuItem openMenuItem;
	
  private MenuItem viewMenuItem;

  private MenuItem exitMenuItem;

  private MenuItem a2bMenuItem;

  private MenuBar getMenuBar() {
    MenuBar bar = new MenuBar();
    {
      Menu m = new Menu("File");
      exitMenuItem = new MenuItem("Exit", new MenuShortcut('X'));
      exitMenuItem.addActionListener(this);
      m.add(exitMenuItem);
      bar.add(m);
    }
    {
      Menu m = new Menu("Architecture");
      openMenuItem = new MenuItem("Load");
      openMenuItem.addActionListener(this);
      m.add(openMenuItem);
      viewMenuItem = new MenuItem("View");
      viewMenuItem.addActionListener(this);
      m.add(viewMenuItem);
      bar.add(m);
    }
    {
      Menu m = new Menu("Tools");
      a2bMenuItem = new MenuItem("Ascii to Binary");
      a2bMenuItem.addActionListener(this);
      m.add(a2bMenuItem);
      bar.add(m);
    }
    return bar;
  }

  public void exec() {
    frame.setVisible(true);
  }

  /**
   * バッチ処理といっても、特に何もしない
   */
  public void exec(File batch) {
    exec();
  }

  public void setMicsEngineEnableState(boolean flag) {
    if (flag) {
      btnStep.setEnabled(true);
      btnReset.setEnabled(true);
      btnRun.setEnabled(true);
      btnStop.setEnabled(false);
    } else {
      btnStep.setEnabled(false);
      btnReset.setEnabled(false);
      btnRun.setEnabled(false);
      btnStop.setEnabled(true);
    }
  }

  public File openFile(String title, FilenameFilter filter) {
    File f = null;
    FileDialog fd = new FileDialog(frame, title, FileDialog.LOAD);
    fd.setFilenameFilter(filter);
    fd.setVisible(true);
    if (fd.getFile() != null) {
      f = new File(fd.getDirectory(), fd.getFile());
    }
    return f;
  }

  class MicsConfigFileFilter implements FilenameFilter {
    public boolean accept(File dir, String name) {
      if (name.endsWith(".xml")) {
	return true;
      } else {
	return false;
      }
    }
  }

  class SimpleProcessorAssemblerFilter implements FilenameFilter {
    public boolean accept(File dir, String name) {
      if (name.endsWith(".s")) {
	return true;
      }
      return false;
    }
  }

  class CoinsInputFilter implements FilenameFilter {
    public boolean accept(File dir, String name) {
      if (name.endsWith(".c")) {
	return true;
      }
      if (name.endsWith(".i")) {
	return true;
      }
      if (name.endsWith(".lir")) {
	return true;
      }
      return false;
    }
  }

  class ConfigDataSourceFilter implements FilenameFilter {
    public boolean accept(File dir, String name) {
      if (name.endsWith(".context")) {
	return true;
      }
      return false;
    }
  }

  public void actionPerformed(ActionEvent ev) {
    Object obj = ev.getSource();
    if (obj == input) {
      try {
	MicsPrompt.execLine(((TextField) obj).getText());
      } catch (IOException e) {
      }
      ((TextField) obj).setText("");
    } else if (obj == openMenuItem || obj == btnLoad) {
      File f = openFile("Open Config File", new MicsConfigFileFilter());
      if (f != null) {
	try {
	  Mics.getInstance().configTargetArchitecture(f);
	} catch (Exception e) {
	  AWTOptionPane.showErrorDialog(frame, "configuration error", e + ":" + e.getMessage());
	  e.printStackTrace();
	}
	Mics.getInstance().engine().show();
      }
    } else if (obj == exitMenuItem) {
      exit(0);
    } else if (obj == a2bMenuItem) {
      File f = openFile("Open", null);
      if (f != null) {
	try {
	  AsciiToBinary asm = new AsciiToBinary();
	  File dest = asm.parser(f);
	  System.out.println("compile succeeded: " + dest.getAbsolutePath());
	} catch (NumberFormatException e) {
	  System.out.println(e.getMessage());
	} catch (FileNotFoundException e) {
	  System.out.println("no such file: " + f.getAbsolutePath());
	} catch (IOException e) {
	  System.out.println("File I/O error");
	}
      }
    } else if (obj == btnRun) {
      setMicsEngineEnableState(false);
      Mics.getInstance().execEngine();
    } else if (obj == btnStop) {
      Mics.getInstance().stopEngine();
      setMicsEngineEnableState(true);
    } else if (obj == btnStep) {
      try {
	int count = 1;
	try {
	  count = Integer.parseInt(stepNumber.getText());
	} catch (NumberFormatException e) {
	  count = 1;
	  stepNumber.setText("1");
	}
	if (count < 1) {
	  count = 1;
	  stepNumber.setText("1");
	}
	Mics.getInstance().execStepByStep(count);
      } catch (MicsException e) {
	System.out.println(e);
      }
    } else if (obj == btnReset) {
      try {
	Mics.getInstance().reset();
      } catch (MicsException e) {
	System.out.println(e);
      }
    } else if (obj == viewMenuItem) {
      if(Mics.getInstance().engine() != null){
	Mics.getInstance().engine().show();
      }
    } else {
      System.out.println("Uninitialized as architecture config.");
    }
  }

  public MicsArchitectureViewer getMicsArchitectureViewerData(MicsCompositeElement composite){
    return new AWTArchitectureViewer(composite);
  }
	
}
