/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */


/**
 * MICSのエンジンおよびMICSで規定するインターフェースの定義などのパッケージ
 */
package net.wasamon.mics;

import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * Micsシミュレーション要素が実装すべき基底クラス
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public abstract class MicsElement implements IMicsElement {

	private String _id;
	private String _base;
	private Node _node;

	protected MicsCompositeElement composite;
	
	/**
	 * このシミュレータ要素の持つ装飾情報を保持するクラス
	 * @author miyo
	 *
	 */
	public class Appearance{
		public final Integer x;
		public final Integer y;
		public final Integer width;
		public final Integer height;
		public final String image;
		
		Appearance(String b, Node node){
			String str;
			if((str = getAttributeAsString(node, "image")) != null){
				image = b + "/" + str;
			}else{
				image = MicsElement.this.getImagePath();
			}
			x = getAttributeAsInt(node, "x");
			y = getAttributeAsInt(node, "y");
			width = getAttributeAsInt(node, "width");
			height = getAttributeAsInt(node, "height");
		}
	}
	
	private Appearance appearance;
	
	/**
	 * このシミュレータ要素の持つ装飾情報のインスタンスを返す
	 */
	public Appearance getAppearance(){
		return appearance;
	}
	
	/**
	 * @param base
	 *          設定ファイルへのディレクトリ名
	 * 
	 * @param id
	 *          このエレメントのID
	 * @param node
	 *          このエレメントを設定するためのノード
	 */
	public void initialize_base(MicsCompositeElement composite, String base,
			String id, Node node) throws MicsException {
		this._id = id;
		this._base = base;
		this._node = node;
		this.composite = composite;
		this.appearance = new Appearance(base, getNamedNode(node, "appearance"));
		initialize(base, node); // 2008.09.04 すべての要素のインスタンス化が終わってから呼ぶように修正した
	}

	public void initialize() throws MicsException{
		initialize(_base, _node);
	}
	
	public abstract void initialize(String base, Node node) throws MicsException;

	/**
	 * このシミュレーション要素の情報を示す文字列を返す。
	 * 
	 * @return
	 */
	public String getInfo() {
		String str = "";
		str += "id: " + id() + "\n";
		str += "class: " + this.getClass().getName() + "\n";
		str += getDescription() + "\n";
		return str;
	}
	
	abstract public String getDescription();

	/**
	 * 継承するクラスでは、このメソッドをオーバーライドしてはならない
	 */
	public String id() {
		return this._id;
	}

	public void shutdown() {

	}

	public abstract String[] getConnectedElements();

	/**
	 * シミュレーション要素において、独自に有するイメージファイルがある場合、そのパスを返す。
	 * @return
	 */
	abstract public String getImagePath();

	/**
	 * 指定した名前の要素の配列を返す。
	 * 
	 * @param node
	 * @param key
	 * @return 指定した名前に対応する要素。要素がないにはnullを返す。
	 */
	protected Node[] getNamedNodeArray(Node node, String name) {
		return XMLParser.getNamedNodeArray(node, name);
	}

	/**
	 * 指定した名前のノードを返す。
	 * 
	 * @param node
	 * @param key
	 * @return 指定した名前に対応する要素。ノードがない場合にはnullを返す。
	 */
	protected Node getNamedNode(Node node, String name) {
		try{
			return XMLParser.getNamedNode(node, name);
		}catch(XMLParserException e){
			return null;
		}
	}

	/**
	 * 指定したkeyに対応する値を文字列として返す
	 * 
	 * @param node
	 * @param key
	 * @return 指定したkeyに対応する文字列。keyがない場合などエラー時にはnullを返す
	 */
	protected String getAttributeAsString(Node node, String key) {
		if(node == null) return null;
		try {
			return XMLParser.getAttribute(node, key).getNodeValue();
		} catch (XMLParserException e) {
			return null;
		}

	}

	/**
	 * 指定したkeyに対応する値を数値として返す
	 * 
	 * @param node
	 * @param key
	 * @return 指定したkeyに対応する値を持つIntegerのインスタンス。keyがない場合などエラー時にはnullを返す
	 */
	protected Integer getAttributeAsInt(Node node, String key) {
		if(node == null) return null;
		try {
			return new Integer(DataUtil.parseInt(XMLParser.getAttribute(node, key).getNodeValue()));
		} catch (XMLParserException e) {
			return null;
		}

	}
	
	protected String toAbsolutePath(String s){
		String s0 = s;
		if (s.charAt(0) != '/') {
			s0 = _base + "/" + s;
		}
		return s0;
	}


}
