/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker;

import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import net.wasamon.mics.architecturemaker.unit.DataBuffer;
import net.wasamon.mics.architecturemaker.unit.HardwareUnit;
import net.wasamon.mics.architecturemaker.unit.InitAttribute;
import net.wasamon.mics.architecturemaker.unit.Mediator;
import net.wasamon.mics.architecturemaker.unit.MicsNode;

@SuppressWarnings("serial")
/**
 * @author Masayuki Morisita
 */
public class VirtualCanvas implements ActionListener {
	private static final int NONE_MODE = 0;
	private static final int CONNECTION_MODE = 1;
	private static final int SELECTION_MODE = 2;
	private static final int MOVEMENT_MODE = 3;

	private Point actionFrom, actionTo;
	private Dimension canvasSize;
	private HauntedMansion mansion;
	private int currentKey, pitch, actionMode, scale;
	private Mediator mediator;

    public VirtualCanvas(int width, int height) {
		currentKey = KeyEvent.VK_0;
		pitch = 25;
		actionMode = NONE_MODE;
		scale = 100;
		canvasSize = new Dimension(width, height);
		mediator = Mediator.getInstance();
	}

    public int getScale(){
	return scale;
    }

    public void setScale(int scale){
	this.scale = scale;
    }

    public int getWidth(){
	return canvasSize.width;
    }

    public int getHeight(){
	return canvasSize.height;
    }

    public void showPopup(JComponent canvas, Point mousePosition) {
		JMenuItem addInit = new JMenuItem("Add Init File");
		JMenuItem cut = new JMenuItem("Cut");
		JMenuItem copy = new JMenuItem("Copy");
		JMenuItem paste = new JMenuItem("Paste");
		JMenuItem delete = new JMenuItem("Delete");
		JMenuItem select = new JMenuItem("Select All");
		addInit.addActionListener(this);
		cut.addActionListener(this);
		copy.addActionListener(this);
		paste.addActionListener(this);
		delete.addActionListener(this);
		select.addActionListener(this);

		HardwareUnit[] selectedUnits = mediator.getSelectedUnits();

		boolean flag = false;
		for (int i = 0; i < selectedUnits.length; ++i) {
			if (selectedUnits[i] instanceof DataBuffer) {
				flag = true;
				break;
			}
		}

		paste.setEnabled(!mediator.copyBufferIsEmpty());

		if (selectedUnits.length == 0) {
			cut.setEnabled(false);
			delete.setEnabled(false);
			copy.setEnabled(false);
		}

		if (mediator.getUnitsCount() == 0) {
			select.setEnabled(false);
		}

		JMenu disconnectMenu = new JMenu("Disconnect from");
		JMenu initMenu = new JMenu("Remove Init File");
		if (selectedUnits.length == 1) {
			if (!(selectedUnits[0] instanceof DataBuffer)) {
				addInit.setEnabled(false);
			}

			for (int i = 0; i < selectedUnits[0].connectedUnits().size(); ++i) {
				JMenuItem item = new JMenuItem(selectedUnits[0].connectedUnits().get(i).getID());
				item.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						HardwareUnit[] target = new HardwareUnit[2];
						target[0] = mediator.getSelectedUnits()[0];
						target[1] = mediator.getUnit(e.getActionCommand());
						mediator.disconnectUnits(target);
					}
				});
				disconnectMenu.add(item);
			}

			for (int i = 0; i < selectedUnits[0].childList().size(); ++i) {
				MicsNode node = selectedUnits[0].childList().get(i);
				if (node instanceof InitAttribute) {
					Iterator<Map.Entry<String, StringBuilder>> iterator = node.attributeTable().entrySet().iterator();
					String element = node.getTag() + " ";
					while (iterator.hasNext()) {
						Map.Entry<String, StringBuilder> entry = iterator
								.next();
						element += entry.getKey() + "=\"" + entry.getValue()
								+ "\" ";
					}
					JMenuItem item = new JMenuItem(element);
					item.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent e) {
							mediator.removeChild(
									mediator.getSelectedUnits()[0], e
											.getActionCommand());
						}
					});
					initMenu.add(item);
				}
			}

			if (disconnectMenu.getMenuComponentCount() == 0) {
				disconnectMenu.setEnabled(false);
			}

			if (initMenu.getMenuComponentCount() == 0) {
				initMenu.setEnabled(false);
			}
		} else {
			addInit.setEnabled(false);
			disconnectMenu.setEnabled(false);
			initMenu.setEnabled(false);
		}

		JPopupMenu pop = new JPopupMenu();
		pop.add(addInit);
		pop.add(initMenu);
		pop.addSeparator();
		pop.add(disconnectMenu);
		pop.addSeparator();
		pop.add(cut);
		pop.add(copy);
		pop.add(paste);
		pop.add(delete);
		pop.addSeparator();
		pop.add(select);

		pop.show(canvas, mousePosition.x, mousePosition.y);
	}

	public void adjustUnitPositions(HardwareUnit[] units) {
		for (int i = 0; i < units.length; ++i) {
			HardwareUnit unit = units[i];
			int newX = (unit.appearance().getX() + pitch / 2) / pitch * pitch;
			int newY = (unit.appearance().getY() + pitch / 2) / pitch * pitch;
			mediator.setValue(unit.appearance(), "x", Integer.toString(newX));
			mediator.setValue(unit.appearance(), "y", Integer.toString(newY));
		}
	}

	public HardwareUnit onUnit(Point mousePosition) {

		HardwareUnit[] allUnits = mediator.getAllUnits();
		for (int i = 0; i < allUnits.length; ++i) {
			HardwareUnit unit = allUnits[i];
			if (unit.inRange(mousePosition.x, mousePosition.y)) {
				return unit;
			}
		}

		return null;
	}

	public void mousePressed(Point mousePosition) {
		HardwareUnit newSelectedUnit = null;

		HardwareUnit[] allUnits = mediator.getAllUnits();
		for (int i = 0; i < allUnits.length; ++i) {
			if (allUnits[i].inRange(mousePosition.x, mousePosition.y)) {
				newSelectedUnit = allUnits[i];
				break;
			}
		}

		if (newSelectedUnit == null) {
			if (currentKey != KeyEvent.VK_CONTROL) {
				mediator.clearSelectedUnits();
			}
			actionFrom = actionTo = mousePosition;
			actionMode = SELECTION_MODE;
			return;
		} else {
			if (currentKey == KeyEvent.VK_CONTROL) {
				if (mediator.selectedUnit(newSelectedUnit)) {
					mediator.deselect(newSelectedUnit);
				} else {
					mediator.select(newSelectedUnit);
				}
				return;
			} else if (currentKey == KeyEvent.VK_SHIFT) {
				mediator.clearSelectedUnits();
				mediator.select(newSelectedUnit);
				actionFrom.x = actionTo.x = newSelectedUnit.appearance().getX();
				actionFrom.y = actionTo.y = newSelectedUnit.appearance().getY();
				actionMode = CONNECTION_MODE;
				return;
			} else {
				if (!mediator.selectedUnit(newSelectedUnit)) {
					mediator.clearSelectedUnits();
					mediator.select(newSelectedUnit);
				}
			}

			actionFrom = mousePosition;
			mansion = new HauntedMansion(mediator.getSelectedUnits(), 128);
			actionMode = MOVEMENT_MODE;
		}
	}

	public void mouseDragged(Point mousePosition) {
		actionTo = mousePosition;

		int dx = 0, dy = 0;
		Point viewPoint = CanvasViewPort.getInstance().getViewPoint();
		Dimension viewPortSize = CanvasViewPort.getInstance().getViewPortSize();
		if((int)(viewPoint.x * 100.0 / scale) > mousePosition.x){
		    dx = -20;
		}
		else if((int)((viewPoint.x + viewPortSize.width) * 100.0 / scale) <= mousePosition.x){
		    dx = 20;
		}

		if((int)(viewPoint.y * 100.0 / scale) > mousePosition.y){
		    dy = -20;
		}
		else if((int)((viewPoint.y + viewPortSize.height) * 100.0 / scale) <= mousePosition.y){
		    dy = 20;
		}

		CanvasViewPort.getInstance().translateViewPoint(dx, dy);

		switch (actionMode) {
		case CONNECTION_MODE:
			break;

		case SELECTION_MODE:
		    break;

		case MOVEMENT_MODE:
			int x = actionTo.x - actionFrom.x;
			int y = actionTo.y - actionFrom.y;
			mansion.setOffset(x, y);
			break;
		}
	}

	public void mouseReleased(Point mousePosition) {
		HardwareUnit[] allUnits = mediator.getAllUnits();
		switch (actionMode) {
		case CONNECTION_MODE:
			for (int i = 0; i < allUnits.length; ++i) {
				HardwareUnit unit = allUnits[i];

				if (mediator.selectedUnit(unit)
						|| !unit.inRange(mousePosition.x, mousePosition.y))
					continue;

				mediator.getSelectedUnits()[0].connect(unit);
				mediator.clearSelectedUnits();
				mediator.select(unit);

				break;
			}

			break;

		case SELECTION_MODE:
			int x = Math.min(actionFrom.x, actionTo.x);
			int y = Math.min(actionFrom.y, actionTo.y);
			int w = Math.abs(actionFrom.x - actionTo.x);
			int h = Math.abs(actionFrom.y - actionTo.y);
			for (int i = allUnits.length - 1; i >= 0; --i) {
				HardwareUnit unit = allUnits[i];
				int ux = unit.appearance().getX();
				int uy = unit.appearance().getY();

				if ((x <= ux && ux <= x + w) && (y <= uy && uy <= y + h)) {
					if (mediator.selectedUnit(unit)) {
						mediator.deselect(unit);
					} else {
						mediator.select(unit);
					}
				}
			}
			break;

		case MOVEMENT_MODE:
			HardwareUnit[] selectedUnits = mansion.bodyList
					.toArray(new HardwareUnit[mansion.bodyList.size()]);
			for (int i = 0; i < selectedUnits.length; ++i) {
				Point newPosition = mansion.getPosition(selectedUnits[i]);
				mediator.setValue(selectedUnits[i].appearance(), "x", Integer
						.toString(newPosition.x));
				mediator.setValue(selectedUnits[i].appearance(), "y", Integer
						.toString(newPosition.y));
			}
			mansion = null;
			break;
		}

		actionMode = NONE_MODE;
	}

	protected void processKeyEvent(KeyEvent e) {
		if (e.getID() == KeyEvent.KEY_PRESSED) {
			keyPressed(e);
		} else if (e.getID() == KeyEvent.KEY_RELEASED) {
			keyReleased(e);
		}
	}

	public void keyPressed(KeyEvent e) {
		currentKey = e.getKeyCode();

		if (actionMode != NONE_MODE) {
			return;
		}

		switch (currentKey) {
		case KeyEvent.VK_J:
		    if(mediator.getSelectedUnitsCount() == 0){
			CanvasViewPort.getInstance().translateViewPoint(0, 10);
		    }
		    else{
			translateUnits(mediator.getSelectedUnits(), 0, 1);
		    }
			break;

		case KeyEvent.VK_K:
		    if(mediator.getSelectedUnitsCount() == 0){
			CanvasViewPort.getInstance().translateViewPoint(0, -10);
		    }
		    else{
			translateUnits(mediator.getSelectedUnits(), 0, -1);
		    }
			break;

		case KeyEvent.VK_L:
		    if(mediator.getSelectedUnitsCount() == 0){
			CanvasViewPort.getInstance().translateViewPoint(10, 0);
		    }
		    else{
			translateUnits(mediator.getSelectedUnits(), 1, 0);
		    }
			break;

		case KeyEvent.VK_H:
		    if(mediator.getSelectedUnitsCount() == 0){
			CanvasViewPort.getInstance().translateViewPoint(-10, 0);
		    }
		    else{
			translateUnits(mediator.getSelectedUnits(), -1, 0);
		    }
			break;

		case KeyEvent.VK_Y:
			mediator.copyUnits(mediator.getSelectedUnits());
			break;

		case KeyEvent.VK_P:
			mediator.pasteUnits();
			break;

		case KeyEvent.VK_X:
			HardwareUnit[] selectedUnits = mediator.getSelectedUnits();
			mediator.copyUnits(selectedUnits);
			mediator.deleteUnits(selectedUnits);
			break;

		case KeyEvent.VK_W:
			int unitsCount = mediator.getUnitsCount();
			if (unitsCount == 0) {
				break;
			}

			HardwareUnit nextUnit = (mediator.getAllUnits())[unitsCount - 1];
			mediator.clearSelectedUnits();
			mediator.select(nextUnit);

			break;

		default:
			break;
		}
	}

	public void keyReleased(KeyEvent e) {
		if (actionMode == SELECTION_MODE) {
			mediator.clearSelectedUnits();
			return;
		}
		actionFrom = actionTo = new Point(0, 0);
		currentKey = KeyEvent.VK_0;
		actionMode = NONE_MODE;
	}

    private void translateUnits(HardwareUnit[] units, int dx, int dy){
	for (int i = 0; i < units.length; ++i) {
	    HardwareUnit unit = units[i];
	    int x = unit.appearance().getX();
	    int y = unit.appearance().getY();
	    mediator.setValue(unit.appearance(), "x", Integer.toString(x + dx));
	    mediator.setValue(unit.appearance(), "y", Integer.toString(y + dy));
	}
    }

    public void paintConnectionLines(Graphics offg, Point offset) {
		offg.setColor(Color.black);
		HardwareUnit[] allUnits = mediator.getAllUnits();
		for (int i = allUnits.length - 1; i >= 0; --i) {
			HardwareUnit unit = allUnits[i];
			int x = (int)(unit.appearance().getX() * scale / 100.0);
			int y = (int)(unit.appearance().getY() * scale / 100.0);
			HardwareUnit[] connectedUnitList = unit.getConnectedUnitList();
			for (int j = 0; j < connectedUnitList.length; ++j) {
				offg.drawLine(x - offset.x, y - offset.y,
					      (int)((connectedUnitList[j].appearance().getX() * scale / 100.0) - offset.x),
					      (int)((connectedUnitList[j].appearance().getY() * scale / 100.0) - offset.y));
			}
		}

		if (actionMode == CONNECTION_MODE) {
		    offg.drawLine((int)(actionFrom.x * scale / 100.0 - offset.x),
				  (int)(actionFrom.y * scale / 100.0 - offset.y),
				  (int)(actionTo.x * scale / 100.0 - offset.x), 
				  (int)(actionTo.y * scale / 100.0 - offset.y));
		}
	}

    public void paintUnits(Graphics offg, Point offset) {
		offg.setColor(Color.black);

		int xMax = 0;
		int yMax = 0;
		HardwareUnit[] allUnits = mediator.getAllUnits();
		for (int i = allUnits.length - 1; i >= 0; --i) {
			HardwareUnit unit = allUnits[i];
			int width = (int)(unit.appearance().getWidth() * scale / 100.0);
			int height = (int)(unit.appearance().getHeight() * scale / 100.0);
			int left = (int)(unit.appearance().getX() * scale / 100.0 - width / 2);
			int top = (int)(unit.appearance().getY() * scale / 100.0 - height / 2);
			FontMetrics fontMetrics = offg.getFontMetrics(offg.getFont());
			int x = left + (width - fontMetrics.stringWidth(unit.getID())) / 2;
			int y = top + height + fontMetrics.getHeight();
			xMax = (left + width > xMax) ? left + width : xMax;
			yMax = (top + height > yMax) ? top + height : yMax;

			offg.drawString(unit.getID(), x - offset.x, y - offset.y);
			offg.drawImage(mediator.getImage(unit.appearance()), left - offset.x, top - offset.y, null);
			if (mediator.selectedUnit(unit)) {
				offg.drawRect(left - offset.x, top - offset.y, width - 1, height - 1);
			}
		}
		canvasSize.width = xMax + 100;
		canvasSize.height = yMax + 100;
	}

    public void paintGhosts(Graphics offg, Point offset) {
		if (actionMode != MOVEMENT_MODE) {
			return;
		}

		offg.setColor(new Color(128, 128, 128));
		int xMax = 0;
		int yMax = 0;
		Ghost[] ghosts = mansion.getGhosts();
		for (int i = ghosts.length - 1; i >= 0; --i) {
			Ghost ghost = ghosts[i];
			int width = (int)(ghost.size.width * scale / 100.0);
			int height = (int)(ghost.size.height * scale / 100.0);
			int left = (int)(ghost.position.x * scale / 100.0 - width / 2);
			int top = (int)(ghost.position.y * scale / 100.0 - height / 2);
			xMax = (left + width > xMax) ? left + width : xMax;
			yMax = (top + height > yMax) ? top + height : yMax;

			offg.drawImage(ghost.image, left - offset.x, top - offset.y, null);
			offg.drawRect(left - offset.x, top - offset.y, width - 1, height - 1);

		}
		if(xMax > canvasSize.width){
		    canvasSize.width = xMax;
		}
		if(yMax > canvasSize.height){
		    canvasSize.height = yMax;
		}
	}

    public void paintGhostLines(Graphics offg, Point offset) {
		if (actionMode != MOVEMENT_MODE) {
			return;
		}

		offg.setColor(new Color(128, 128, 128));

		Ghost[] ghosts = mansion.getGhosts();
		for (int i = ghosts.length - 1; i >= 0; --i) {
			Ghost ghost = ghosts[i];
			int fromX = (int)(ghost.position.x * scale / 100.0);
			int fromY = (int)(ghost.position.y * scale / 100.0);

			for (int j = 0; j < ghost.connectingPoints.size(); ++j) {
			    int toX = (int)(ghost.connectingPoints.get(j).x * scale / 100.0);
			    int toY = (int)(ghost.connectingPoints.get(j).y * scale / 100.0);
			    offg.drawLine(fromX - offset.x, fromY - offset.y, toX - offset.x, toY - offset.y);
			}
		}
	}

    public void paintSelectionRectangle(Graphics offg, Point offset) {
		if (actionMode == SELECTION_MODE) {
			offg.setColor(new Color(0, 64, 255, 16));
			int x = (int)(Math.min(actionFrom.x, actionTo.x) * scale / 100.0);
			int y = (int)(Math.min(actionFrom.y, actionTo.y) * scale / 100.0);
			int w = (int)(Math.abs(actionFrom.x - actionTo.x) * scale / 100.0);
			int h = (int)(Math.abs(actionFrom.y - actionTo.y) * scale / 100.0);

			offg.fillRect(x - offset.x, y - offset.y, w - 1, h - 1);
			offg.setColor(new Color(0, 100, 255));
			offg.drawRect(x - offset.x, y - offset.y, w - 1, h - 1);
		}
	}

	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();

		if (command.equals("Add Init File")) {
			mediator.addInitFile(mediator.getSelectedUnits()[0], "./init_file",
					"0");
		} else if (command.equals("Cut")) {
			mediator.copyUnits(mediator.getSelectedUnits());
			mediator.deleteUnits(mediator.getSelectedUnits());
		} else if (command.equals("Copy")) {
			mediator.copyUnits(mediator.getSelectedUnits());
		} else if (command.equals("Paste")) {
			mediator.pasteUnits();
		} else if (command.equals("Delete")) {
			mediator.deleteUnits(mediator.getSelectedUnits());
		} else if (command.equals("Select All")) {
			mediator.selectAll();
		}
	}
}
