/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.AdjustmentEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.AdjustmentListener;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import javax.swing.JScrollBar;

import net.wasamon.mics.architecturemaker.unit.HardwareUnit;

@SuppressWarnings("serial")
/**
 * @author Masayuki Morisita
 */
public class CanvasViewPort extends JComponent implements MouseListener,
							  MouseMotionListener {
	private Image offimage;
	private Graphics offg;
	private Point viewPoint;
	private Dimension viewPortSize;
	private VirtualCanvas virtualCanvas;
	private JScrollBar verticalBar, horizontalBar;
	private static CanvasViewPort instance = new CanvasViewPort();

	private CanvasViewPort() {
		super();
		setBackground(Color.white);
		addMouseListener(this);
		addMouseMotionListener(this);
		setFocusTraversalKeysEnabled(false);
		viewPoint = new Point(0, 0);
		viewPortSize = new Dimension(getWidth(), getHeight());
		virtualCanvas = new VirtualCanvas(viewPortSize.width, viewPortSize.height);
		setLayout(new BorderLayout());
		verticalBar = new JScrollBar(JScrollBar.VERTICAL, 0, 10, 0, 100);
		horizontalBar = new JScrollBar(JScrollBar.HORIZONTAL, 0, 10, 0, 100);
		add(verticalBar, "East");
		add(horizontalBar, "South");

		horizontalBar.addAdjustmentListener(new AdjustmentListener() {
			public void adjustmentValueChanged(AdjustmentEvent e) {
			    translateViewPoint(horizontalBar.getValue() - viewPoint.x, 0);
			    repaint();
			}
		    });

		verticalBar.addAdjustmentListener(new AdjustmentListener() {
			public void adjustmentValueChanged(AdjustmentEvent e) {
			    translateViewPoint(0, verticalBar.getValue() - viewPoint.y);
			    repaint();
			}
		    });
	}

	public static CanvasViewPort getInstance() {
		return instance;
	}

    public Point getViewPoint(){
	return new Point(viewPoint);
    }

    public Dimension getViewPortSize(){
	return new Dimension(viewPortSize);
    }

    public void translateViewPoint(int dx, int dy){
	if(dx > 0){
	    if(viewPoint.x + dx <= virtualCanvas.getWidth() - viewPortSize.width){
		viewPoint.translate(dx, 0);
	    }
	}
	else{
	    if(0 <= viewPoint.x + dx){
		viewPoint.translate(dx, 0);
	    }
	}

	if(dy > 0){
	    if(viewPoint.y + dy <= virtualCanvas.getHeight() - viewPortSize.height){
		viewPoint.translate(0, dy);
	    }
	}
	else{
	    if(0 <= viewPoint.y + dy){
		viewPoint.translate(0, dy);
	    }
	}
    }

    public void setScale(int scale){
	virtualCanvas.setScale(scale);
    }

    public int getScale(){
	return virtualCanvas.getScale();
    }

	public void adjustUnitPositions(HardwareUnit[] units) {
	    virtualCanvas.adjustUnitPositions(units);
	}

	public Point getVirtualMousePosition(Point physicalPosition){
	    int x = (int)((physicalPosition.x + viewPoint.x) * 100.0 / virtualCanvas.getScale());
	    int y = (int)((physicalPosition.y + viewPoint.y) * 100.0 / virtualCanvas.getScale());

	    return new Point(x, y);
	}

    public void mouseClicked(MouseEvent e) {
	Point mousePosition = e.getPoint();

	if (SwingUtilities.isRightMouseButton(e)) {
	    virtualCanvas.showPopup(this, mousePosition);
	}

    }

	public void mouseEntered(MouseEvent e) {
	}

	public void mouseExited(MouseEvent e) {
		e.getPoint().move(0, 0);
	}

	public void mousePressed(MouseEvent e) {
		requestFocusInWindow();
		Point mousePosition = getVirtualMousePosition(e.getPoint());

		virtualCanvas.mousePressed(mousePosition);

		repaint();
	}

	public void mouseDragged(MouseEvent e) {
		Point mousePosition = getVirtualMousePosition(e.getPoint());

		virtualCanvas.mouseDragged(mousePosition);

		repaint();
	}

	public void mouseReleased(MouseEvent e) {
		Point mousePosition = getVirtualMousePosition(e.getPoint());

		virtualCanvas.mouseReleased(mousePosition);

		repaint();
	}

	public void mouseMoved(MouseEvent e) {
		Point mousePosition = getVirtualMousePosition(e.getPoint());

		if(virtualCanvas.onUnit(mousePosition) != null){
		    setCursor(new Cursor(Cursor.HAND_CURSOR));
		}
		else{
		    setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		}
	}

	protected void processKeyEvent(KeyEvent e) {
		if (e.getID() == KeyEvent.KEY_PRESSED) {
		    virtualCanvas.keyPressed(e);
		} else if (e.getID() == KeyEvent.KEY_RELEASED) {
		    virtualCanvas.keyReleased(e);
		}
		repaint();
	}

	private void paintScrollBar(){
	    int hMax = Math.max(viewPoint.x + viewPortSize.width, virtualCanvas.getWidth());
	    horizontalBar.setValues(viewPoint.x, viewPortSize.width, 0, hMax);

	    int vMax = Math.max(viewPoint.y + viewPortSize.height, virtualCanvas.getHeight());
	    verticalBar.setValues(viewPoint.y, viewPortSize.height, 0, vMax);
	}

	public void update(Graphics g) {
		paintComponent(g);
	}

	public void paintComponent(Graphics g) {
	    if (offimage == null || 
		viewPortSize.width != getWidth() || viewPortSize.height != getHeight()) {
			offimage = createImage(getWidth(), getHeight());
			offg = offimage.getGraphics();
			viewPortSize.width = getWidth();
			viewPortSize.height = getHeight();
			offg.setFont(new Font("Dialog", Font.BOLD, 12));
		}

		offg.setColor(Color.white);
		offg.fillRect(0, 0, viewPortSize.width, viewPortSize.height);

		virtualCanvas.paintConnectionLines(offg, viewPoint);

		virtualCanvas.paintGhostLines(offg, viewPoint);

		virtualCanvas.paintUnits(offg, viewPoint);

		virtualCanvas.paintGhosts(offg, viewPoint);

		virtualCanvas.paintSelectionRectangle(offg, viewPoint);

		paintScrollBar();

		g.drawImage(offimage, 0, 0, this);
	}
}
