/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * MICSシミュレーションのグラフィカルインターフェースを提供するパッケージ
 */
package net.wasamon.mics.gui;

import java.util.Hashtable;
import java.util.Iterator;

import net.wasamon.mics.MicsCompositeElement;
import net.wasamon.mics.MicsElement;

/**
 * アーキテクチャの内部情報を表示するためのクラスが実装すべき基底クラス
 * 
 * @author miyo
 * 
 */
public abstract class MicsArchitectureViewer {

	/**
	 * {@link String}, {@link MicsArchitectureViewer}
	 */
	private Hashtable<String, MicsArchitectureViewer.Node> data;

	/**
	 * 現在選択されているノードインスタンス
	 */
	protected MicsArchitectureViewer.Node selectedNode;

	protected MicsArchitectureViewer(MicsCompositeElement composite) {
		data = new Hashtable<String, MicsArchitectureViewer.Node>();
		data.clear();
		MicsElement[] elements = composite.getMicsElementArray();
		int count = 0;
		for (int i = 0; i < elements.length; i++) {
			MicsElement e = elements[i];
			data.put(e.id(), new MicsArchitectureViewer.Node(e, count * 30));
			count++;
		}
	}

	/**
	 * 保持するデータを描画する
	 */
	public void paint() {
		if (data == null) {
			return;
		}
		Iterator<MicsArchitectureViewer.Node> it;
		it = data.values().iterator();
		while (it.hasNext()) {
			it.next().paintEdge();
		}
		it = data.values().iterator();
		while (it.hasNext()) {
			MicsArchitectureViewer.Node n = it.next();
			if (n.isSelected()) {
				n.paintNode(true);
			} else {
				n.paintNode(false);
			}
		}
	}

	/**
	 * 指定した座標にあるノードを選択する
	 * 
	 * @param x
	 * @param y
	 */
	public void setSelectedNode(int x, int y) {
		Iterator<MicsArchitectureViewer.Node> it = data.values().iterator();
		if (selectedNode != null) {
			selectedNode.setSelected(false);
		}
		selectedNode = null;
		while (it.hasNext()) {
			MicsArchitectureViewer.Node n = it.next();
			if (n.isContain(x, y)) {
				n.setSelected(true);
				selectedNode = n;
				break;
			}
		}
		return;
	}
	
	public MicsArchitectureViewer.Node getSelectedNode(){
		return selectedNode;
	}

	public abstract ViwerCanvas canvas();

	/**
	 * ビューワの描画対象であるキャンバスが実装すべきインターフェース
	 * 
	 * @author miyo
	 * 
	 */
	public interface ViwerCanvas {

		public void setColorBlack();

		public void setColorRed();

		public void setFGColor();

		public void setBGColor();

		public void drawLine(int x0, int y0, int x1, int y1);

		public void drawImage(String name, int x, int y, int w, int h);

		public void drawRect(int x, int y, int w, int h);

		public void fillRect(int x, int y, int w, int h);

		public void drawString(String str, int x, int y);

		public int width();

		public int height();

		public void setSize(int x, int y);

	}

	/**
	 * 描画の対象となるシミュレーション要素の情報を保持する単位ユニット。
	 * 
	 * @author miyo
	 * 
	 */
	public class Node {
		public MicsElement element;
		String name;
		String image = null;
		boolean fSelected = false;

		public int X;
		public int Y;
		public int SIZE = 30;

		public Node(MicsElement element, int offset) {
			this.element = element;
			MicsElement.Appearance appearance = element.getAppearance();
			if (appearance.x == null) {
				X = 20 + offset;
			} else {
				X = appearance.x.intValue();
			}
			if (appearance.y == null) {
				Y = 20 + offset;
			} else {
				Y = appearance.y.intValue();
			}
			name = element.id() + ":" + getShortName(element.getClass().getName());
			image = appearance.image;
		}

		private String getShortName(String s) {
			int ch = s.lastIndexOf('.');
			if (ch < 0) {
				return s;
			} else {
				return s.substring(ch);
			}
		}

		private int marginY = 15;
		private int ascent;

		public int getCenterX() {
			return X + SIZE / 2;
		}

		public int getCenterY() {
			return Y + SIZE / 2;
		}

		public void paintEdge() {
			canvas().setColorBlack();
			String connecteds[] = element.getConnectedElements();
			if (connecteds != null) {
				for (int i = 0; i < connecteds.length; i++) {
					MicsArchitectureViewer.Node dest = (MicsArchitectureViewer.Node) (data
							.get(connecteds[i]));
					if (dest != null) {
						canvas().drawLine(getCenterX(), getCenterY(), dest.getCenterX(),
								dest.getCenterY());
					}
				}
			}
		}

		public void paintNode(boolean flag) {
			if (flag) {
				canvas().setFGColor();
			} else {
				canvas().setBGColor();
			}
			if (image != null) {
				canvas().drawImage(image, X, Y, SIZE, SIZE);
			} else {
				canvas().fillRect(X, Y, SIZE, SIZE);
			}
			canvas().setFGColor();
			canvas().drawString(name, X, Y + ascent + marginY + SIZE);
			if (flag) {
				canvas().setColorRed();
			}
			canvas().drawRect(X, Y, SIZE, SIZE);
			canvas().drawRect(X - 1, Y - 1, SIZE + 2, SIZE + 2);
			int w = canvas().width();
			int h = canvas().height();
			if (X + SIZE + 1 > w || Y + SIZE + 1 > h) {
				int x = X + SIZE + 1 > w ? X + SIZE + 100 : w;
				int y = Y + SIZE + 1 > h ? Y + SIZE + 100 : h;
				canvas().setSize(x, y);
			}
		}

		/**
		 * 指定した座標が、このノードの占有する範囲にふくまれるかどうか判定する。
		 * 
		 * @param x
		 * @param y
		 * @return
		 */
		public boolean isContain(int x, int y) {
			return ((x > X) && (x < X + SIZE) && (y > Y) && (y < Y + SIZE));
		}

		/**
		 * このノードの位置を指定した座標に設定する
		 * 
		 * @param x
		 * @param y
		 * @return
		 */
		public void setPosition(int x, int y) {
			this.X = x;
			this.Y = y;
		}

		/**
		 * このノードが選択されている状態かどうか判定する
		 * 
		 * @return
		 */
		public boolean isSelected() {
			return fSelected;
		}

		/**
		 * このノードを選択状態にする
		 */
		public void setSelected(boolean flag) {
			fSelected = flag;
		}
	}

	/**
	 * このビューを表示する。
	 */
	abstract public void show();

}
