#include <gtest/gtest.h> 
#include <fstream>
#include <sstream>

#include <meshio.h>
using namespace meshio;

TEST(binary, FileReader)
{
	binary::FileReader reader("cube.mqo");

	char buf[1024];
	while(reader.read(buf, 1024)){
	}
	EXPECT_EQ(1235, reader.getPos());
}

TEST(binary, MemoryReader)
{
	std::vector<char> all;
	binary::readAll("cube.mqo", all);
	binary::MemoryReader reader(&all[0], all.size());
	char buf[1024];
	while(reader.read(buf, 1024)){
	}
	EXPECT_EQ(1235, reader.getPos());
}

TEST(text, LineReader)
{
	binary::FileReader reader("cube.mqo");
	text::LineReader<
		text::IsCRLF, 
		text::IsWhiteSpace, 
		text::IsEmpty> l(reader);
	EXPECT_EQ("Metasequoia Document", l.getLine().str());
	EXPECT_EQ("Format Text Ver 1.0", l.getLine().str());

	{
		text::cstr line=l.getLine();
		text::LineSplitter<text::IsWhiteSpace> splitter(line);
		EXPECT_EQ("Scene", splitter.get().str());
	}

	{
		text::cstr line=l.getLine();
		text::LineSplitter<text::IsWhiteSpace> splitter(line);
		EXPECT_EQ("pos", splitter.get().str());
		EXPECT_EQ(0.0000f, splitter.getFloat());
		EXPECT_EQ(0.0000f, splitter.getFloat());
		EXPECT_EQ(1500.0000f, splitter.getFloat());
	}
  
	while(true)
	{
		text::cstr line=l.getLine();
		if(l.isEnd()){
			break;
		}
	}
	EXPECT_EQ(43, l.getLineCount());
}

TEST(mqo, read) 
{ 
	mqo::IO io;
	EXPECT_TRUE(io.read("cube.mqo"));
	EXPECT_EQ(1, io.objects.size());
	EXPECT_EQ(8, io.objects[0].vertices.size());
	EXPECT_EQ(6, io.objects[0].faces.size());
}

TEST(mqo, write) 
{ 
	{
		mqo::IO io;
		EXPECT_TRUE(io.read("cube.mqo"));
		EXPECT_EQ(1, io.objects.size());
		EXPECT_EQ(8, io.objects[0].vertices.size());
		EXPECT_EQ(6, io.objects[0].faces.size());
		EXPECT_TRUE(io.write("cube2.mqo"));
		EXPECT_EQ(0, io.materials.size());
	}

	{
		mqo::IO io;
		EXPECT_TRUE(io.read("cube2.mqo"));
		EXPECT_EQ(1, io.objects.size());
		EXPECT_EQ(8, io.objects[0].vertices.size());
		EXPECT_EQ(6, io.objects[0].faces.size());
	}
}

TEST(pmd, read) 
{ 
	pmd::IO io;
	EXPECT_TRUE(io.read(L"~NVer2.pmd"));
	EXPECT_STREQ("~N", io.name);
	EXPECT_EQ(17, io.materials.size());
	EXPECT_EQ(12354, io.vertices.size());
	EXPECT_EQ(68883, io.indices.size());
	EXPECT_EQ(45, io.rigidbodies.size());
	EXPECT_EQ(27, io.constraints.size());
}

TEST(pmd, write) 
{ 
	{
		pmd::IO io;
		EXPECT_TRUE(io.read(L"~NVer2.pmd"));
		EXPECT_STREQ("~N", io.name);
		EXPECT_EQ(17, io.materials.size());
		EXPECT_EQ(12354, io.vertices.size());
		EXPECT_EQ(68883, io.indices.size());
		EXPECT_EQ(45, io.rigidbodies.size());
		EXPECT_EQ(27, io.constraints.size());
		EXPECT_TRUE(io.write("miku.pmd"));
	}

	{
		pmd::IO io;
		EXPECT_TRUE(io.read(L"miku.pmd"));
		EXPECT_STREQ("~N", io.name);
		EXPECT_EQ(17, io.materials.size());
		EXPECT_EQ(12354, io.vertices.size());
		EXPECT_EQ(68883, io.indices.size());
		//EXPECT_EQ(45, io.rigidbodies.size());
		//EXPECT_EQ(27, io.constraints.size());
	}

	{
		pmd::IO io;
		EXPECT_TRUE(io.write("miku.pmd"));
	}
}

TEST(vmd, read)
{
	vmd::IO io;
	EXPECT_TRUE(io.read("mikumiku.vmd"));
	EXPECT_EQ(122, io.boneMap.size());
	EXPECT_EQ(16, io.morphMap.size());
}

int main(int argc, char **argv)
{
	testing::InitGoogleTest(&argc, argv); 
	RUN_ALL_TESTS();
}

