using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using hardware;

namespace imagefile{
	abstract class EncodeMap{
		protected struct EncodePage{
			public Archiver.Bpe.Container Data{
				get; set;
			}
			public int Pagenum{
				get; set;
			}
		}
		protected const int evaluate_buffer_first = 0x200;
		protected static readonly int [] evaluate_buffer = new int[] {
			0x400, 0x800, 0x1000, 0x2000
		};
		virtual protected bool buffer_ask(int size)
		{
			if(evaluate_buffer_first == size){
				return true;
			}
			foreach(int t in evaluate_buffer){
				if(t == size){
					return true;
				}
			}
			return false;
		}
		abstract public bool TypeAsk(string t);
		abstract public bool HaveStream();
		//for VBR and CBR multi
		virtual protected void buffersize_set(int page, int size)
		{
			Debug.Assert(false);
		}
		public bool BuffersizeSet(int page, int size)
		{
			if(buffer_ask(size) == false){
				return false;
			}
			buffersize_set(page, size);
			return true;
		}
		virtual public int Length()
		{
			return 0;
		}
		//for CBR single
		virtual protected void buffersize_set(int page)
		{
			Debug.Assert(false);
		}
		public bool BuffersizeSet(int size)
		{
			if(buffer_ask(size) == false){
				return false;
			}
			buffersize_set(size);
			return true;
		}
		virtual public int Encode(byte[] rom, bool use_tail, int limit)
		{
			Debug.Assert(false);
			return 0;
		}
		abstract public bool BuffersizeDone();
		//backlog offset をつけると CBR になるので注意
		//VBR 固定なので backlog offset はつけない
		virtual public void BacklogOffsetSet(int offset)
		{
		}
		virtual public void Write(ref int offset, byte [] rom)
		{
		}
		static public void struct_set(byte page_count, byte org_page, byte dest_page, int offset, List<byte> list)
		{
			//0 1 count
			list.Add(page_count);
			//1 1 もとのデータの本来の page
			list.Add(org_page);
			//2 1 展開されたデータの配置 page
			list.Add(dest_page);
			//3 3 *image
			mmc5.offset_add(offset, list);
		}
		virtual public void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
		}
		virtual public bool PackedPagenum(int page)
		{
			return false;
		}
		protected void container_write(Archiver.Bpe.Container c, ref int offset, byte [] rom)
		{
			byte [] au = c.EncodeDataGet();
			au.CopyTo(rom, offset);
			offset += au.Length;
			Debug.Assert(au.Length == c.EncodedLength());
		}
	}
	class EncodeRaw : EncodeMap{
		public EncodeRaw()
		{
		}
		override public bool TypeAsk(string t)
		{
			return t == "raw";
		}
		override public bool BuffersizeDone()
		{
			return true;
		}
		override public bool HaveStream()
		{
			return false;
		}
		virtual protected void Write(ref int offset, byte [] rom, byte [] raw)
		{
		}
	}
	class EncodeVbr : EncodeMap{
		struct Map{
			public int Page{
				get;set;
			}
			public int Size{
				get;set;
			}
		}
		int m_pagesize;
		List<Map> m_buffer = new List<Map>();
		EncodePage [] m_page;
		public EncodeVbr(int pagesize)
		{
			m_pagesize = pagesize;
		}
		override public bool TypeAsk(string t)
		{
			return t == "vbrpage";
		}
		override public bool HaveStream()
		{
			return true;
		}
		override public int Length()
		{
			int size = 0;
			foreach(EncodePage t in m_page){
				size += t.Data.EncodedLength();
			}
			return size;
		}
		override protected void buffersize_set(int page, int size)
		{
			Map t = new Map();
			t.Page = page;
			t.Size = size;
			m_buffer.Add(t);
		}
		protected byte[][] vbr_block_make(byte [] rom, int pagesize, bool use_tail)
		{
			int pagecount = rom.Length / pagesize;
			if(use_tail == false){
				pagecount -= 1;
			}
			byte [][]pagedata = new byte[pagecount][];
			for(int page = 0; page < pagecount; page++){
				pagedata[page] = new byte [pagesize];
				Buffer.BlockCopy(rom, page * pagesize, pagedata[page], 0, pagesize);
			}
			return pagedata;
		}
		override public int Encode(byte[] rom, bool use_tail, int limit)
		{
			byte[][]data = vbr_block_make(rom, mmc5.PAGESIZE, use_tail);
			List<EncodePage> list = new List<EncodePage>();
			Debug.Assert(m_buffer.Count == m_pagesize);
			for(int i = 0; i < Math.Min(limit, m_buffer.Count); i++){
				Map map = m_buffer[i];
				Archiver.Bpe.Container ee = new Archiver.Bpe.Container(map.Size, Archiver.Bpe.Container.type.VBR);
				ee.Encode(data[map.Page], false);
				EncodePage t = new EncodePage();
				t.Pagenum = map.Page;
				t.Data = ee;
				list.Add(t);
				Debug.Assert(ee.DecodeTest(data[map.Page]));
			}
			//page 順に並べ直し、bios で pagetable を作りやすくする
			IEnumerable<EncodePage> page_order = (
				from s in list.ToArray()
				orderby s.Pagenum
				select s
			);
			list = new List<EncodePage>(page_order);
			m_page = list.ToArray();
			return this.Length();
		}
		override public bool BuffersizeDone()
		{
			return m_buffer.Count == m_pagesize;
		}
		override public void Write(ref int offset, byte [] rom)
		{
			foreach(EncodePage t in m_page){
				container_write(t.Data, ref offset, rom);
			}
		}
		override public void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
			foreach(EncodePage t in m_page){
				struct_set(1, (byte) t.Pagenum, rombuffer.Dequeue(), offset, list);
				offset += t.Data.EncodedLength();
			}
		}
		override public bool PackedPagenum(int page)
		{
			foreach(EncodePage t in m_page){
				if(page == t.Pagenum){
					return true;
				}
			}
			return false;
		}
	}
	class EncodeCbrSingle : EncodeMap{
		int m_buffer;
		bool m_backstream;
		Archiver.Bpe.Container m_encoder = null;
#if DEBUG
		byte [] m_plainrom; // for decode test
#endif
		public EncodeCbrSingle(bool backstream)
		{
			m_backstream = backstream;
			m_buffer = 0;
		}
		override public bool TypeAsk(string t)
		{
			return t == "cbr";
		}
		override public bool HaveStream()
		{
			return true;
		}
		override public int Length()
		{
			if(m_encoder == null){
				return 0;
			}
			return m_encoder.EncodedLength();
		}
		override protected void buffersize_set(int size)
		{
			m_buffer = size;
		}
		override public bool BuffersizeDone()
		{
			return m_buffer != 0;
		}
		override public int Encode(byte[] rom, bool use_tail, int limit)
		{
			m_encoder = new Archiver.Bpe.Container(m_buffer, Archiver.Bpe.Container.type.CBR);
			m_encoder.Encode(rom, m_backstream);
#if DEBUG
			m_plainrom = rom;
#endif
			return this.Length();
		}
		override public void BacklogOffsetSet(int offset)
		{
			m_encoder.BacklogOffsetSet(offset);
#if DEBUG
			Debug.Assert(m_encoder.DecodeTest(m_plainrom));
#endif
		}
		override public void Write(ref int offset, byte [] rom)
		{
			container_write(m_encoder, ref offset, rom);
		}
		override public void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
			//0, 0 は ppurom しか使わないので定数だが、 cpurom で使うことになれば変数にすること
			struct_set(pagesize, 0, 0, offset, list);
		}
	}

	class EncodeCbrMulti : EncodeMap{
		struct Context{
			public byte [] Data{
				get; set;
			}
			public int BufferSize{
				get; set;
			}
			public Archiver.Bpe.Container Encoder{
				get; set;
			}
			public int RawLength{
				get; set;
			}
		}
		public const int RAW = 0x4649;
		const int UNDEF = -1;
		protected const string SAFIX = "cbrfile";
		readonly Context [] m_data;
		int m_encode_index = 0, m_write_index = 0;
		public EncodeCbrMulti(int filecount)
		{
			m_data = new Context[filecount];
			for(int i = 0; i < filecount; i++){
				Context t = new Context();
				t.BufferSize = UNDEF;
			}
		}
		override public bool TypeAsk(string t)
		{
			return t == SAFIX;
		}
		override public bool HaveStream()
		{
			return true;
		}
		override public int Length()
		{
			int size = 0;
			foreach(Context t in m_data){
				Debug.Assert(t.BufferSize != UNDEF);
				size += t.Data.Length;
			}
			return size;
		}
		override protected bool buffer_ask(int size)
		{
			if(size == RAW){
				return true;
			}
			return base.buffer_ask(size);
		}
		override protected void buffersize_set(int file, int size)
		{
			if(file > m_data.Length){
				return;
			}
			Archiver.Bpe.Container ee;
			if(size == RAW){
				ee = null;
			}else{
				ee = new Archiver.Bpe.Container(size, Archiver.Bpe.Container.type.CBR);
			}
			m_data[file].BufferSize = size;
			m_data[file].Encoder = ee;
		}
		override public int Encode(byte[] file, bool forceraw, int limit)
		{
			//script 単位で load をかけているので、2度使用すると index が飽和してしまうので仕方なく 0 に補正をかける
			if(m_encode_index == m_data.Length){
				m_encode_index = 0;
			}
			int index = m_encode_index++;
			m_data[index].RawLength = file.Length;
			if((forceraw == true) || (m_data[index].Encoder == null)){
				m_data[index].Data = file;
				m_data[index].Encoder = null;
				return m_data[index].RawLength;
			}else{
				m_data[index].Encoder.Encode(file, false);
				m_data[index].Data = m_data[index].Encoder.EncodeDataGet();
				return 1 + m_data[index].Data.Length; //1 is blocknum
			}
		}
		override public bool BuffersizeDone()
		{
			foreach(Context t in m_data){
				if(t.BufferSize == UNDEF){
					return false;
				}
			}
			return true;
		}
		override public void Write(ref int offset, byte [] rom)
		{
			if(m_write_index == m_data.Length){
				m_write_index = 0; //補正理由は encode と同じ
			}
			Context t = m_data[m_write_index];
			if(t.Encoder != null){
				rom[offset - 1] |= 0x80;
				int block_num = t.RawLength / t.Encoder.BufferSize;
				if((t.RawLength % t.Encoder.BufferSize) != 0){
					block_num += 1;
				}
				rom[offset++] = (byte) block_num;
			}
			t.Data.CopyTo(rom ,offset);
			offset += t.Data.Length;
			m_write_index++;
		}
	}
	sealed class EvaluateRom : EncodeVbr{
		const int EVALUATE_LIMIT_NONE = 0;
		int m_vbr_size;
		EncodePage [] m_vbr_page;
		public EvaluateRom(): base(evaluate_buffer_first)
		{
		}
		public void VbrEvaluate(byte [] rom, int limit, bool use_tail)
		{
			byte[][] pagedata = vbr_block_make(rom, mmc5.PAGESIZE, use_tail);
			int pagecount = pagedata.Length;
			List<EncodePage> bpelist = new List<EncodePage>();
			for(int i = 0; i < pagecount; i++){
				Archiver.Bpe.Container ee = new Archiver.Bpe.Container(evaluate_buffer_first, Archiver.Bpe.Container.type.VBR);
				ee.Encode(pagedata[i], false);
				foreach(int buffersize in evaluate_buffer){
					Archiver.Bpe.Container am = new Archiver.Bpe.Container(buffersize, Archiver.Bpe.Container.type.VBR);
					am.Encode(pagedata[i], false);
					if(am.EncodedLength() < ee.EncodedLength()){
						ee = am;
					}
				}
				EncodePage t = new EncodePage();
				t.Pagenum = i;
				t.Data = ee;
				bpelist.Add(t);
			}
			IEnumerable<EncodePage> compress_order = (
				from s in bpelist.ToArray()
				orderby s.Data.EncodedLength()
				select s
			);
			bpelist = new List<EncodePage>(compress_order);
			if((limit != EVALUATE_LIMIT_NONE) && (bpelist.Count > limit)){
				bpelist.RemoveRange(limit, bpelist.Count - limit);
			}
			m_vbr_size = 0;
			m_vbr_page = bpelist.ToArray();
			foreach(EncodePage t in m_vbr_page){
				m_vbr_size += t.Data.EncodedLength();
			}
		}
		public void VbrResultWrite(string type_prefix, List<string> rank)
		{
			string hoe = String.Format(
				"{0}.type = VBR,0x{1:x2}",
				type_prefix, m_vbr_page.Length
			);
			rank.Add(hoe);
			foreach(EncodePage t in m_vbr_page){
				string au = String.Format(
					"{0}.vbrpage = 0x{1:x2},0x{2:x4} #{3:P}",
					type_prefix,
					t.Pagenum, t.Data.BufferSize, 
					(double) t.Data.EncodedLength() / (double) mmc5.PAGESIZE
				);
				rank.Add(au);
			}
		}
/*
board.cpu.rom.encode.type = NONE
board.ppu.rom.encode.type = CBR
board.cpu.rom.encode.cbr = 0x0800 #66.34%
board.cpu.rom.encode.type = VBR,0x10
board.cpu.rom.encode.vbrpage = 0x09,0x1000 #56.91%
*/
		public void CompressEvaluate(byte [] rom, bool tail_use, string type_prefix, List<string> rank)
		{
			if(rom.Length == mmc5.PAGESIZE){
				this.VbrEvaluate(rom, EVALUATE_LIMIT_NONE, tail_use);
				VbrResultWrite(type_prefix, rank);
				return;
			}
			//cbr 最小値を算出
			Archiver.Bpe.Container ee = new Archiver.Bpe.Container(evaluate_buffer_first, Archiver.Bpe.Container.type.CBR);
			ee.Encode(rom, true);
			foreach(int buffersize in evaluate_buffer){
				Archiver.Bpe.Container am = new Archiver.Bpe.Container(buffersize, Archiver.Bpe.Container.type.CBR);
				am.Encode(rom, true);
				if(am.EncodedLength() < ee.EncodedLength()){
					ee = am;
				}
			}
			
			this.VbrEvaluate(rom, EVALUATE_LIMIT_NONE, tail_use);
			if(m_vbr_size < ee.EncodedLength()){
				VbrResultWrite(type_prefix, rank);
			}else{
				rank.Add(type_prefix + ".type = CBR");
				string au = String.Format(
					"{0}.cbr = 0x{1:x4} #{2:P}",
					type_prefix, ee.BufferSize, 
					(double)ee.EncodedLength() / (double)rom.Length
				);
				rank.Add(au);
			}
		}
	}

	class EvaluateDisk : EncodeCbrMulti{
		int m_fileindex;
		readonly string m_prefix;
		readonly List<string> m_list;
		public EvaluateDisk(string prefix) : base(0)
		{
			m_prefix = prefix;
			m_list = new List<string>();
			m_fileindex = 0;
		}
		public void CompressEvaluate(byte [] data)
		{
			Archiver.Bpe.Container ee = new Archiver.Bpe.Container(evaluate_buffer_first, Archiver.Bpe.Container.type.CBR);
			ee.Encode(data, false);
			if(data.Length <= ee.EncodedLength()){
				string t = String.Format(
					"{0}encode.{1} = {2},RAW",
					m_prefix, SAFIX, m_fileindex
				);
				m_list.Add(t);
				m_fileindex += 1;
				return;
			}
			foreach(int bufsize in evaluate_buffer){
				Archiver.Bpe.Container rr = new Archiver.Bpe.Container(bufsize, Archiver.Bpe.Container.type.CBR);
				rr.Encode(data, false);
				if(rr.EncodedLength() < ee.EncodedLength()){
					ee = rr;
				}
				if(bufsize >= data.Length){
					break;
				}
			}
			string au = String.Format(
				"{0}encode.{1} = {2},0x{3:x4} #{4:P}",
				m_prefix, SAFIX, m_fileindex, ee.BufferSize,
				(double)ee.EncodedLength() / (double) data.Length
			);
			m_fileindex += 1;
			m_list.Add(au);
		}
		public void ListAdd(List<string> list)
		{
			list.Add(m_prefix + "encode.type = CBR," + m_fileindex.ToString());
			foreach(string t in m_list.ToArray()){
				list.Add(t);
			}
		}
	}
	class CompressRom{
		protected int m_mdcoffset = 0;
		protected readonly EncodeMap m_map;
		public int MdcOffset{
			set {m_mdcoffset = value;}
		}
		public int Length{
			get {return m_map.Length();}
		}
		protected CompressRom(EncodeMap map)
		{
			m_map = map;
		}
		virtual public bool Write(ref int offset, byte [] rom)
		{
			return true;
		}
		protected bool write(ref int offset, byte [] rom)
		{
			if(m_mdcoffset != offset){
				return false;
			}
			m_map.BacklogOffsetSet(m_mdcoffset);
			m_map.Write(ref offset, rom);
			return true;
		}
	}

	class CompressCpuRom : CompressRom{
		readonly int m_wram_count;
		public int PackedPageCount{
			get {return m_wram_count;}
		}
		public CompressCpuRom(int count, EncodeMap map) : base(map)
		{
			m_wram_count = count;
		}
		public void Compress(byte []rom)
		{
			m_map.Encode(rom, false, m_wram_count);
		}
		public void StructSet(Queue<byte> rombuffer, List<byte> list)
		{
			Debug.Assert(m_mdcoffset != 0);
			m_map.StructSet(1, m_mdcoffset, rombuffer, list);
		}
		public bool PackedPagenum(int page)
		{
			return m_map.PackedPagenum(page);
		}
		override public bool Write(ref int offset, byte [] rom)
		{
			if(this.Length == 0){
				return true;
			}
			return write(ref offset, rom);
		}
	}
	class CompressPpuRom : CompressRom{
		int m_rawsize = 0;
		public CompressPpuRom(EncodeMap map) : base(map)
		{
		}
		public void Compress(byte[] rom)
		{
			m_rawsize = rom.Length;
			if(m_rawsize == 0){
				return;
			}
			m_map.Encode(rom, true, m_rawsize / mmc5.PAGESIZE);
		}
		public void StructSet(List<byte> list)
		{
			Debug.Assert(
				((m_rawsize == 0) && (m_mdcoffset == 0)) ||
				((m_rawsize != 0) && (m_mdcoffset != 0))
			);
			EncodeMap.struct_set((byte) (m_rawsize / mmc5.PAGESIZE), 0, 0, m_mdcoffset, list);
		}
		override public bool Write(ref int offset, byte [] rom)
		{
			if(m_rawsize == 0){
				return true;
			}
			return write(ref offset, rom);
		}
	}
}
