﻿using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using hardware;

namespace imagefile{
	class Nes : GameImage{
		public enum mapper : byte{
			NROM = 0, SxROM, UxROM, CxROM, TxROM, ExROM,
			VRC3 = 73
		};
		enum mirror{ Programmable, Horizonal, Vertical };
		static readonly byte [] HEADER = {
			(byte)'N', (byte)'E', (byte)'S', 0x1a, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0
		};
		byte [] m_cpu_rom, m_ppu_rom;
		EncodeMap m_cpu_encode, m_ppu_encode;
		int m_cpu_ram_size;
		mirror m_mirror;
		mapper m_mappernum;
		bool m_workram = false; //true is need workram without battery
		SaveArgument m_save = null;
		//---- method ----
		public Nes() : base(mdc5.Script.imagetype.rom)
		{
		}
		bool rom_set(int [] oksize, byte [] input, out byte [] output){
			foreach(int t in oksize){
				if(t == input.Length){
					//output = new byte [t];
					//input.CopyTo(output, 0);
					output = input;
					return true;
				}
			}
			output = new byte [0];
			return false;
		}
		bool ram_set(int [] oksize, int input, out int output){
			foreach(int t in oksize){
				if(t == input){
					output = t;
					return true;
				}
			}
			output = 0;
			return false;
		}
		public bool ImageMake(byte [] cpu_rom, int cpu_ram_size, byte [] ppu_rom, mapper mappernum)
		{
			if(rom_set(mmc5.CPU_ROMSIZE, cpu_rom, out m_cpu_rom) == false){
				return false;
			}
			if(ram_set(mmc5.CPU_RAMSIZE, cpu_ram_size, out m_cpu_ram_size) == false){
				return false;
			}
			if(rom_set(mmc5.PPU_ROMSIZE, ppu_rom, out m_ppu_rom) == false){
				return false;
			}
			m_mappernum = mappernum;
			return true;
		}
		int header_set(int cpu_rom_size, int cpu_ram_size, int ppu_rom_size, int mappernum, out byte [] header){
			header = HEADER;
			header[4] = (byte) (cpu_rom_size / 0x4000);
			header[5] = (byte) (ppu_rom_size / 0x2000);
			header[6] = 0;
			header[6] |= (byte) (mappernum << 4); //byte 型での shift 演算を受け付けないので mappernum の型を int にした
			header[8] = (byte) (cpu_ram_size / 0x2000);
			//header[0x0a] = 0x0a; //charcter RAM capacity
			//header[0x0b] = 0x0b;
			return header.Length + cpu_rom_size + ppu_rom_size;
		}
		public byte [] ImageGet(){
			byte [] header;
			int imagesize = header_set(
				m_cpu_rom.Length, m_cpu_ram_size, m_ppu_rom.Length, 
				(int) m_mappernum, out header
			);
			byte [] image = new byte[imagesize];
			int offset = 0;
			
			header.CopyTo(image, offset);
			offset += header.Length;
			m_cpu_rom.CopyTo(image, offset);
			offset += m_cpu_rom.Length;
			m_ppu_rom.CopyTo(image, offset);
			offset += m_ppu_rom.Length;
			return image;
		}
		Interface.ImageHash romhash_get(byte [] buf)
		{
			Interface.ImageHash c = new Interface.ImageHash();
			c.Transform(buf);
			c.Final();
			return c;
		}
		override public bool Load(mdc5.Script s)
		{
			m_name = s.BiosName;
			m_code = s.GameCode;
			m_version = s.GameVersion;
			s.EncodeMapGet(out m_cpu_encode, out m_ppu_encode);
			return Load(s.ImageFilename);
		}
		override public bool Load(string file)
		{
			byte [] buf;
			if(Static.Utility.binary_load(file, out buf) == false){
				return false;
			}
			if(buf.Length < HEADER.Length){
				return false;
			}
			if(buf[0] != 'N' || buf[1] != 'E' || buf[2] != 'S' || buf[3] != 0x1a){
				return false;
			}
			int cpu_rom_size, ppu_rom_size;
			cpu_rom_size = ((int) (buf[4])) * 0x4000;
			ppu_rom_size = ((int) (buf[5])) * 0x2000;
			if(buf.Length != (HEADER.Length + cpu_rom_size + ppu_rom_size)){
				return false;
			}
			{
				int mappernum = buf[6] >> 4;
				mappernum |= buf[7] & 0xf0;
				m_mappernum = (mapper) mappernum;
				if(m_mappernum == mapper.VRC3){ //やっつけなのでスクリプト実装次第削除
					m_workram = true;
				}
			}
			m_mirror = mirror.Horizonal;
			if((buf[6] & 1) == 1){
				m_mirror = mirror.Vertical;
			}
			if((buf[6] & 0x02) == 0x02){
				SaveArgument t = new SaveArgument();
				t.Name = m_code + " battery backuped RAM";
				t.Length = mmc5.PAGESIZE;
				t.ImageOffset = 0;
				m_save = t;
			}else{
				m_save = null;
			}
			int offset = HEADER.Length;
			m_cpu_rom = new byte [cpu_rom_size];
			m_ppu_rom = new byte [ppu_rom_size];
			Buffer.BlockCopy(buf, offset, m_cpu_rom, 0, cpu_rom_size);
			offset += cpu_rom_size;
			Buffer.BlockCopy(buf, offset, m_ppu_rom, 0, ppu_rom_size);
			
			//hash calculate
			Interface.ImageHash c = romhash_get(m_cpu_rom);
			if(ppu_rom_size == 0){
				m_hash = c;
			}else{
				m_hash = new Interface.ImageHash();
				m_hash.Transform(c.Hash);
				m_hash.Transform(romhash_get(m_ppu_rom).Hash);
				m_hash.Final();
			}
			return true;
		}

		override public void HashListAdd(List<string> ret)
		{
			ret.Add("board.cpu.rom.hash = " + romhash_get(m_cpu_rom).ToString());
			EvaluateRom t = new EvaluateRom();
			t.VbrEvaluate(m_cpu_rom, 8, false);
			t.VbrResultWrite("board.cpu.rom.encode", ret);
			if(m_ppu_rom.Length != 0){
				ret.Add("board.ppu.rom.hash = " + romhash_get(m_ppu_rom).ToString());
				t.CompressEvaluate(m_ppu_rom, true, "board.ppu.rom.encode", ret);
			}
		}

		const int MDC5_PAGE_W_RAM_DISABLE = 0xff;
		override public patchresult PatchManual(RomRecoard.MotorolaSRecoard r, out string log)
		{
			uint offset = r.Address - r.TargetOffset;
			log = "";
			if(m_cpu_rom.Length < offset){
				return patchresult.AREA_NOTFOUND;
			}
			if(m_cpu_rom.Length < (offset + r.Data.Length - 1)){
				return patchresult.AREA_NOTFOUND;
			}
			PatchMemory(r, ref m_cpu_rom, "CPU.ROM 0x{0:x6}", ref log);
			return patchresult.OK;
		}
		override public int RawByte()
		{
			return m_cpu_rom.Length + m_ppu_rom.Length;
		}
		override public bool SavedataSearch(imagestream.W_Ram workram)
		{
			if(m_save != null){
				workram.RomSaveRequest(m_save);
			}
			return true;
		}

		CompressCpuRom m_cpu_rom_stream;
		CompressPpuRom m_ppu_rom_stream;
		int m_cpu_rom_block_offset;
		bool m_cpu_stream_write_done = false, m_ppu_stream_write_done = false;
		bool m_block_write_done = false;
		override public bool StreamMake(imagestream.W_Ram workram, out int blocksize, out int streamsize)
		{
			streamsize = 0;
			blocksize = 0;
			{
				int buffercount = workram.CpuromBufferCount();
				if(m_workram == true){
					if(workram.CpuromWorkRequest() == false){
						return false;
					}
					buffercount -= 1;
				}
				if(m_cpu_encode.HaveStream() == true){
					int pagecount = m_cpu_rom.Length / mmc5.PAGESIZE;
					pagecount--;
					m_cpu_rom_stream = new CompressCpuRom(Math.Min(buffercount, pagecount), m_cpu_encode);
					m_cpu_rom_stream.Compress(m_cpu_rom);
				}else{
					buffercount = 0;
					m_cpu_rom_stream = new CompressCpuRom(buffercount, m_cpu_encode); //dummy
				}
			}
			if(m_ppu_encode.HaveStream() == true){
				m_ppu_rom_stream = new CompressPpuRom(m_ppu_encode);
			}else{ //ここはスクリプトの記載ミス
				EncodeCbr t = new EncodeCbr(true);
				t.BuffersizeSet(0x0800);
				m_ppu_rom_stream = new CompressPpuRom(t);
			}
			m_ppu_rom_stream.Compress(m_ppu_rom);
			streamsize = m_cpu_rom_stream.Length;
			if(m_ppu_rom.Length != 0){
				streamsize += m_ppu_rom_stream.Length;
			}
			blocksize = (m_cpu_rom.Length / mmc5.PAGESIZE) - m_cpu_rom_stream.PackedPageCount;
			Debug.Assert(blocksize >= 1);
			return true;
		}
		
		override public void RomBlockAllocate(ref int offset)
		{
			m_cpu_rom_block_offset = offset;
			offset += (m_cpu_rom.Length / mmc5.PAGESIZE) - m_cpu_rom_stream.PackedPageCount;
		}
		override public void RomStreamAllocate(ref int offset)
		{
			m_cpu_rom_stream.MdcOffset = offset;
			offset += m_cpu_rom_stream.Length;
			if(m_ppu_rom.Length != 0){
				m_ppu_rom_stream.MdcOffset = offset;
				offset += m_ppu_rom_stream.Length;
			}
		}
		override protected void manage_make(imagestream.W_Ram workram, List<byte>manage)
		{
			//0 cpu_rowrom_page_top
			manage.Add((byte) (m_cpu_rom_block_offset | mmc5.PAGE_ROM));
			//1 cpu_rom_page_count
			manage.Add((byte) (m_cpu_rom.Length / mmc5.PAGESIZE));
			//2 cpu rom stream count
			manage.Add((byte) m_cpu_rom_stream.PackedPageCount);
			//3 cpu_bank_size
			manage.Add((byte) mmc5.cpu_bank_mode.cpu_8ace);
			//4 cpu_wram_page
			Queue<byte> bufferram = workram.CpuromBufferGet();
			if(m_save != null){
				int saverampage;
				workram.CpuromSaveRequest(out saverampage);
				manage.Add((byte) saverampage);
			}else if(m_workram == true){
				manage.Add(bufferram.Dequeue());
			}else{
				manage.Add((byte) MDC5_PAGE_W_RAM_DISABLE);
			}
			//5 nametable mirroring
			{
				int n = mmc5.MIRROR_V;
				if(m_mirror == mirror.Horizonal){
					n = mmc5.MIRROR_H;
				}
				manage.Add((byte) n);
			}
			//6 ppu bank size
			{
				mmc5.ppu_bank_mode p = mmc5.ppu_bank_mode.ppu_2000x1;
				if(m_ppu_rom.Length != 0){
					switch(m_mappernum){
					case mapper.SxROM: //SLROM SKROM
						p = mmc5.ppu_bank_mode.ppu_1000x2;
						break;
					case mapper.TxROM: //6bank TLROM TKROM
						p = mmc5.ppu_bank_mode.ppu_0400x8;
						break;
					}
				}
				manage.Add((byte) p);
			}
			//7 ppu rom stream pointer
			m_ppu_rom_stream.StructSet(manage);
			//7+6 cpu rom stream struct
			m_cpu_rom_stream.StructSet(bufferram, manage);
			Debug.Assert(manage.Count == 7 + 7 + 6 + 6*m_cpu_rom_stream.PackedPageCount); //末尾.null assign
		}
		override public bool RomBlockWrite(ref int offset, byte [] rom)
		{
			//m_cpu_rom_block_offset は page 単位なので byte 単位にあわせる
			if((m_cpu_rom_block_offset * mmc5.PAGESIZE) != offset){ 
				return m_block_write_done;
			}
			for(int i = 0; i < m_cpu_rom.Length; i += mmc5.PAGESIZE){
				int romimage_page = i / mmc5.PAGESIZE;
				if(m_cpu_rom_stream.PackedPagenum(romimage_page) == false){
					Buffer.BlockCopy(m_cpu_rom, i, rom, offset, mmc5.PAGESIZE);
					offset += mmc5.PAGESIZE;
				}
			}
			m_block_write_done = true;
			return true;
		}
		override public bool RomStreamWrite(ref int offset, byte [] rom)
		{
			if(m_cpu_rom_stream.Length == 0 && m_ppu_rom_stream.Length == 0){
				return true;
			}
			if(m_cpu_stream_write_done == false){
				m_cpu_stream_write_done = m_cpu_rom_stream.Write(ref offset, rom);
			}
			if(m_ppu_rom_stream.Length == 0){
				m_ppu_stream_write_done = true;
			}else if(m_ppu_stream_write_done == false){
				m_ppu_stream_write_done = m_ppu_rom_stream.Write(ref offset, rom);
				/*if(m_ppu_stream_write_done == true){
					Debug.Assert(m_ppu_rom_stream.Test(m_ppu_rom));
				}*/
			}
			return m_cpu_stream_write_done == true && m_ppu_stream_write_done == true;
		}
	}
}
