using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using hardware;

namespace imagefile{
	abstract class EncodeMap{
		protected struct EncodePage{
			public Archiver.Bpe.Container Data{
				get; set;
			}
			public int Pagenum{
				get; set;
			}
		}
		protected enum type{
			NONE, RAW, CBR, VBR
		};
		readonly type m_type;
		protected const int evaluate_buffer_first = 0x200;
		protected static readonly int [] evaluate_buffer = new int[] {
			0x400, 0x800, 0x1000, 0x2000
		};
		bool buffer_ask(int size)
		{
			if(evaluate_buffer_first == size){
				return true;
			}
			foreach(int t in evaluate_buffer){
				if(t == size){
					return true;
				}
			}
			return false;
		}
		protected EncodeMap(type t)
		{
			m_type = t;
		}
		public abstract bool TypeAsk(string t);
		public abstract bool HaveStream();
		//for VBR
		protected virtual void buffersize_set(int page, int size)
		{
			Debug.Assert(false);
		}
		public bool BuffersizeSet(int page, int size)
		{
			if(buffer_ask(size) == false){
				return false;
			}
			buffersize_set(page, size);
			return true;
		}
		public virtual int Length()
		{
			return 0;
		}
		//for CBR
		protected virtual void buffersize_set(int page)
		{
			Debug.Assert(false);
		}
		public bool BuffersizeSet(int size)
		{
			if(buffer_ask(size) == false){
				return false;
			}
			buffersize_set(size);
			return true;
		}
		public virtual void Encode(byte[] rom, bool use_tail, int limit)
		{
			Debug.Assert(false);
		}
		public abstract bool BuffersizeDone();
		//backlog offset をつけると CBR になるので注意
		//VBR 固定なので backlog offset はつけない
		public virtual void BacklogOffsetSet(int offset)
		{
		}
		public virtual void Write(ref int offset, byte [] rom)
		{
		}
		static public void struct_set(byte page_count, byte org_page, byte dest_page, int offset, List<byte> list)
		{
			//0 1 count
			list.Add(page_count);
			//1 1 もとのデータの本来の page
			list.Add(org_page);
			//2 1 展開されたデータの配置 page
			list.Add(dest_page);
			//3 3 *image
			mmc5.offset_add(offset, list);
		}
		public virtual void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
		}
		public virtual bool PackedPagenum(int page)
		{
			return false;
		}
	}
	class EncodeRaw : EncodeMap{
		public EncodeRaw() : base(type.RAW)
		{
		}
		public override bool TypeAsk(string t)
		{
			return t == "raw";
		}
		public override bool BuffersizeDone()
		{
			return true;
		}
		public override bool HaveStream()
		{
			return false;
		}
	}
	class EncodeVbr : EncodeMap{
		struct Map{
			public int Page{
				get;set;
			}
			public int Size{
				get;set;
			}
		}
		int m_pagesize;
		List<Map> m_buffer = new List<Map>();
		EncodePage [] m_page;
		public EncodeVbr(int pagesize) : base(type.VBR)
		{
			m_pagesize = pagesize;
		}
		public override bool TypeAsk(string t)
		{
			return t == "vbrpage";
		}
		public override bool HaveStream()
		{
			return true;
		}
		public override int Length()
		{
			int size = 0;
			foreach(EncodePage t in m_page){
				size += t.Data.EncodedLength();
			}
			return size;
		}
		protected override void buffersize_set(int page, int size)
		{
			Map t = new Map();
			t.Page = page;
			t.Size = size;
			m_buffer.Add(t);
		}
		protected byte[][] vbr_block_make(byte [] rom, int pagesize, bool use_tail)
		{
			int pagecount = rom.Length / pagesize;
			if(use_tail == false){
				pagecount -= 1;
			}
			byte [][]pagedata = new byte[pagecount][];
			for(int page = 0; page < pagecount; page++){
				pagedata[page] = new byte [pagesize];
				Buffer.BlockCopy(rom, page * pagesize, pagedata[page], 0, pagesize);
			}
			return pagedata;
		}
		public override void Encode(byte[] rom, bool use_tail, int limit)
		{
			byte[][]data = vbr_block_make(rom, mmc5.PAGESIZE, use_tail);
			List<EncodePage> list = new List<EncodePage>();
			Debug.Assert(m_buffer.Count == m_pagesize);
			for(int i = 0; i < Math.Min(limit, m_buffer.Count); i++){
				Map map = m_buffer[i];
				Archiver.Bpe.Container ee = new Archiver.Bpe.Container(map.Size, Archiver.Bpe.Container.type.VBR);
				ee.Encode(data[map.Page], false);
				EncodePage t = new EncodePage();
				t.Pagenum = map.Page;
				t.Data = ee;
				list.Add(t);
			}
			//page 順に並べ直し、bios で pagetable を作りやすくする
			IEnumerable<EncodePage> page_order = (
				from s in list.ToArray()
				orderby s.Pagenum
				select s
			);
			list = new List<EncodePage>(page_order);
			m_page = list.ToArray();
		}
		public override bool BuffersizeDone()
		{
			return m_buffer.Count == m_pagesize;
		}
		public override void Write(ref int offset, byte [] rom)
		{
			foreach(EncodePage t in m_page){
				byte [] au = t.Data.EncodeDataGet();
				au.CopyTo(rom, offset);
				offset += au.Length;
				Debug.Assert(au.Length == t.Data.EncodedLength());
			}
		}
		public override void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
			foreach(EncodePage t in m_page){
				struct_set(1, (byte) t.Pagenum, rombuffer.Dequeue(), offset, list);
				offset += t.Data.EncodedLength();
			}
		}
		public override bool PackedPagenum(int page)
		{
			foreach(EncodePage t in m_page){
				if(page == t.Pagenum){
					return true;
				}
			}
			return false;
		}
	}
	class EncodeCbr : EncodeMap{
		int m_buffer;
		bool m_backstream;
		Archiver.Bpe.Container m_encoder = null;
		public EncodeCbr(bool backstream) : base(type.CBR)
		{
			m_backstream = backstream;
			m_buffer = 0;
		}
		public override bool TypeAsk(string t)
		{
			return t == "cbr";
		}
		public override bool HaveStream()
		{
			return true;
		}
		public override int Length()
		{
			if(m_encoder == null){
				return 0;
			}
			return m_encoder.EncodedLength();
		}
		protected override void buffersize_set(int size)
		{
			m_buffer = size;
		}
		public override bool BuffersizeDone()
		{
			return m_buffer != 0;
		}
		public override void Encode(byte[] rom, bool use_tail, int limit)
		{
			m_encoder = new Archiver.Bpe.Container(m_buffer, Archiver.Bpe.Container.type.CBR);
			m_encoder.Encode(rom, m_backstream);
		}
		public override void BacklogOffsetSet(int offset)
		{
			m_encoder.BacklogOffsetSet(offset);
		}
		public override void Write(ref int offset, byte [] rom)
		{
			byte [] au = m_encoder.EncodeDataGet();
			au.CopyTo(rom, offset);
			offset += au.Length;
			Debug.Assert(au.Length == m_encoder.EncodedLength());
		}
		public override void StructSet(byte pagesize, int offset, Queue<byte> rombuffer, List<byte> list)
		{
			//0, 0 は ppurom しか使わないので定数だが、 cpurom で使うことになれば変数にすること
			struct_set(pagesize, 0, 0, offset, list);
		}
	}

	class EvaluateRom : EncodeVbr{
		const int EVALUATE_LIMIT_NONE = 0;
		int m_vbr_size;
		EncodePage [] m_vbr_page;
		public EvaluateRom(): base(evaluate_buffer_first)
		{
		}
		public void VbrEvaluate(byte [] rom, int limit, bool use_tail)
		{
			byte[][] pagedata = vbr_block_make(rom, mmc5.PAGESIZE, use_tail);
			int pagecount = pagedata.Length;
			List<EncodePage> bpelist = new List<EncodePage>();
			for(int i = 0; i < pagecount; i++){
				Archiver.Bpe.Container ee = new Archiver.Bpe.Container(evaluate_buffer_first, Archiver.Bpe.Container.type.VBR);
				ee.Encode(pagedata[i], false);
				foreach(int buffersize in evaluate_buffer){
					Archiver.Bpe.Container am = new Archiver.Bpe.Container(buffersize, Archiver.Bpe.Container.type.VBR);
					am.Encode(pagedata[i], false);
					if(am.EncodedLength() < ee.EncodedLength()){
						ee = am;
					}
				}
				EncodePage t = new EncodePage();
				t.Pagenum = i;
				t.Data = ee;
				bpelist.Add(t);
			}
			IEnumerable<EncodePage> compress_order = (
				from s in bpelist.ToArray()
				orderby s.Data.EncodedLength()
				select s
			);
			bpelist = new List<EncodePage>(compress_order);
			if((limit != EVALUATE_LIMIT_NONE) && (bpelist.Count > limit)){
				bpelist.RemoveRange(limit, bpelist.Count - limit);
			}
			m_vbr_size = 0;
			m_vbr_page = bpelist.ToArray();
			foreach(EncodePage t in m_vbr_page){
				m_vbr_size += t.Data.EncodedLength();
			}
		}
		public void VbrResultWrite(string type_prefix, List<string> rank)
		{
			string hoe = String.Format(
				"{0}.type = VBR,0x{1:x2}",
				type_prefix, m_vbr_page.Length
			);
			rank.Add(hoe);
			foreach(EncodePage t in m_vbr_page){
				string au = String.Format(
					"{0}.vbrpage = 0x{1:x2},0x{2:x4} #{3:P}",
					type_prefix,
					t.Pagenum, t.Data.BufferSize, 
					(double) t.Data.EncodedLength() / (double) mmc5.PAGESIZE
				);
				rank.Add(au);
			}
		}
/*
board.cpu.rom.encode.type = NONE
board.ppu.rom.encode.type = CBR
board.cpu.rom.encode.cbr = 0x0800 #66.34%
board.cpu.rom.encode.type = VBR,0x10
board.cpu.rom.encode.vbrpage = 0x09,0x1000 #56.91%
*/
		public void CompressEvaluate(byte [] rom, bool tail_use, string type_prefix, List<string> rank)
		{
			if(rom.Length == mmc5.PAGESIZE){
				this.VbrEvaluate(rom, EVALUATE_LIMIT_NONE, tail_use);
				VbrResultWrite(type_prefix, rank);
				return;
			}
			//cbr 最小値を算出
			Archiver.Bpe.Container ee = new Archiver.Bpe.Container(evaluate_buffer_first, Archiver.Bpe.Container.type.CBR);
			ee.Encode(rom, true);
			foreach(int buffersize in evaluate_buffer){
				Archiver.Bpe.Container am = new Archiver.Bpe.Container(buffersize, Archiver.Bpe.Container.type.CBR);
				am.Encode(rom, true);
				if(am.EncodedLength() < ee.EncodedLength()){
					ee = am;
				}
			}
			
			this.VbrEvaluate(rom, EVALUATE_LIMIT_NONE, tail_use);
			if(m_vbr_size < ee.EncodedLength()){
				VbrResultWrite(type_prefix, rank);
			}else{
				rank.Add(type_prefix + ".type = CBR");
				string au = String.Format(
					"{0}.cbr = 0x{1:x4} #{2:P}",
					type_prefix, ee.BufferSize, 
					(double)ee.EncodedLength() / (double)rom.Length
				);
				rank.Add(au);
			}
		}
	}

	class CompressRom{
		protected int m_mdcoffset = 0;
		protected readonly EncodeMap m_map;
		public int MdcOffset{
			set {m_mdcoffset = value;}
		}
		public int Length{
			get {return m_map.Length();}
		}
		protected CompressRom(EncodeMap map)
		{
			m_map = map;
		}
		virtual public bool Write(ref int offset, byte [] rom)
		{
			return true;
		}
		protected bool write(ref int offset, byte [] rom)
		{
			if(m_mdcoffset != offset){
				return false;
			}
			m_map.BacklogOffsetSet(m_mdcoffset);
			m_map.Write(ref offset, rom);
			return true;
		}
	}

	class CompressCpuRom : CompressRom{
		readonly int m_wram_count;
		public int PackedPageCount{
			get {return m_wram_count;}
		}
		public CompressCpuRom(int count, EncodeMap map) : base(map)
		{
			m_wram_count = count;
		}
		public void Compress(byte []rom)
		{
			m_map.Encode(rom, false, m_wram_count);
		}
		public void StructSet(Queue<byte> rombuffer, List<byte> list)
		{
			Debug.Assert(m_mdcoffset != 0);
			m_map.StructSet(1, m_mdcoffset, rombuffer, list);
		}
		public bool PackedPagenum(int page)
		{
			return m_map.PackedPagenum(page);
		}
		override public bool Write(ref int offset, byte [] rom)
		{
			if(this.Length == 0){
				return true;
			}
			return write(ref offset, rom);
		}
	}
	class CompressPpuRom : CompressRom{
		int m_rawsize = 0;
		public CompressPpuRom(EncodeMap map) : base(map)
		{
		}
		public void Compress(byte[] rom)
		{
			m_rawsize = rom.Length;
			if(m_rawsize == 0){
				return;
			}
			m_map.Encode(rom, true, m_rawsize / mmc5.PAGESIZE);
		}
		public void StructSet(List<byte> list)
		{
			Debug.Assert(
				((m_rawsize == 0) && (m_mdcoffset == 0)) ||
				((m_rawsize != 0) && (m_mdcoffset != 0))
			);
			/*Queue<byte> tt = new Queue<byte>();
			for(int i = 0; i < m_rawsize / mmc5.PAGESIZE; i++){
				tt.Enqueue((byte) i);
			}
			m_map.StructSet((byte) (m_rawsize / mmc5.PAGESIZE), m_mdcoffset, tt, list);*/
			EncodeMap.struct_set((byte) (m_rawsize / mmc5.PAGESIZE), 0, 0, m_mdcoffset, list);
		}
		override public bool Write(ref int offset, byte [] rom)
		{
			if(m_rawsize == 0){
				return true;
			}
			return write(ref offset, rom);
		}
		/*public bool Test(byte[] raw)
		{
			return m_encoded_data.DecodeTest(raw);
		}*/
	}
}
