﻿using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;

namespace imagefile{
	abstract class GameImage{
		protected Interface.ImageHash m_hash;
		protected string m_name = "";
		readonly mdc5.Script.imagetype m_type;
		public enum patchresult{
			OK, AREA_TOOMUCH, AREA_NOTFOUND
		};
		//---- property ----
		public string Hash{
			get{return m_hash.ToString();}
		}
		public string Name{
			get {return m_name;}
		}
		public mdc5.Script.imagetype Type{
			get {return m_type;}
		}
		//---- method ----
		public GameImage(mdc5.Script.imagetype type)
		{
			m_type = type;
		}
		abstract public bool Load(mdc5.Script s);
		abstract public string [] HashListGet();
		abstract public byte [] ToRomImage(int pagesize, ref int pageoffset, out int [] sideoffset, out mdc5.SaveArgument [] saveinfo);
		abstract public patchresult PatchManual(RomRecoard.MotorolaSRecoard r, out string log);
		//linker.cs GameImage.script_load で disk と rom を統合するために作成.
		//ROM の場合は呼び出されないので assert が発動する
		virtual public patchresult PatchAuto(out string [] log)
		{
			Debug.Assert(false);
			log = null;
			return patchresult.AREA_NOTFOUND;
		}

		static public void PatchMemory(RomRecoard.MotorolaSRecoard r, ref byte [] memory, string logformat, ref string log)
		{
			uint offset = r.Address - r.TargetOffset;
			string olddata = "", newdata = "";
			for(uint i = 0; i < r.Data.Length; i++){
				olddata += String.Format("${0:x2} ", memory[offset + i]);
				newdata += String.Format("${0:x2} ", r.Data[i]);
				memory[offset + i] = r.Data[i];
			}
			log += String.Format(logformat + "/${1:x4}: {2}->{3}", offset, r.Address, olddata, newdata);
		}
	}
	class Nes : GameImage{
		static readonly byte [] HEADER = {
			(byte)'N', (byte)'E', (byte)'S', 0x1a, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0
		};
		byte [] m_cpu_rom, m_ppu_rom;
		int m_cpu_ram_size;
		mirror m_mirror;
		mapper m_mappernum;
		public enum mapper : byte{
			NROM = 0, SxROM, UxROM, CxROM, TxROM, ExROM,
			VRC3 = 73
		};
		enum mirror{ Programmable, Horizonal, Vertical };
		//---- ROM size constant ----
		const int MBIT = 0x20000;
		static readonly int [] CPU_ROMSIZE = {
			0x4000, 0x8000, 0x10000,
			1 * MBIT, 2 * MBIT, 4 * MBIT, 8 * MBIT
		};
		static readonly int [] PPU_ROMSIZE = {
			0, 0x2000, 0x4000, 0x8000, 0x10000,
			1 * MBIT, 2 * MBIT, 4 * MBIT, 8 * MBIT
		};
		//---- method ----
		public Nes() : base(mdc5.Script.imagetype.rom)
		{
		}
		bool rom_set(int [] oksize, byte [] input, out byte [] output){
			foreach(int t in oksize){
				if(t == input.Length){
					//output = new byte [t];
					//input.CopyTo(output, 0);
					output = input;
					return true;
				}
			}
			output = new byte [0];
			return false;
		}
		static readonly int [] CPU_RAMSIZE = {
			0, 
			0x2000, //6264x1:[ST]NROM, TSROM, [STE][KL]ROM
			0x4000*2, //6264x2:SOROM, ETROM
			0x8000, //62256x1: SXROM, EWROM
			//0x8000 + 0x2000, //ETROM for MDC5 personal NES ヘッダ規定外なのでエミュレータ上では debug を使う
			0x8000*2 //62256x2: ETROM for MDC5 debug
		};
		bool ram_set(int [] oksize, int input, out int output){
			foreach(int t in oksize){
				if(t == input){
					output = t;
					return true;
				}
			}
			output = 0;
			return false;
		}
		public bool ImageSet(byte [] cpu_rom, int cpu_ram_size, byte [] ppu_rom, mapper mappernum)
		{
			if(rom_set(CPU_ROMSIZE, cpu_rom, out m_cpu_rom) == false){
				return false;
			}
			if(ram_set(CPU_RAMSIZE, cpu_ram_size, out m_cpu_ram_size) == false){
				return false;
			}
			if(rom_set(PPU_ROMSIZE, ppu_rom, out m_ppu_rom) == false){
				return false;
			}
			m_mappernum = mappernum;
			return true;
		}
		int header_set(int cpu_rom_size, int cpu_ram_size, int ppu_rom_size, int mappernum, out byte [] header){
			header = HEADER;
			header[4] = (byte) (cpu_rom_size / 0x4000);
			header[5] = (byte) (ppu_rom_size / 0x2000);
			/*if(cpu_ram_size != 0){
				//battery-backup flag set
				//これは RAMSIZE とは直結しないのだが、便宜上これで.
				header[6] = 2;
			}else{
				header[6] = 0;
			}*/
			header[6] = 0;
			header[6] |= (byte) (mappernum << 4); //byte 型での shift 演算を受け付けないので mappernum の型を int にした
			header[8] = (byte) (cpu_ram_size / 0x2000);
			return header.Length + cpu_rom_size + ppu_rom_size;
		}
		public byte [] ImageGet(){
			byte [] header;
			int imagesize = header_set(
				m_cpu_rom.Length, m_cpu_ram_size, m_ppu_rom.Length, 
				(int) m_mappernum, out header
			);
			byte [] image = new byte[imagesize];
			int offset = 0;
			
			header.CopyTo(image, offset);
			offset += header.Length;
			m_cpu_rom.CopyTo(image, offset);
			offset += m_cpu_rom.Length;
			m_ppu_rom.CopyTo(image, offset);
			offset += m_ppu_rom.Length;
			return image;
		}
		Interface.ImageHash romhash_get(byte [] buf)
		{
			Interface.ImageHash c = new Interface.ImageHash();
			c.Transform(buf);
			c.Final();
			return c;
		}
		override public bool Load(mdc5.Script s)
		{
			m_name = s.BiosName;
			return Load(s.ImageFilename);
		}
		public bool Load(string file)
		{
			byte [] buf;
			if(Static.Utility.binary_load(file, out buf) == false){
				return false;
			}
			if(buf.Length < HEADER.Length){
				return false;
			}
			if(buf[0] != 'N' || buf[1] != 'E' || buf[2] != 'S' || buf[3] != 0x1a){
				return false;
			}
			int cpu_rom_size, ppu_rom_size;
			cpu_rom_size = ((int) (buf[4])) * 0x4000;
			ppu_rom_size = ((int) (buf[5])) * 0x2000;
			if(buf.Length != (HEADER.Length + cpu_rom_size + ppu_rom_size)){
				return false;
			}
			{
				int mappernum = buf[6] >> 4;
				mappernum |= buf[7] & 0xf0;
				m_mappernum = (mapper) mappernum;
			}
			m_mirror = mirror.Horizonal;
			if((buf[6] & 1) == 1){
				m_mirror = mirror.Vertical;
			}
			int offset = HEADER.Length;
			m_cpu_rom = new byte [cpu_rom_size];
			m_ppu_rom = new byte [ppu_rom_size];
			Buffer.BlockCopy(buf, offset, m_cpu_rom, 0, cpu_rom_size);
			offset += cpu_rom_size;
			Buffer.BlockCopy(buf, offset, m_ppu_rom, 0, ppu_rom_size);
			
			//hash calculate
			Interface.ImageHash c = romhash_get(m_cpu_rom);
			if(ppu_rom_size == 0){
				m_hash = c;
			}else{
				m_hash = new Interface.ImageHash();
				m_hash.Transform(c.Hash);
				m_hash.Transform(romhash_get(m_ppu_rom).Hash);
				m_hash.Final();
			}
			return true;
		}
		override public string [] HashListGet()
		{
			List<string> ret = new List<string>();
			ret.Add("board.cpu.rom.hash = " + romhash_get(m_cpu_rom).ToString());
			if(m_ppu_rom.Length != 0){
				ret.Add("board.ppu.rom.hash = " + romhash_get(m_ppu_rom).ToString());
			}
			return ret.ToArray();
		}

		//disk における side offset に ROM の設定項目を入れる
		const int MMC5_MIRROR_H = 0x50;
		const int MMC5_MIRROR_V = 0x44;
		const int MMC5_PAGE_ROM = 0x80;
		const int MMC5_PAGE_W_RAM_0 = 0; //から3まで.battery backup
		const int MMC5_PAGE_W_RAM_1 = 4; //から7まで
		const int MDC5_PAGE_W_RAM_DISABLE = 0xff;
		//0x4000 以上のモードは page 設定時に alignment 制限があり、
		//ROMを柔軟に押し込めないので 0x2000x4 = 8ace mode を使用する
		enum cpu_bank_mode : int {
			cpu_8888 = 0, cpu_88cc, cpu_88ce, cpu_8ace
		};
		enum ppu_bank_mode : int {
			ppu_2000x1 = 0, ppu_1000x2, ppu_0800x4, ppu_0x0400x8
		};
		override public byte [] ToRomImage(int pagesize, ref int pageoffset, out int [] sideoffset, out mdc5.SaveArgument [] saveinfo)
		{
			List<int> config = new List<int>();
			//0: pageoffset
			config.Add(pageoffset | MMC5_PAGE_ROM); 
			//1: CPU ROM pagenum (pagesize is 0x2000)
			config.Add(m_cpu_rom.Length / pagesize);
			//2: PPU charcter ROM pagenum (pagesize is 0x2000)
			//0 is charcter RAM!!
			config.Add(m_ppu_rom.Length / pagesize);
			//3: mirroring
			config.Add(m_mirror == mirror.Horizonal ? MMC5_MIRROR_H : MMC5_MIRROR_V);
			//4: memory page for $6000-
			//SNROM, TNROM を動かすとすれば W-RAM-0 の page 1 とかでいいと思うけど、管理の対応がいるし、ハードウェアのRAMの容量を増やす必要がある
			int w_ram_page;
			switch(m_mappernum){
			case mapper.SxROM:
			case mapper.TxROM: //for TSROM. TNROM なら W_RAM_0 の 1 以降
			case mapper.VRC3:
				w_ram_page = MMC5_PAGE_W_RAM_1;
				break;
			default:
				w_ram_page = MDC5_PAGE_W_RAM_DISABLE;
				break;
			}
			config.Add(w_ram_page);
			//5: MMC5 cpu bank mode
			config.Add((int) cpu_bank_mode.cpu_8ace);
			//6: MMC5 ppu bank mode
			ppu_bank_mode p;
			switch(m_mappernum){
			case mapper.SxROM: //SGROM SNROM
				p = ppu_bank_mode.ppu_1000x2;
				break;
			case mapper.TxROM: //6個だったよな...
				p = ppu_bank_mode.ppu_0x0400x8;
				break;
			default: //NROM, CxROM, UxROM, VRCIII, SUNSOFT-1
				p = ppu_bank_mode.ppu_2000x1;
				break;
			}
			config.Add((int) p);
			sideoffset = config.ToArray();

			//ROM data copy. 単純につなげて渡すだけ
			int offset = 0;
			byte [] rom = new byte[m_cpu_rom.Length + m_ppu_rom.Length];			pageoffset += rom.Length / pagesize;
			Buffer.BlockCopy(m_cpu_rom, 0, rom, offset, m_cpu_rom.Length);
			offset += m_cpu_rom.Length;
			Buffer.BlockCopy(m_ppu_rom, 0, rom, offset, m_ppu_rom.Length);
			//まじにやるならここも直すこと
			saveinfo = new mdc5.SaveArgument[0];
			return rom;
		}
		override public patchresult PatchManual(RomRecoard.MotorolaSRecoard r, out string log)
		{
			uint offset = r.Address - r.TargetOffset;
			log = "";
			if(m_cpu_rom.Length < offset){
				return patchresult.AREA_NOTFOUND;
			}
			if(m_cpu_rom.Length < (offset + r.Data.Length - 1)){
				return patchresult.AREA_NOTFOUND;
			}
			PatchMemory(r, ref m_cpu_rom, "CPU.ROM 0x{0:x6}", ref log);
			return patchresult.OK;
		}
	}
}

