/*
 * dsk324sr.c
 *
 *  Created on: 8/05/2021
 *      Author: alexrayne <alexraynepe196@gmail.com>
 */

#include <c_compat.h>
#include "dsk324sr.h"
#include "r_i2c_master_api.h"
#include "board.h"
#include <sys/pt.h>
#include <project-conf.h>

// provide: TM_YEAR0
#include <OsTime.h>


#if 1
#include <assert.h>
#define ASSERT(...) assert( __VA_ARGS__ )
#else
#define ASSERT(...)
#endif



////////////////////////////////////////////////////////////////////////////////////
#include "sys/log.h"
LOG_MODULE_LOCAL("dsk324");
#ifdef LOG_LEVEL_DSK324
#define LOG_LEVEL LOG_LEVEL_DSK324
#else
#define LOG_LEVEL LOG_LEVEL_NONE
#endif

#include <trace_probes.h>
#ifndef trace_rtc_read
trace_need(rtc_valid);
trace_need(rtc_read);
trace_need(rtc_set);
trace_need(rtc_invalidate);
trace_need(rtc_i2c);
#endif


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///

#ifndef RTC_DSK324_I2C
#error  "need DSK324SR rtc i2c-master"
#endif


enum DSK324RtcOperation{
    //< when initialised, and ready to comm
    DSK324RTC_IDLE,
    DSK324RTC_READY = DSK324RTC_IDLE,

    //< when powerup, but not valid
    DSK324RTC_NONE,
    DSK324RTC_READ,
    DSK324RTC_START,
    // setup rtc mode only
    DSK324RTC_REINIT,
    // cleanup rtc and mode
    DSK324RTC_RESET,
    DSK324RTC_SET,
};
typedef enum DSK324RtcOperation    DSK324RtcOperation;

struct  DSK324RTCtx {

    //< this false while I2C activity
    volatile  bool  valid_regs;

    DSK324RtcOperation     op;

    struct pt       pt;
};
typedef struct  DSK324RTCtx DSK324RTCtx;

typedef enum DSK324CtxState DSK324CtxState;
typedef enum DSK324TimeState DSK324TimeState;


DSK324Ctx       rtc_dsk324_ctx;
DSK324CtxState  rtc_dsk324_valid;
DSK324TimeState rtc_dsk324_time_status;
struct tm       rtc_dsk324_now;

DSK324Ctx   rtc_dsk324_last;
i2cHandle   rtc_dsk324_port;
DSK324RTCtx rtc_dsk324;



///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///
#if 1

#define SELFARG
#define SELF        (rtc_dsk324)

#define API(method) (g_i2c_master_on_iic.method)

#else
// system use multiple rtc
#define SELFARG     DSK324RTCtx* self,
#define SELF        (*self)

#define API(method) (*api->method)

#endif



#ifndef CODE_ISR
#ifdef __HOTFUNC
#define CODE_ISR    __HOTFUNC
#else
#define CODE_ISR
#endif
#endif



///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///
#include "lib/bcdnum.h"

// после обновления содержимого RTC, надо пересчитать структуру rtc_dsk324_now
static
void rtc_dsk324_updated(void){
    DSK324Ctx*  regs = &rtc_dsk324_ctx;
    struct tm*  x = &rtc_dsk324_now;

    if ( memcmp(regs->reg, rtc_dsk324_last.reg, DSK324REG_DATETIME_TOTAL) != 0 ){
        x->tm_sec   = bcd2num(regs->sec);
        x->tm_min   = bcd2num(regs->min);
        x->tm_hour  = bcd2num(regs->hour);
        x->tm_mday  = bcd2num(regs->day)+1;
        x->tm_mon   = bcd2num(regs->month);
        x->tm_year  = bcd2num(regs->year) + RTC_DSK324_YEAR0 - TM_YEAR0;
        x->tm_wday  = regs->week;
        x->tm_isdst = (regs->ctrl & DSK324CTRL_RAM) != 0;
        x->tm_yday  = 0; // mktime ignore it
    }
    rtc_dsk324_last = *regs;
}



static
void rtc_dsk324_clear(void){
    DSK324Ctx*  regs = &rtc_dsk324_ctx;
    memset(regs->reg, 0, sizeof(regs->reg) );
    rtc_dsk324_updated();
}



//  всякое назначение состояния SELF.op требует работы. //уведомлю приложение что dsk324 хочет работать
static
void dsk324sr_poll(void) {
    rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_BUSY);
}



CODE_ISR
void dsk324sr_i2c_callback_dummy(i2cHandle h, i2c_master_event_t ev, void* ctx)
{
    (void)h;(void)ev;(void)ctx;
    trace_rtc_i2c_twist();

    //rtc_dsk324_process();
    dsk324sr_poll();
}

/// @brief обработчик приложения событий от прерывания i2c на dsk324.
///         пользовательский код может сделать свою реализацию.
///         Необходимо чтобы на события i2c вызывался rtc_dsk324_process
void dsk324sr_i2c_callback(i2cHandle h, i2c_master_event_t ev, void* ctx)
    __WEAK_REF("dsk324sr_i2c_callback_dummy");


void dsk324_aquire( void ){
    //DSK324Ctx* regs = &rtc_dsk324_ctx;
    //fsp_err_t ok;

    I2C_assign_handler( rtc_dsk324_port, &(dsk324sr_i2c_callback), NULL);
}

void dsk324_on_event(i2cHandle port, DSK324EventID ev){
    (void)port; (void)ev;
}

/// @brief обработчик приложения событий от dsk324. Вызывается из rtc_dsk324_process
///         пользовательский код может сделать свою реализацию
void rtc_dsk324_on_event(i2cHandle port, DSK324EventID ev)
    __WEAK_REF("dsk324_on_event");



PT_THREAD ( rtc_dsk324_process(void) ){
    struct pt* pt = &(SELF.pt);
    DSK324Ctx*  regs = &rtc_dsk324_ctx;
    fsp_err_t   ok;
    I2C_Status  i2cok;

    PT_BEGIN( pt )

rtc_dsk324_process_again:

    if (SELF.op == DSK324RTC_IDLE){
        if (rtc_dsk324_valid > DSK324CTX_INVALID)
            PT_EXIT(pt);
        LOG_DBG("invalidated refresh\n");
        SELF.op = DSK324RTC_READ;
    }

    if (SELF.op == DSK324RTC_READ){
        LOG_DBG("read\n");
rtc_dsk324_process_read:

        trace_rtc_read_on();
        rtc_dsk324_valid = DSK324CTX_UPDATE;
        trace_rtc_valid_off();

        // запуск команды чтения часов
        PT_WAIT_UNTIL(pt, ( ok = I2C_status(rtc_dsk324_port)) != I2C_BUSY );
        dsk324_aquire();

        // post adress for reading
        SELF.op = DSK324RTC_READ;

        SELF.valid_regs  = false;
        ok = I2C_read_regs(rtc_dsk324_port, DSK324ADDR, 0, regs, DSK324REG_TOTAL);

        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );

        if ( rtc_dsk324_valid < DSK324CTX_UPDATE ){
            // invalidated via was restart diuring read
            //continue;
            //PT_YIELD(pt);
            LOG_DBG("read retry\n");
            goto rtc_dsk324_process_read;
        }

        if (i2cok == I2C_OK){

            SELF.valid_regs  = true;
            rtc_dsk324_valid        = DSK324CTX_VALID;
            trace_rtc_valid_on();

            if ( (regs->flag & DSK324FLAG_VDHF) == 0)
                rtc_dsk324_time_status   = DSK324_TIME_VALID;
            else
                rtc_dsk324_time_status   = DSK324_TIME_COLD;

            if ( (regs->flag & DSK324FLAG_VDLF) == 0){
                SELF.op = DSK324RTC_IDLE;

                rtc_dsk324_updated();
                rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_AFTER_REFRESH);

                goto rtc_dsk324_process_again;

            }
            else {

                // rtc was power-up.
                LOG_WARN("rtc power-up, now default time\n");

                // provide default time from now
                rtc_dsk324_clear();
                SELF.op = DSK324RTC_RESET;

                rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_POWERUP);
            }
        }
        else {
            //?? try again later
            //SELF.op = DSK324RTC_IDLE;
            PT_EXIT(pt);
        }
    }
    trace_rtc_read_off();

    if (SELF.op == DSK324RTC_START)
    {
        LOG_DBG("start\n");

        PT_WAIT_UNTIL(pt, ( ok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );
        dsk324_aquire();

        regs->sel   = DSK324SEL_CFS_1HZ | DSK324SEL_TCS_2HZ | DSK324SEL_UTS_SEC;
        rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_PREPARE_RESET);

        ok = I2C_write_regs(rtc_dsk324_port, DSK324ADDR, DSK324REG_SELECT, &regs->sel, 1);
        ASSERT(ok == FSP_SUCCESS);
        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );

        if ( UNLIKELY(i2cok != I2C_OK) ) {
            //?? try again later
            LOG_ERR("start fails %d\n", i2cok);
            SELF.op = DSK324RTC_IDLE;
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_ERROR_INIT);
            PT_EXIT(pt);
        }

        // посмотрю статус холодного старта чипа.
        //  загружу и control - чтобы проверить DSK324CTRL_UTIE
        ok = I2C_read_next(rtc_dsk324_port, &regs->flag, 2);
        ASSERT(ok == FSP_SUCCESS);
        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );

        if ( UNLIKELY(i2cok != I2C_OK) ) {
            //?? try again later
            LOG_ERR("start fails %d\n", i2cok);
            SELF.op = DSK324RTC_IDLE;
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_ERROR_INIT);
            PT_EXIT(pt);
        }

        if ( (regs->flag & DSK324FLAG_VDLF) == 0){
            rtc_dsk324_time_status   = DSK324_TIME_VALID;

            if ((regs->ctrl & DSK324CTRL_UTIE) != 0){
                // прерывание от часов уже включено, нормальноо работаем
                rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_AFTER_INIT);
                SELF.op = DSK324RTC_READ;
                goto rtc_dsk324_process_read;
            }

            // надо включить прерывание от часов, посему пойдем от полного ресета
            SELF.op = DSK324RTC_REINIT;
        }
        else {
            SELF.op = DSK324RTC_RESET;
            LOG_WARN("start cold\n");
        }
    }

    if (SELF.op == DSK324RTC_REINIT)
    {
        LOG_DBG("reinit\n");

        regs->sel   = DSK324SEL_CFS_1HZ | DSK324SEL_TCS_2HZ | DSK324SEL_UTS_SEC;
        // keep power up status
        regs->flag  = DSK324FLAG_VDLF | DSK324FLAG_VDHF;
        // profide 1sec irq on RTC_IRQ
        regs->ctrl  =  (regs->ctrl & DSK324CTRL_RAM) | DSK324CTRL_UTIE;
        rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_PREPARE_RESET);

        ok = I2C_write_regs(rtc_dsk324_port, DSK324ADDR, DSK324REG_SELECT, &regs->sel, 3);
        ASSERT(ok == FSP_SUCCESS);
        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );

        if (i2cok == I2C_OK){
            SELF.op = DSK324RTC_READ;
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_AFTER_INIT);
            goto rtc_dsk324_process_read;
        }
        else {
            //?? try again later
            LOG_ERR("reinit fails %d\n", i2cok);
            SELF.op = DSK324RTC_IDLE;
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_ERROR_INIT);
            PT_EXIT(pt);
        }
    }

    if (SELF.op == DSK324RTC_RESET){
        LOG_DBG("reset clear\n");

        rtc_dsk324_valid        = DSK324CTX_UPDATE;
        rtc_dsk324_clear();
        rtc_dsk324_time_status   = DSK324_TIME_COLD;
        regs->flag  = DSK324FLAG_VDHF;
        rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_POWERUP);

        regs->sel   = DSK324SEL_CFS_1HZ | DSK324SEL_TCS_2HZ | DSK324SEL_UTS_SEC;
        // profide 1sec irq on RTC_IRQ
        regs->ctrl  =  DSK324CTRL_UTIE;
        rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_PREPARE_RESET);

        PT_WAIT_UNTIL(pt, ( ok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );
        dsk324_aquire();

        ok = I2C_write_regs(rtc_dsk324_port, DSK324ADDR, 0, &regs->reg, DSK324REG_TOTAL);
        ASSERT(ok == FSP_SUCCESS);
        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );

        if (i2cok == I2C_OK){
            rtc_dsk324_valid        = DSK324CTX_VALID;
            rtc_dsk324_updated();
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_AFTER_INIT);
        }
        else {
            //?? try again later
            LOG_ERR("reset fails %d\n", i2cok);
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_ERROR_INIT);
        }
        SELF.op = DSK324RTC_IDLE;
        PT_EXIT(pt);
    }
    else
    if (SELF.op == DSK324RTC_SET){
        LOG_DBG("set:\n");

rtc_dsk324_process_reset:
        trace_rtc_set_on();

        PT_WAIT_UNTIL(pt, ( ok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );
        dsk324_aquire();

        rtc_dsk324_valid = DSK324CTX_UPDATE;
        regs->flag  &= ~DSK324FLAG_VDHF;
        ok = I2C_write_regs(rtc_dsk324_port, DSK324ADDR, 0, &regs->reg, DSK324REG_TOTAL);
        ASSERT(ok == FSP_SUCCESS);

        PT_WAIT_UNTIL(pt, ( i2cok = I2C_wait(rtc_dsk324_port)) != I2C_BUSY );
        if (i2cok == I2C_OK){
            if ( rtc_dsk324_valid < DSK324CTX_UPDATE ){
                LOG_DBG("set again\n");
                // invalidate during setup. resetup again
                //continue;
                //PT_YIELD(pt);
                goto rtc_dsk324_process_reset;
            }
            rtc_dsk324_time_status  = DSK324_TIME_VALID;
            rtc_dsk324_valid        = DSK324CTX_VALID;
            rtc_dsk324_updated();
            SELF.op = DSK324RTC_IDLE;
            rtc_dsk324_on_event(rtc_dsk324_port, DSK324EV_AFTER_STORE);
        }
        else {
            //?? try again later
            LOG_ERR("set fails %d\n", i2cok);
            SELF.op = DSK324RTC_IDLE;
        }
    }
    else {
        LOG_ERR("uncknown op%d\n", SELF.op);
        SELF.op = DSK324RTC_IDLE;
    }
    trace_rtc_set_off();

    if (rtc_dsk324_valid <= DSK324CTX_INVALID){
        LOG_DBG("valid regs %d:\n", SELF.valid_regs);
        SELF.op = DSK324RTC_READ;
        goto rtc_dsk324_process_read;
    }

    PT_END( pt );
}



/// @brief invalidates rtc_dsk324_ctx, leading to refresh.
///         it may informed on RTC 1sec int
void        rtc_dsk324_invalidate(void){
    rtc_dsk324_valid = DSK324CTX_INVALID;
    trace_rtc_invalidate_twist();
    trace_rtc_valid_off();

    //уведомлю приложение что dsk324 требует работы
    dsk324sr_poll();
}



fsp_err_t rtc_dsk324_refresh( void ){
    DSK324Ctx* regs = &rtc_dsk324_ctx;
    fsp_err_t ok;

    if (SELF.op != DSK324RTC_IDLE)
        return RTC_BUSY;

    dsk324_aquire();

    // post adress for reading
    SELF.op = DSK324RTC_READ;
    ok = I2C_read_regs(rtc_dsk324_port, DSK324ADDR, 0, regs, DSK324REG_TOTAL);

    ASSERT( ok == FSP_SUCCESS );
    return ok;
}



void rtc_dsk324_init(i2cHandle h) {
    (void)h;
    //DSK324Ctx* regs = &rtc_dsk324_ctx;
    //fsp_err_t ok;
    struct pt* pt = &(SELF.pt);

    rtc_dsk324_port = h;
    rtc_dsk324_invalidate();
    rtc_dsk324_time_status   = DSK324_TIME_UNCKNOWN;

    SELF.op = DSK324RTC_START; //DSK324RTC_REINIT;

    PT_INIT(pt);
    dsk324sr_poll();
}


RTCStatus_t rtc_dsk324_wait(void){
    if ( PT_SCHEDULE(rtc_dsk324_process() ) )
        return RTC_BUSY;
    return RTC_OK;
}


static bool rtc_valid_tm(const struct tm* p_time);

RTCStatus_t rtc_dsk324_CalendarTimeSet(const struct tm* p_time){
    if (SELF.op != DSK324RTC_IDLE)
        return RTC_BUSY;

    if (!rtc_valid_tm(p_time))
        return FSP_ERR_INVALID_ARGUMENT;

    DSK324Ctx*  regs = &rtc_dsk324_ctx;
    const struct tm*  x = p_time;

    regs->sec   = num2bcd(x->tm_sec);
    regs->min   = num2bcd(x->tm_min);
    regs->hour  = num2bcd(x->tm_hour);
    regs->day   = num2bcd(x->tm_mday-1);
    regs->month = num2bcd(x->tm_mon-1);
    regs->year  = num2bcd(x->tm_year + TM_YEAR0 - RTC_DSK324_YEAR0);
    regs->week  = x->tm_wday;
    if (x->tm_isdst > 0)
        regs->ctrl |= DSK324CTRL_RAM;
    else
        regs->ctrl &= ~DSK324CTRL_RAM;

    // маркирую валидность времени в часах
    regs->flag &= ~DSK324FLAG_VDHF;

    SELF.op = DSK324RTC_SET;
    dsk324sr_poll();
    return RTC_OK;
}

static
bool rtc_valid_tm(const struct tm* x){
    return (x->tm_sec < 60)
          && (x->tm_min < 60)
          && (x->tm_hour < 24)
          && (x->tm_mday <= 31) && (x->tm_mday > 0)
          && (x->tm_mon > 0) && (x->tm_mon <= 12)
          && (x->tm_year >= (RTC_DSK324_YEAR0-TM_YEAR0)) && (x->tm_year <= ( (RTC_DSK324_YEAR0+99)-TM_YEAR0))
          && (x->tm_wday < 7)
          ;
}
