/*
 * ru utf8
  *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------

  АПИ драйвера UART.
  предоставляется драйвер с буферами приема/передачи. реализующий базовый АПИ
  HAL_IO_Device:  read/writeByte/Data
  неблокирующие вызовы для АПИ StdIO_Device, HAL_INBlock_Device:
      post/getData
  И функции ожидания для АПИ StdIO_Device, HAL_INBlock_Device:
      put_wait/flush,get_wait

  Драйвер дает открытый доступ к буфферам rx/tx.
  так же предоставляются функции работы с буферами :
      - rx_drop   очистка приемника
      - rx_wait - ожидание приема
      - tx_wait - ожидание опустошения tx
      - tx_busy - ожидание освобождения передатчика
  эти функции аппаратно-специфичны, и вероятно реализуются конкретным драйвером

  методы SendData/send_char - реализуют аппаратнно-зависимую неблокирующую отправку.
      конкретный драйвер УАРТа, вероятно их переопределяет их.
  функции отправки write/postXXX пользуются SendData/send_char, и потребуют
      переписывания под платформу только для ускорения.

  для обработки ситуации переполнения буффера (или ошибки на линии) предоставляется
  метод on_rx_over
 * */
#ifndef HAL_UART_H
#define HAL_UART_H

#include <project-conf.h>
#include "hal_device.h"
#include "lib/hal_ringbuffer.h"

#include <OsSync.h>


// размер встроенных в UART_BUFDevice буфферов задаю этим макро
#ifndef UART_BUFFER_SIZE
#define UART_BUFFER_SIZE 1000
#endif



class UART_BUFDevice
    : public Power_Manager
    //, public HAL_Device
    , public HAL_IO_Device
    //, public HAL_OUTBlock_Device
    , public StdIO_Device
    , public HAL_INBlock_Device
{
public:
    UART_BUFDevice();

    //  HAL_IO_Device API
    virtual int IOCtrl(ioctl_string cmd);
    virtual int readByte(u8 *data);
    virtual int writeByte(u8 data);
    virtual int readData(u8 *data, size_t size);
    virtual int writeData(const u8 *data, size_t size);
    virtual int writeData(const char *data);

    // Power_Manager API
    virtual int setPower(POWER_STATE state);

    // УАРТ имеет свойство соединенности. в несоединенном состоянии методы
    //  некоторые write/read будут возвращат DEV_FAIL
    void setConnected(bool conn){
        connected = conn;
    };

public:
    HAL_RingStore<(UART_BUFFER_SIZE)> rx;
    HAL_RingStore<(UART_BUFFER_SIZE)> tx;

    // WaitResult использую в функция ожидания для протониток.
    //  \return 0 - на время ожидания
    //  \return < 0 - по таймауту или ошибке
    //  \return > 0 - количество оступных данных
    typedef int WaitResult;

    // ожидание поступления символов в приемник
    virtual WaitResult rx_wait(unsigned to = toInfinite);

    // очистка буффера приемника
    void rx_drop(){rx.Drop();};

    // пользовательский обработчик переполнения входного буфера
    typedef void (*rx_over_event)(void* arg, UART_BUFDevice* uart, unsigned c);
    // назначает пользовательский обработчик
    void    handle_rx_over(rx_over_event handle, void* arg){
        ev_rx_over  = handle;
        rx_over_arg = arg;
    };

    // there HW specific TX completion API
    //  check tx is active
    virtual bool tx_busy() = 0;

    //  invokes wait until tx is done and inactive
    virtual WaitResult tx_wait() = 0;

public:
    //*******             stdout_device       ***************
    //* блокирующая печать
    virtual int putChar(int ch);
    virtual int puts( const char* str);
    //*  \return - длинна отправленного участка
    virtual int putData ( const void* str, unsigned len);
    //*  неблокирующая печать
    virtual int postData ( const void* str, unsigned len);

    // функции ожидания ждут поллингом содержимого порта с некоторым достаточно
    //  малым периодом (надеюсь что поллинг даст немного надежности от зависания)
    static const unsigned TO_Poll = 5;

    //*  ожидание доступности печати.
    //      ожидается возвращение результата при изменении заполнения буфера.
    //      в реальности возвращает результат при опустошении буфера, по прерыванию
    //      пеердатчика.
    // \return 0            - until waiting
    // \return DEV_TIME_OUT - when nothing on timeout
    //*  \return - количество байт возможных для неблокирующей печати
    virtual WaitResult put_wait(unsigned to = TO_Poll);
    //*  почти тоже put_wait, ждет полного опустошения
    virtual WaitResult put_flush(unsigned to = toInfinite);

    //*  очищает буфер, прерывая текущую отправку
    virtual int put_drop();

    //* монополизация вывода. Конкурирующие нитки могут эксклюзивно разделять
    //      доступ к УАРТ этим методом
    //* \arg onoff - захват/освобождение
    //* \return    - состояние захвачн ли вывод
    virtual bool put_access(bool onoff, unsigned to = toInfinite);

    class GuardPut{
        public:
            typedef UART_BUFDevice io_t;
            io_t* self;
            GuardPut(io_t* x, unsigned to = toInfinite){
                self = x;
                x->put_access(true, to);
            };
            ~GuardPut(){
                self->put_access(false);
            }
    };

public:
    //*******             stdin_device       ***************
    virtual int get_char();
    // \return 0            - until waiting
    // \return DEV_TIME_OUT - when nothing on timeout
    //  !!! блокирующее
    virtual WaitResult get_wait(unsigned to = toInfinite);

public:
    //*******             HAL_INBlock_Device       ***************
    static const unsigned TO_PollRXIRQ = 100;
    // блокирующий прием
    //*  \return - длинна полученного участка
    virtual int getData ( void* dst, unsigned len);
    //*  ожидание amount доступных символов. !!! блокирующее
    virtual WaitResult get_waitfor(unsigned amount, unsigned to = toInfinite);

protected:
    BinSemaphore    tx_empty;
    Mutex           tx_access;
    BinSemaphore    rx_avail;
    // это счетчик ожидаемых к приему байт, перед тем как прерывание приемника
    //  просигналит в rx_avail наличие данных.
    // если мы заранее знаем сколько нужно данных, может уменьшить количество
    //  преждевременных сигналов.
    int             rx_sigcount;
    //* позволяет форсировать сигнал rx_avail при непустом буфере
    //*   с этим периодом
    static const unsigned rx_sig_force_to = 4;

    void init(void);

    bool      connected;
    // красивый псевдоним connected
    bool      is_ready()const {return connected;};

    // this is low-level API for HW implement of data send
    // \arg data_size == 0 валидный вызов, дожен уверисться/инициировать отправку
    //      буффера tx в порт.
    virtual int SendData(const void *data, const u32 data_size) =0;
    virtual int send_char(int x) = 0;

    // поведение при заполнени входного буфера может быть разным
    // по дефолту отбрасываю голову буфера. Это идентично поведению при
    //  загрузке буфера по ДМА
    rx_over_event    ev_rx_over;
    void*            rx_over_arg;
    virtual void on_rx_over(unsigned c);
};




#endif // V12_UART_H
