/*
 * timer_hal.hpp
 *
 *  Created on: 28 нояб. 2018 г.
  *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
     АПИ аппраратного таймера.
     включает:
         управление предделитетем/источником
         управление периодом, знаением счетчика
         настройку компаратора/ов
         управление прерыванием

     текущие требования к драйверу:
       - надо реализовать тактирование либо от FCPU с предделителем, либо от
         другого таймера.
       - надо назначать период счета
       - надо иметь компаратор с прерыванием - для будильника rtimer, или
           будильников модели.
 */

#ifndef HAL_TIMER_HAL_HPP_
#define HAL_TIMER_HAL_HPP_

#include "hal_device.h"


class TIM_Device {
public:
    // вкл/выкл счетчика
    void enable(bool onoff);
    bool is_enabled() const;

    //источники тактирования счетчиков
    enum SourceID{
          srcPCLK   = 0
        , srcTIMER1 = 1
        , srcTIMER2 = 2
        , srcTIMER3 = 3
    };
    //  тактирование от системного клока требует прескалера, и мы его используем
    DevResult   cfg_internalCLK(unsigned prescale);
    DevResult   cfg_source(SourceID x);

    enum ModeID{
          modeUP    = 1
        , modeDOWN  = 2
        , modeSWEEP = 3
    };
    typedef unsigned timer_mode_t;
    DevResult   set_period(unsigned period, timer_mode_t mode);
    // содержимое счетчика
    unsigned    value() const;
    void        value(unsigned x);
    // содержимое регистра перезагрузки
    unsigned    period() const;
    void        period(unsigned x);

public:
    // каналы сравнения использую для организации таймеров событий на часах.
    enum CompareModeID{
          compOFF       = 0
        , compPULSE     = 1
        , compPULSENEG  = 2
        , compTWIST     = 3
        , compFASTPWM   = 6
        , compFASTPWMNEG= 7
    };
    typedef unsigned chanel_id;
    typedef unsigned chanel_mode_t;
    DevResult   set_compare(chanel_id chanel, unsigned value, chanel_mode_t mode);
    // содержимое компаратора
    unsigned    compare(chanel_id chanel) const;
    void        compare(chanel_id chanel, unsigned x);

    DevResult   compare_mode(chanel_id ch, chanel_mode_t mode);

public:
    // разрешение/запрещение прерывания NVIC таймера
    DevResult    irq_enable();
    DevResult    irq_disable();

    /*
    enum IRQId{
          irqZERO   = 1
        , irqRELOAD = 2

        // прерывания событий от компараторов
        , irqOC1    = 0x10
        , irqOC2    = 0x20
        , irqOC3    = 0x40
        , irqOC4    = 0x80
        , irqOC_Msk = 0xf0, irqOC_Pos = 4
    };
    */
    // \arg mode - набор прерываний IRQId, зависит от локализации к железу
    DevResult    irq_enable(unsigned mode);
    DevResult    irq_disable(unsigned mode);

public:
    //  обработчики прерываний сделаю полями, чтобы ускорить вызов - избегаю
    //      диспетчеризации виртуальных методов, с медленной флешой.

    typedef void (*event_handle)(TIM_Device* self);
    // \arg chanels - битовая маска набор активных каналов
    typedef void (*chanel_handle)(TIM_Device* self, unsigned chanels);

    // обработчики событий счетчика:
    //  по переполнению/заворачиванию счетчика
    event_handle    on_period;
    //  событие от компаратора
    chanel_handle   on_compare;
};



#endif /* HAL_TIMER_HAL_HPP_ */
