// sfmt.cpp cCve[Vt@C
#include "sfmt.h"
#include <time.h>                       // time()
#include <assert.h>


_SGC_BEGIN                              // namespace sgc {


////////////////////////////////////////////////////////////////////////////////
// public

sfmt::sfmt(const uint32_t seed /* = 0 */)
{
	init_param();

	// {block}
	// V[h̏
	{
		uint32_t seed2 = seed;
		if(seed2 == 0)
		{
			const time_t unixtime = time(NULL);
			seed2 = static_cast<uint32_t>(unixtime);
		}
		init_gen_rand(seed2);
	}
}

sfmt::sfmt(const uint32_t *init_key, const int key_length)
{
	init_param();
	init_by_array(init_key, key_length);
}

sfmt::~sfmt()
{
}

uint32_t sfmt::gen32(void)
{
	return gen_rand32();
}

uint32_t sfmt::gen32(const uint32_t upper_bound)
{
	return gen32() % (upper_bound + 1);
}

uint32_t sfmt::gen32(const uint32_t lower_bound, const uint32_t upper_bound)
{
	return gen32(upper_bound - lower_bound) + lower_bound;
}

uint64_t sfmt::gen64(void)
{
	return gen_rand64();
}

uint64_t sfmt::gen64(const uint64_t upper_bound)
{
	return gen64() % (upper_bound + 1);
}

uint64_t sfmt::gen64(const uint64_t lower_bound, const uint64_t upper_bound)
{
	return gen64(upper_bound - lower_bound) + lower_bound;
}


////////////////////////////////////////////////////////////////////////////////

// p[^̏
void sfmt::init_param(void)
{
	m_psfmt32 = &m_sfmt[0].u[0];
	m_psfmt64 = (uint64_t *)&m_sfmt[0].u[0];

	m_initialized = 0;

	m_parity[0] = PARITY1;
	m_parity[1] = PARITY2;
	m_parity[2] = PARITY3;
	m_parity[3] = PARITY4;
}


////////////////////////////////////////////////////////////////////////////////

uint32_t sfmt::gen_rand32(void)
{
    uint32_t r;

    assert(m_initialized);
    if (m_idx >= N32)
	{
		gen_rand_all();
		m_idx = 0;
    }
    r = m_psfmt32[m_idx++];
    return r;
}

uint64_t sfmt::gen_rand64(void)
{
    uint64_t r;

    assert(m_initialized);
    assert(m_idx % 2 == 0);

    if (m_idx >= N32)
	{
		gen_rand_all();
		m_idx = 0;
    }
    r = m_psfmt64[m_idx / 2];
    m_idx += 2;
    return r;
}


void sfmt::fill_array32(uint32_t *array, int size)
{
    assert(m_initialized);
    assert(m_idx == N32);
    assert(size % 4 == 0);
    assert(size >= N32);

    gen_rand_array((w128_t *)array, size / 4);
    m_idx = N32;
}

void sfmt::fill_array64(uint64_t *array, int size)
{
    assert(m_initialized);
    assert(m_idx == N32);
    assert(size % 2 == 0);
    assert(size >= N64);

    gen_rand_array((w128_t *)array, size / 2);
    m_idx = N32;
}

void sfmt::init_gen_rand(uint32_t seed)
{
    int i;

    m_psfmt32[0] = seed;
    for (i = 1; i < N32; i++)
	{
		m_psfmt32[i] = 1812433253UL * (m_psfmt32[i - 1] ^ (m_psfmt32[i - 1] >> 30)) + i;
    }
    m_idx = N32;
    period_certification();
    m_initialized = 1;
}

void sfmt::init_by_array(const uint32_t *init_key, int key_length)
{
    int i, j, count;
    uint32_t r;
    int lag;
    int mid;
    int size = N * 4;

    if (size >= 623)
	{
		lag = 11;
    }
	else if (size >= 68)
	{
		lag = 7;
    }
	else if (size >= 39)
	{
		lag = 5;
    }
	else
	{
		lag = 3;
    }
    mid = (size - lag) / 2;

    memset(m_sfmt, 0x8b, sizeof(m_sfmt));
    if (key_length + 1 > N32)
	{
		count = key_length + 1;
    }
	else
	{
		count = N32;
    }
    r = func1(m_psfmt32[0] ^ m_psfmt32[mid] 
	      ^ m_psfmt32[N32 - 1]);
    m_psfmt32[mid] += r;
    r += key_length;
    m_psfmt32[mid + lag] += r;
    m_psfmt32[0] = r;

    count--;
    for (i = 1, j = 0; (j < count) && (j < key_length); j++)
	{
		r = func1(m_psfmt32[i] ^ m_psfmt32[(i + mid) % N32] 
			  ^ m_psfmt32[(i + N32 - 1) % N32]);
		m_psfmt32[(i + mid) % N32] += r;
		r += init_key[j] + i;
		m_psfmt32[(i + mid + lag) % N32] += r;
		m_psfmt32[i] = r;
		i = (i + 1) % N32;
    }
    for (; j < count; j++)
	{
		r = func1(m_psfmt32[i] ^ m_psfmt32[(i + mid) % N32] 
			  ^ m_psfmt32[(i + N32 - 1) % N32]);
		m_psfmt32[(i + mid) % N32] += r;
		r += i;
		m_psfmt32[(i + mid + lag) % N32] += r;
		m_psfmt32[i] = r;
		i = (i + 1) % N32;
    }
    for (j = 0; j < N32; j++)
	{
		r = func2(m_psfmt32[i] + m_psfmt32[(i + mid) % N32] 
			  + m_psfmt32[(i + N32 - 1) % N32]);
		m_psfmt32[(i + mid) % N32] ^= r;
		r -= i;
		m_psfmt32[(i + mid + lag) % N32] ^= r;
		m_psfmt32[i] = r;
		i = (i + 1) % N32;
    }

    m_idx = N32;
    period_certification();
    m_initialized = 1;
}


////////////////////////////////////////////////////////////////////////////////
// private

void sfmt::rshift128(w128_t *out,  w128_t const *in, int shift)
{
    uint64_t th, tl, oh, ol;

    th = ((uint64_t)in->u[3] << 32) | ((uint64_t)in->u[2]);
    tl = ((uint64_t)in->u[1] << 32) | ((uint64_t)in->u[0]);

    oh = th >> (shift * 8);
    ol = tl >> (shift * 8);
    ol |= th << (64 - shift * 8);
    out->u[1] = (uint32_t)(ol >> 32);
    out->u[0] = (uint32_t)ol;
    out->u[3] = (uint32_t)(oh >> 32);
    out->u[2] = (uint32_t)oh;
}

void sfmt::lshift128(w128_t *out,  w128_t const *in, int shift)
{
    uint64_t th, tl, oh, ol;

    th = ((uint64_t)in->u[3] << 32) | ((uint64_t)in->u[2]);
    tl = ((uint64_t)in->u[1] << 32) | ((uint64_t)in->u[0]);

    oh = th << (shift * 8);
    ol = tl << (shift * 8);
    oh |= tl >> (64 - shift * 8);
    out->u[1] = (uint32_t)(ol >> 32);
    out->u[0] = (uint32_t)ol;
    out->u[3] = (uint32_t)(oh >> 32);
    out->u[2] = (uint32_t)oh;
}

void sfmt::do_recursion(w128_t *r, w128_t *a, w128_t *b, w128_t *c, w128_t *d)
{
    w128_t x;
    w128_t y;

    lshift128(&x, a, SL2);
    rshift128(&y, c, SR2);
    r->u[0] = a->u[0] ^ x.u[0] ^ ((b->u[0] >> SR1) & MSK1) ^ y.u[0] 
	^ (d->u[0] << SL1);
    r->u[1] = a->u[1] ^ x.u[1] ^ ((b->u[1] >> SR1) & MSK2) ^ y.u[1] 
	^ (d->u[1] << SL1);
    r->u[2] = a->u[2] ^ x.u[2] ^ ((b->u[2] >> SR1) & MSK3) ^ y.u[2] 
	^ (d->u[2] << SL1);
    r->u[3] = a->u[3] ^ x.u[3] ^ ((b->u[3] >> SR1) & MSK4) ^ y.u[3] 
	^ (d->u[3] << SL1);
}

void sfmt::gen_rand_all(void)
{
    int i;
    w128_t *r1, *r2;

    r1 = &m_sfmt[N - 2];
    r2 = &m_sfmt[N - 1];
    for (i = 0; i < N - POS1; i++)
	{
		do_recursion(&m_sfmt[i], &m_sfmt[i], &m_sfmt[i + POS1], r1, r2);
		r1 = r2;
		r2 = &m_sfmt[i];
    }
    for (; i < N; i++)
	{
		do_recursion(&m_sfmt[i], &m_sfmt[i], &m_sfmt[i + POS1 - N], r1, r2);
		r1 = r2;
		r2 = &m_sfmt[i];
    }
}

void sfmt::gen_rand_array(w128_t *array, int size)
{
    int i, j;
    w128_t *r1, *r2;

    r1 = &m_sfmt[N - 2];
    r2 = &m_sfmt[N - 1];
    for (i = 0; i < N - POS1; i++)
	{
		do_recursion(&array[i], &m_sfmt[i], &m_sfmt[i + POS1], r1, r2);
		r1 = r2;
		r2 = &array[i];
    }
    for (; i < N; i++)
	{
		do_recursion(&array[i], &m_sfmt[i], &array[i + POS1 - N], r1, r2);
		r1 = r2;
		r2 = &array[i];
    }
    for (; i < size - N; i++)
	{
		do_recursion(&array[i], &array[i - N], &array[i + POS1 - N], r1, r2);
		r1 = r2;
		r2 = &array[i];
    }
    for (j = 0; j < 2 * N - size; j++)
	{
		m_sfmt[j] = array[j + size - N];
    }
    for (; i < size; i++, j++)
	{
		do_recursion(&array[i], &array[i - N], &array[i + POS1 - N], r1, r2);
		r1 = r2;
		r2 = &array[i];
		m_sfmt[j] = array[i];
    }
}

uint32_t sfmt::func1(uint32_t x)
{
    return (x ^ (x >> 27)) * (uint32_t)1664525UL;
}

uint32_t sfmt::func2(uint32_t x)
{
    return (x ^ (x >> 27)) * (uint32_t)1566083941UL;
}

void sfmt::period_certification(void)
{
    int inner = 0;
    int i, j;
    uint32_t work;

    for (i = 0; i < 4; i++)
	inner ^= m_psfmt32[i] & m_parity[i];
    for (i = 16; i > 0; i >>= 1)
	inner ^= inner >> i;
    inner &= 1;
    /* check OK */
    if (inner == 1)
	{
		return;
    }
    /* check NG, and modification */
    for (i = 0; i < 4; i++)
	{
		work = 1;
		for (j = 0; j < 32; j++)
		{
			if ((work & m_parity[i]) != 0)
			{
				m_psfmt32[i] ^= work;
				return;
			}
			work = work << 1;
		}
    }
}

_SGC_END                                // }
