/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "window.h"

namespace Graph2D
{
	float Window::getAnimationTime()
	{
		return 0.38f;
	}

	Window::Window() : animationType(WINDOW_ANIMATION_TYPE_CLOSED)
	{
		type = TYPE_WINDOW;
		//setClipping(true);
		setVisible(false);
	}

	void Window::open()
	{
		setVisible(true);

		// アニメーション時間
		animationWindowTime = 0;

		// アニメーション開始（変更禁止）
		animationType = WINDOW_ANIMATION_TYPE_OPEN;

		// 元の色（変更禁止）
		animationWindowColor.set(0, 0, 0, 0);

		// 元のスケール（変更禁止）
		animationWindowScale.set(0, 0);
	}

	void Window::close()
	{
		// アニメーション時間
		animationWindowTime = 0;

		// アニメーション開始（変更禁止）
		animationType = WINDOW_ANIMATION_TYPE_CLOSE;

		// 元の色（変更禁止）
		animationWindowColor.set(1, 1, 1, 1);

		// 元のスケール（変更禁止）
		animationWindowScale.set(1, 1);
	}

	bool Window::closed() const
	{
		return animationType == WINDOW_ANIMATION_TYPE_CLOSED;
	}

	bool Window::isAnimationStoped() const
	{
		return animationType >= WINDOW_ANIMATION_TYPE_OPENED;
	}

	void Window::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);

		mana_stream_push_unsigned_char(stream, static_cast<unsigned char>(animationType));
		mana_stream_push_float(stream, animationWindowTime);
		animationWindowColor.serialize(stream);
		animationWindowScale.serialize(stream);

		mana_stream_mark(stream);
	}

	void Window::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);

		animationType = (WindowAnimationType)mana_stream_pop_unsigned_char(stream);
		animationWindowTime = mana_stream_pop_float(stream);
		animationWindowColor.deserialize(stream);
		animationWindowScale.deserialize(stream);

		mana_stream_check(stream);
	}
	
	void Window::onUpdateLayout()
	{
	}

	void Window::onUpdate(const UpdateInfomation& updateInfomation)
	{
		if(animationType < WINDOW_ANIMATION_TYPE_OPENED)
		{
			const float normal = animationWindowTime / getAnimationTime();
			const float radian = 3.14159265f * 0.5f * normal;

			if(animationType == WINDOW_ANIMATION_TYPE_OPEN)
			{
				animationWindowColor.set(normal, normal, normal, normal);

				const float ratio = sinf(radian);
				animationWindowScale.set(ratio, ratio);
			}
			else
			{
				const float inverse = 1.0f - normal;
				animationWindowColor.set(inverse, inverse, inverse, inverse);

				const float ratio = cosf(radian);
				animationWindowScale.set(ratio, ratio);
			}

			animationWindowTime += updateInfomation.deltaTime;
			if(animationWindowTime >= getAnimationTime())
			{
				if(animationType == WINDOW_ANIMATION_TYPE_OPEN)
				{
					animationWindowColor.set(1, 1, 1, 1);
					animationWindowScale.set(1, 1);
					animationType = WINDOW_ANIMATION_TYPE_OPENED;
				}
				else
				{
					animationWindowColor.set(0, 0, 0, 0);
					animationWindowScale.set(0, 0);
					setVisible(false);
					animationType = WINDOW_ANIMATION_TYPE_CLOSED;
				}
			}
		}
		super::onUpdate(updateInfomation);
	}

	void Window::onDraw(const DrawRect& drawRect)
	{
		const Vector2 s = drawRect.getDrawSize() * 0.5f;
		const Vector2 p = drawRect.getDrawLeftTopPosition() + s - s * animationWindowScale;

		DrawRect newDrawRect(drawRect);
		newDrawRect.position = p;
		newDrawRect.scale = drawRect.scale * animationWindowScale;
		newDrawRect.color = drawRect.color * animationWindowColor;
		newDrawRect.updateClippingInfomation(drawRect);

		super::onDraw(newDrawRect);
	}

	bool Window::touchesBegan(const Vector2& localPosition)
	{
		return isAnimationStoped() ? super::touchesBegan(localPosition) : false;
	}

	bool Window::touchesMoved(const Vector2& localPosition)
	{
		return isAnimationStoped() ? super::touchesMoved(localPosition) : false;
	}

	bool Window::touchesEnded(const Vector2& localPosition)
	{
		return isAnimationStoped() ? super::touchesEnded(localPosition) : false;
	}

	bool Window::touchesCancelled(const Vector2& localPosition)
	{
		return isAnimationStoped() ? super::touchesCancelled(localPosition) : false;
	}

	bool Window::compare(const Window& window) const
	{
		if(!super::compare(window))
			return false;
		if(animationType != window.animationType)
			return false;
		if(animationWindowScale != window.animationWindowScale)
			return false;
		if(animationWindowColor != window.animationWindowColor)
			return false;
		return true;
	}
}
