/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "common.h"
#include "abort.h"
#include "graphicDevice.h"
#include "messageTranslater.h"
#include "text.h"

#define TEXT_SIZE (12)

namespace Graph2D
{
	Text::Text()
		: instance(NULL)
		, dirty(true)
	{
		type = TYPE_TEXT;
		color = Color(0.0f, 0.0f, 0.0f, 0.0f);

		setOwnerDraw(true);

		instance = Implementation::Text::create(TEXT_SIZE);
	}

	Text::Text(const size_t fontSize)
		: instance(NULL)
		, dirty(true)
	{
		type = TYPE_TEXT;
		color = Color(0.0f, 0.0f, 0.0f, 0.0f);

		setOwnerDraw(true);
		
		instance = Implementation::Text::create(fontSize);
	}

	Text::~Text()
	{
		if(instance)
		{
			Implementation::Text::release(instance);
			instance = NULL;
		}
	}

	void Text::clear()
	{
		if(!text.empty())
		{
			text.clear();
			dirty = true;
		}
	}

	void Text::setFontName(const std::string& fontName)
	{
		Implementation::Text::setFontName(instance, fontName);
		dirty = true;
	}

	void Text::setFontSize(const float fontSize)
	{
		Implementation::Text::setFontSize(instance, fontSize);
		dirty = true;
	}

	const Color& Text::getTextColor() const
	{
		return textColor;
	}

	void Text::setTextColor(const Color& color)
	{
		textColor = color;
	}

	TextAlignmentH Text::getTextAlignmentH() const
	{
		return TEXT_ALIGNMENT_LEFT;
	}

	void Text::setTextAlignmentH(const TextAlignmentH textAlignment)
	{
	}

	TextAlignmentV Text::getTextAlignmentV() const
	{
		return TEXT_ALIGNMENT_TOP;
	}

	void Text::setTextAlignmentV(const TextAlignmentV textAlignment)
	{
	}

	const std::string& Text::getString() const
	{
		return text;
	}

	void Text::setString(const std::string& text)
	{
		if(this->text != text)
		{
			this->text = text;

			if(this->text.length() > 0)
			{
				MessageTranslater::translate(this->text);
			}
			else
			{
				this->text.clear();
			}

			dirty = true;
		}
	}

	void Text::format(const char* format, ...)
	{
		va_list ap;
		va_start(ap, format);
		vformat(format, ap);
		va_end(ap);
	}
	
	void Text::vformat(const char* format, va_list ap)
	{
		char *p, *np;
		int size = 100;
		
		if((p = reinterpret_cast<char*>(g2d_malloc(size))) == NULL)
			return;
		
		for(;;)
		{
#if __STDC_WANT_SECURE_LIB__
			const int n = vsnprintf_s(p, size, _TRUNCATE, format, ap);
#else
			const int n = vsnprintf(p, size, format, ap);
#endif
			
			if(n > -1 && n < size)
			{
				setString(std::string(p));
				break;
			}
			if(n > -1)
				size = n + 1;
			else
				size *= 2;
			if((np = reinterpret_cast<char*>(g2d_realloc(p, size))) == NULL)
			{
				break;
			}

			p = np;
		}
		
		setString(std::string(p));
		
		g2d_free(p);
	}

	void Text::replace(const char* key, const char* word)
	{
		const size_t length = strlen(key);

		int position = 0;
		while((position = text.find(key, position)) != std::string::npos)
		{
			text.replace(position, length, word);
		}
	}

	void Text::adjustSize()
	{
		setSize(getAdjustedSize());
	}

	Vector2 Text::getAdjustedSize()
	{
		return Implementation::Text::getSize(instance, text);
	}

	void Text::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);

		/*
		mana_stream_push_string(stream, (char*)text.c_str());
		mana_stream_push_string(stream, (char*)fontName.c_str());
		mana_stream_push_float(stream, fontSize);
		textColor.serialize(stream);
		*/
		mana_stream_mark(stream);
	}

	void Text::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);

		/*
		[this setString:[NSString stringWithUTF8String:mana_stream_get_string_pointer(stream)]];
		mana_steram_seek(stream, mana_stream_get_string_length(stream));

		[this setFontName:[NSString stringWithUTF8String:mana_stream_get_string_pointer(stream)]];
		mana_steram_seek(stream, mana_stream_get_string_length(stream));

		[this setFontSize:mana_stream_pop_float(stream)];
		textColor.deserialize(stream);
		*/
		mana_stream_check(stream);
	}

	void Text::onUpdate(const UpdateInfomation& updateInfomation)
	{
		if(dirty)
		{
			adjustSize();
			Implementation::Text::set(instance, size.x, size.y, text);
			dirty = false;
		}
		
		super::onUpdate(updateInfomation);
	}

	void Text::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);

		if(!text.empty())
		{
			Texture::unbind();
			GraphicDevice::enableTexture(true);
			GraphicDevice::enableOpaque(false);
			//GraphicDevice::enableScissor(false);
			GraphicDevice::enableColorArray(false);
			GraphicDevice::enableTextureCoordArray(true);
			{
				const Vector2 drawPosition = drawRect.getDrawLeftTopPosition();
				const Vector2 drawSize = drawRect.getDrawSize();

				const Color currentColor = getDrawColor(drawRect, textColor);

				const Color shadowColor(currentColor.r * 0.1f, currentColor.g * 0.1f, currentColor.b * 0.1f, currentColor.a);
				GraphicDevice::setColor(shadowColor);
				Implementation::Text::draw(instance, drawPosition + Vector2(1.f, 1.f), drawSize);

				GraphicDevice::setColor(currentColor);
				Implementation::Text::draw(instance, drawPosition, drawSize);
			}
			Texture::unbind();
			GraphicDevice::applyDrawArrayPointer();
		}
	}

	bool Text::compare(const Text& other) const
	{
		if(!super::compare(other))
			return false;
		if(text != other.text)
			return false;
		if(textColor != other.textColor)
			return false;
		if(Implementation::Text::getFontSize(instance) != Implementation::Text::getFontSize(other.instance))
			return false;
		if(strcmp(Implementation::Text::getFontName(instance), Implementation::Text::getFontName(other.instance)) != 0)
			return false;
		return true;
	}
}
