#include <libkern/OSByteOrder.h>
#include <fstream>
#include "PVRTCLoader.h"

using namespace std;

namespace Graph2D
{
	namespace Implementation
	{
		const char PVRTCLoader::PVRIdentifier[4] = { 'P', 'V', 'R', '!' };

		PVRTCLoader::PVRTCLoader()
			: name(0)
			, numSurfaces(0)
		{
		}

		PVRTCLoader::PVRTCLoader(const char* path)
			: name(0)
			, numSurfaces(0)
		{
			loadFromFile(path);
		}

		PVRTCLoader::~PVRTCLoader()
		{
			glDeleteTextures(1, &name);
		}

		bool PVRTCLoader::loadFromFile(const char* path)
		{
			try
			{
				std::ifstream file;
				file.open(path);
				if(!file.good())
					return false;

				file.seekg(0, ios::end);
				int size = file.tellg();
				file.seekg(0, ios::beg);

				readBuffer = new char[size];
				file.read(readBuffer, size);

				bool result = loadFromMemory(readBuffer);

				delete[] readBuffer;
				readBuffer = NULL;

				return result;
			}
			catch(...)
			{
				delete[] readBuffer;
				readBuffer = NULL;

				throw;
			}
		}

		bool PVRTCLoader::loadFromMemory(const void* pData)
		{
			const Header& header = *reinterpret_cast<const Header *>(pData);
			unsigned int tag = OSSwapLittleToHostInt32(header.tag);

			if(
				PVRIdentifier[0] != ((tag >>  0) & 0xff) ||
				PVRIdentifier[1] != ((tag >>  8) & 0xff) ||
				PVRIdentifier[2] != ((tag >> 16) & 0xff) ||
				PVRIdentifier[3] != ((tag >> 24) & 0xff) )
			{
				return false;
			}

			if(name)
			{
				glDeleteTextures(1, &name);
				name = 0;
			}

			unsigned int flags       = OSSwapLittleToHostInt32(header.flags);
			unsigned int formatFlags = flags & PVRTC_FLAG_TYPE_MASK;

			if(formatFlags == PVRTC_FLAG_TYPE_PVRTC_4 || formatFlags == PVRTC_FLAG_TYPE_PVRTC_2)
			{
				if(formatFlags == PVRTC_FLAG_TYPE_PVRTC_4)
					format = GL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;
				else if(formatFlags == PVRTC_FLAG_TYPE_PVRTC_2)
					format = GL_COMPRESSED_RGBA_PVRTC_2BPPV1_IMG;
				else
					return false;

				width       = OSSwapLittleToHostInt32(header.width);
				height      = OSSwapLittleToHostInt32(header.height);
				alpha       = OSSwapLittleToHostInt32(header.bitmaskAlpha) ? true : false;
				numSurfaces = 0;

				GLuint         w      = width;
				GLuint         h      = height;
				GLuint         offset = 0;
				GLuint         size   = OSSwapLittleToHostInt32(header.dataSize);
				const uint8_t* pBytes = reinterpret_cast<const uint8_t*>(pData) + sizeof(header);

				while(offset < size && numSurfaces < PVRTC_MAX_SURFACES)
				{
					GLuint   blockSize, widthBlocks, heightBlocks, bpp;
					Surface& surface = surfaces[numSurfaces++];

					if(formatFlags == PVRTC_FLAG_TYPE_PVRTC_4)
					{
						blockSize    = 4 * 4;
						widthBlocks  = w / 4;
						heightBlocks = h / 4;
						bpp = 4;
					}
					else
					{
						blockSize    = 8 * 4;
						widthBlocks  = w / 8;
						heightBlocks = h / 4;
						bpp = 2;
					}

					if(widthBlocks < 2)
						widthBlocks = 2;
					if(heightBlocks < 2)
						heightBlocks = 2;

					surface.size = widthBlocks * heightBlocks * ((blockSize  * bpp) / 8);
					surface.bits = &pBytes[offset];
					w >>= 1;
					if(w <= 0)
						w = 1;
					h >>= 1;
					if(h <= 0)
						h = 1;
					offset += surface.size;
				}

				glGenTextures(1, &name);
				glBindTexture(GL_TEXTURE_2D, name);

				w = width;
				h = height;

				for(GLuint i = 0 ; i < numSurfaces; ++i)
				{
					const Surface& surface = surfaces[i];

					glCompressedTexImage2D(
						GL_TEXTURE_2D,
						i,
						format,
						w,
						h,
						0,
						surface.size,
						surface.bits
					);
					w >>= 1;
					if(w <= 0)
						w = 1;
					h >>= 1;
					if(h <= 0)
						h = 1;
				}

				return true;
			}
			else
			{
				return false;
			}
		}
	}
}
