/*
 * graph2D
 * Copyright (c) 2011 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#import <StoreKit/StoreKit.h>

@interface Kiosk_impl : NSObject<SKProductsRequestDelegate, SKPaymentTransactionObserver>
{
	bool requesting;
	bool success;
}
- (id)init;
- (void)dealloc;
- (bool)valid;
- (bool)request:(NSString*)productIdentifier;
- (bool)busy;
- (bool)result;
@end

@implementation Kiosk_impl

- (id)init
{
	if((self = [super init]) != nil)
	{
		requesting = false;
		success = false;
	}
	
	return self;
}

////////////////////////////////////////////////////////////////////////////////
- (void)dealloc
{
	while(requesting)
	{
		[NSThread sleepForTimeInterval:0.5];
	}

	[super dealloc];
}

////////////////////////////////////////////////////////////////////////////////
- (bool)valid
{
	return [SKPaymentQueue canMakePayments];
}

////////////////////////////////////////////////////////////////////////////////
- (bool)request:(NSString*)productIdentifier
{
	if(requesting)
		return false;

	[[SKPaymentQueue defaultQueue] addTransactionObserver:self];

	NSSet* productIdentifiers = [NSSet setWithObject:productIdentifier];
	SKProductsRequest* skProductRequest = [[SKProductsRequest alloc] initWithProductIdentifiers:productIdentifiers];
	skProductRequest.delegate = self;
	[skProductRequest start];

	requesting = true;
	success = false;

	return true;
}

////////////////////////////////////////////////////////////////////////////////
- (bool)busy
{
	return requesting;
}

////////////////////////////////////////////////////////////////////////////////
- (bool)result
{
	while(requesting)
	{
		[NSThread sleepForTimeInterval:0.5];
	}

	return success;
}

////////////////////////////////////////////////////////////////////////////////
#pragma mark -
#pragma mark SKProductsRequestDelegate

- (void)productsRequest:(SKProductsRequest *)request didReceiveResponse:(SKProductsResponse *)response
{
	if(response == nil)
	{
		NSLog(@"Product Response is nil");
		return;
	}

	// 確認できなかったidentifierをログに記録
	for(NSString* identifier in response.invalidProductIdentifiers)
	{
		NSLog(@"invalid product identifier: %@", identifier);
	}

	// 確認できたidentifierをログに記録
	for(SKProduct* product in response.products)
	{
		NSLog(@"valid product identifier: %@", product.productIdentifier);

		SKPayment* payment = [SKPayment paymentWithProduct:product];
		[[SKPaymentQueue defaultQueue] addPayment:payment];
	}
}

////////////////////////////////////////////////////////////////////////////////
#pragma mark -
#pragma mark SKPaymentTransactionObserver

- (void)paymentQueue:(SKPaymentQueue*)queue updatedTransactions:(NSArray*)transactions
{
	BOOL purchasing = YES;

	for(SKPaymentTransaction* transaction in transactions)
	{
		switch(transaction.transactionState)
		{
		// 購入中
		case SKPaymentTransactionStatePurchasing:
			{
				NSLog(@"Payment Transaction Purchasing");
				break;
			}
		// 購入成功
		case SKPaymentTransactionStatePurchased:
			{
				NSLog(@"Payment Transaction END Purchased: %@", transaction.transactionIdentifier);
				purchasing = NO;
				success = true;
				[queue finishTransaction:transaction];
				break;
			}
		// 購入失敗
		case SKPaymentTransactionStateFailed:
			{
				NSLog(@"Payment Transaction END Failed: %@ %@", transaction.transactionIdentifier, transaction.error);
				if(transaction.error.code != SKErrorPaymentCancelled)
				{
					// 必要に応じてここでエラーを表示する
				}
				purchasing = NO;
				[queue finishTransaction:transaction];
				break;
			}
		// 購入履歴復元
		case SKPaymentTransactionStateRestored:
			{
				NSLog(@"Payment Transaction END Restored: %@", transaction.transactionIdentifier);
				purchasing = NO;
				[queue finishTransaction:transaction];
				break;
			}
		}
	}

	if(purchasing == NO)
	{
		requesting = false;
	}
}

@end

////////////////////////////////////////////////////////////////////////////////
namespace Graph2D
{
	namespace Implementation
	{
		namespace Kiosk
		{
			void* initialize()
			{
				Kiosk_impl* self = [[Kiosk_impl alloc] init];

				return reinterpret_cast<void*>(self);
			}

			void finalize(void* instance)
			{
				Kiosk_impl* self = reinterpret_cast<Kiosk_impl*>(instance);

				[self release];
			}

			bool valid(void* instance)
			{
				Kiosk_impl* self = reinterpret_cast<Kiosk_impl*>(instance);

				return [self valid];
			}

			bool request(void* instance, const char* productIdentifier)
			{
				Kiosk_impl* self = reinterpret_cast<Kiosk_impl*>(instance);

				return [self request:[NSString stringWithUTF8String:productIdentifier]];
			}

			bool busy(void* instance)
			{
				Kiosk_impl* self = reinterpret_cast<Kiosk_impl*>(instance);

				return [self busy];
			}

			bool result(void* instance)
			{
				Kiosk_impl* self = reinterpret_cast<Kiosk_impl*>(instance);

				return [self result];
			}
		}
	}
}

