/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#import "desktopView.h"
#import "text_impl.h"
#import "../../desktop.h"
#import "../../profile.h"

@interface DesktopView(Private)
+(Class)layerClass;
-(BOOL)createFramebuffer;
-(void)destroyFramebuffer;
-(void)drawView;
@end

@implementation DesktopView

@synthesize context;
@synthesize animationTimer;
@synthesize animationInterval;

static DesktopView* _instances = nil;

+(DesktopView*)setup:(CGRect)aRect scale:(float)contentScaleFactor
{
	if(_instances == nil)
	{
		_instances = [[DesktopView alloc] initWithFrame:aRect scale:contentScaleFactor];
	}
	return _instances;
}

-(void)finalize
{
	if(_instances == nil)
	{
		[_instances release];
		_instances = nil;
	}
	[super finalize];
}

+(DesktopView*)instance
{
	if(_instances == nil)
	{
		_instances = [[DesktopView alloc] init];
	}
	return _instances;
}

+(id)allocWithZone:(NSZone*)zone
{
	if(_instances == nil)
	{
		_instances = [super allocWithZone:zone];
	}
	return _instances;
}

-(id)copyWithZone:(NSZone*)zone
{
	return self;
}

-(id)retain
{
	return self;
}

-(unsigned)returnCount
{
	return UINT_MAX;
}

-(void)release
{
}

-(id)autorelease
{
	return self;
}

-(id)init
{
	UIScreen* mainScreen = [UIScreen mainScreen];
	return [self initWithFrame:[mainScreen bounds] scale:mainScreen.scale];
}

-(id)initWithFrame:(CGRect)aRect scale:(float)contentScaleFactor
{
	// アプリケーションの表示フレームを取得する
	CGRect frame;
	frame = aRect;

	// 中心位置を作成
	CGPoint center;
	center.x = CGRectGetWidth(frame) * 0.5f;
	center.y = CGRectGetHeight(frame) * 0.5f;

	// バウンズを作成
	CGRect bounds;
	bounds.origin = CGPointZero;
	bounds.size.width = CGRectGetHeight(frame);
	bounds.size.height = CGRectGetWidth(frame);

	if(self = [super initWithFrame:bounds])
	{
		// アフィン変換を作成
		CGAffineTransform transform;
		transform = CGAffineTransformMakeRotation(M_PI_2);

		// ビューを回転させる
		self.center = center;
		self.bounds = bounds;
		self.transform = transform;

		// アタッチされているUIWindowを初期化
		window = nil;

		// 描画レイヤーを設定
		CAEAGLLayer* eaglLayer = (CAEAGLLayer*)self.layer;
		eaglLayer.opaque = YES;
		eaglLayer.drawableProperties = [NSDictionary dictionaryWithObjectsAndKeys:
										[NSNumber numberWithBool:NO], kEAGLDrawablePropertyRetainedBacking,
										kEAGLColorFormatRGBA8, kEAGLDrawablePropertyColorFormat,
										nil];

		// 描画コンテキストを生成
		context = [[EAGLContext alloc] initWithAPI:kEAGLRenderingAPIOpenGLES1];
		if(!context || ![EAGLContext setCurrentContext:context])
		{
		    [self release];
		    return nil;
		}

		// スケールファクタを設定
		self.contentScaleFactor = contentScaleFactor;

		// アニメーションインターバルを設定
		[self setAnimationInterval:1.0f / 60.0f];

		// コールバック関数を設定
		[[UIDevice currentDevice] beginGeneratingDeviceOrientationNotifications];
		[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(didRotate:) name:UIDeviceOrientationDidChangeNotification object:nil];

		// リソースを初期化
		Graph2D::Desktop::getInstance().initializeResources();

		// テキストキャッシュを初期化
		Graph2D::Implementation::Text::Cache::initialize(100);
	}

	return self;
}

-(void)dealloc
{
	// テキストキャッシュを解放
	Graph2D::Implementation::Text::Cache::finalize();
	// アニメーションを停止
	[self stopAnimation];
	// フレームバッファーを開放
	[self destroyFramebuffer];
	// 描画コンテキストを解放
	if([EAGLContext currentContext] == context)
	{
		[EAGLContext setCurrentContext:nil];
	}
	[context release];
	// アタッチされているUIWindowを解放
	[window release];
	// 自身を解放
	[super dealloc];
}

// You must implement this method
+(Class)layerClass
{
    return [CAEAGLLayer class];
}

/*!
 * @param[in]	notification	NSNotification
 */
-(void)didRotate:(NSNotification*)notification
{
	UIDeviceOrientation orientation = [[notification object] orientation];

	if(
		orientation == UIDeviceOrientationLandscapeLeft ||
		orientation == UIDeviceOrientationLandscapeRight)
	{
		[UIView beginAnimations:nil context:nil];
		[UIView setAnimationCurve:UIViewAnimationCurveEaseOut];
		[UIView setAnimationDuration:0.50];

		switch(orientation)
		{
			case UIDeviceOrientationLandscapeLeft:
				[self setTransform:CGAffineTransformMakeRotation(M_PI / 2.0)];
				break;
			case UIDeviceOrientationLandscapeRight:
				[self setTransform:CGAffineTransformMakeRotation(M_PI / -2.0)];
				break;
			default:
				break;
		}

		[UIView commitAnimations];
	}
}

-(void)layoutSubviews
{
	[EAGLContext setCurrentContext:context];
	[self destroyFramebuffer];
	[self createFramebuffer];
	[self drawView];
}

-(BOOL)createFramebuffer
{
	GLint backingWidth;
	GLint backingHeight;

	glGenFramebuffersOES(1, &viewFramebuffer);
	glGenRenderbuffersOES(1, &viewRenderbuffer);

	glBindFramebufferOES(GL_FRAMEBUFFER_OES, viewFramebuffer);
	glBindRenderbufferOES(GL_RENDERBUFFER_OES, viewRenderbuffer);
	[context renderbufferStorage:GL_RENDERBUFFER_OES fromDrawable:(CAEAGLLayer*)self.layer];
	glFramebufferRenderbufferOES(GL_FRAMEBUFFER_OES, GL_COLOR_ATTACHMENT0_OES, GL_RENDERBUFFER_OES, viewRenderbuffer);

	glGetRenderbufferParameterivOES(GL_RENDERBUFFER_OES, GL_RENDERBUFFER_WIDTH_OES, &backingWidth);
	glGetRenderbufferParameterivOES(GL_RENDERBUFFER_OES, GL_RENDERBUFFER_HEIGHT_OES, &backingHeight);

	glGenRenderbuffersOES(1, &depthRenderbuffer);
	glBindRenderbufferOES(GL_RENDERBUFFER_OES, depthRenderbuffer);
	glRenderbufferStorageOES(GL_RENDERBUFFER_OES, GL_DEPTH_COMPONENT16_OES, backingWidth, backingHeight);
	glFramebufferRenderbufferOES(GL_FRAMEBUFFER_OES, GL_DEPTH_ATTACHMENT_OES, GL_RENDERBUFFER_OES, depthRenderbuffer);

	if(glCheckFramebufferStatusOES(GL_FRAMEBUFFER_OES) != GL_FRAMEBUFFER_COMPLETE_OES)
	{
		NSLog(@"failed to make complete framebuffer object %x", glCheckFramebufferStatusOES(GL_FRAMEBUFFER_OES));
		return NO;
	}

	Graph2D::GraphicDevice::initialize(
		Graph2D::Vector2(backingWidth, backingHeight) * (1.f / self.contentScaleFactor),
		self.contentScaleFactor
	);

	return YES;
}

-(void)destroyFramebuffer
{
	Graph2D::GraphicDevice::finalize();

	glDeleteFramebuffersOES(1, &viewFramebuffer);
	viewFramebuffer = 0;
	glDeleteRenderbuffersOES(1, &viewRenderbuffer);
	viewRenderbuffer = 0;
	glDeleteRenderbuffersOES(1, &depthRenderbuffer);
	depthRenderbuffer = 0;
}

-(void)attachInWindow:(UIWindow*)_window
{
	if(window)
	{
		[self dettach];
		[window release];
	}
	window = _window;
	if(window)
	{
		[window retain];
		[self attach];
	}
}

-(void)attach
{
	if(window)
	{
		[window addSubview:self];
		[self layoutSubviews];
		[self startAnimation];
	}
}

-(void)dettach
{
	if(window)
	{
		[self stopAnimation];
		[self removeFromSuperview];
	}
}

////////////////////////////////////////////////////////////////////////////////
// draw
////////////////////////////////////////////////////////////////////////////////
-(void)drawView
{
	[EAGLContext setCurrentContext:context];

	glBindFramebufferOES(GL_FRAMEBUFFER_OES, viewFramebuffer);
	glBindRenderbufferOES(GL_RENDERBUFFER_OES, viewRenderbuffer);

	Graph2D::Desktop::getInstance().updateAndDraw();

	[context presentRenderbuffer:GL_RENDERBUFFER_OES];
}

////////////////////////////////////////////////////////////////////////////////
// animation
////////////////////////////////////////////////////////////////////////////////
-(void)startAnimation
{
	self.animationTimer = [NSTimer scheduledTimerWithTimeInterval:animationInterval target:self selector:@selector(drawView) userInfo:nil repeats:YES];
}

-(void)stopAnimation
{
	if(animationTimer)
	{
		[animationTimer invalidate];
		[animationTimer release];
		animationTimer = nil;
	}
}

-(void)setAnimationTimer:(NSTimer*)newTimer
{
	[animationTimer invalidate];
	animationTimer = newTimer;
}

- (void)setAnimationInterval:(NSTimeInterval)interval
{
	animationInterval = interval;
	if(animationTimer)
	{
		[self stopAnimation];
		[self startAnimation];
	}

	Graph2D::Desktop::getInstance().setRefreshTime(interval);

	PROFILE_SET_BASE_FPS(60.0 / (interval / (1.0 / 60.0)) + 0.5);
}

////////////////////////////////////////////////////////////////////////////////
// touch
////////////////////////////////////////////////////////////////////////////////
-(void)touchesBegan:(NSSet*)touches withEvent:(UIEvent*)event
{
	UITouch* touch = [[event allTouches] anyObject];
	CGPoint location = [touch locationInView:touch.view];
	Graph2D::Desktop::getInstance().touchesBegan(Graph2D::Vector2(location.x, location.y));
}

-(void)touchesMoved:(NSSet*)touches withEvent:(UIEvent*)event
{
	UITouch* touch = [[event allTouches] anyObject];
	CGPoint location = [touch locationInView:touch.view];
	Graph2D::Desktop::getInstance().touchesMoved(Graph2D::Vector2(location.x, location.y));
}

-(void)touchesEnded:(NSSet*)touches withEvent:(UIEvent*)event
{
	UITouch* touch = [[event allTouches] anyObject];
	CGPoint location = [touch locationInView:touch.view];
	Graph2D::Desktop::getInstance().touchesEnded(Graph2D::Vector2(location.x, location.y));
}

-(void)touchesCancelled:(NSSet*)touches withEvent:(UIEvent*)event
{
	UITouch* touch = [[event allTouches] anyObject];
	CGPoint location = [touch locationInView:touch.view];
	Graph2D::Desktop::getInstance().touchesCancelled(Graph2D::Vector2(location.x, location.y));
}

@end
