/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#define INCLUDE_FROM_OBJC_C
#include "data_impl.h"

@implementation DataController

/*!
 * ドキュメントパスを取得します
 */
+(NSString*)getDocumentPath
{
	NSArray* paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	return [paths objectAtIndex:0];
}

/*!
 * ドキュメントパスを取得します
 */
+(NSString*)getDocumentPathWithString:(NSString*)filename
{
	NSArray* paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	return [[paths objectAtIndex:0] stringByAppendingPathComponent:filename];
}

/*!
 * リソースパスを取得します
 */
+(NSString*)getResoucePath
{
	return [[NSBundle mainBundle] resourcePath];
}

/*!
 * リソースパスを取得します
 */
+(NSString*)getResoucePathWithString:(NSString*)filename
{
	NSString* bundlePath = [[NSBundle mainBundle] resourcePath];
	return [bundlePath stringByAppendingPathComponent:filename];
}

/*!
 * リソースパスをURI形式で取得します
 */
+(NSURL*)getResouceUriWithString:(NSString*)filename
{
	return [NSURL fileURLWithPath:[DataController getResoucePathWithString:filename]];
}

/*!
 * ファイルが存在するか調べます
 */
+(BOOL)fileExistsAtPath:(NSString*)path
{
	return [[NSFileManager defaultManager] fileExistsAtPath:path];
}

@end

////////////////////////////////////////////////////////////////////////////////
#include "../../abort.h"

namespace Graph2D
{
	namespace Implementation
	{
		namespace Data
		{
			/*!
			 * ドキュメントパスを取得します
			 */
			const char* getDocumentPath()
			{
				return [[DataController getDocumentPath] UTF8String];
			}
			
			/*!
			 * リソースパスを取得します
			 */
			const char* getResoucePath()
			{
				return [[DataController getResoucePath] UTF8String];
			}
			
			/*!
			 * ファイルの存在を確認します
			 */
			bool exist(const std::string& filename, const bool resource)
			{
				NSString* name = [NSString stringWithUTF8String:filename.c_str()];

				NSString* path = resource ?
					[DataController getResoucePathWithString:name] :
					[DataController getDocumentPathWithString:name] ;

				return [[NSFileManager defaultManager] fileExistsAtPath:path] ? true : false;
			}

			/*!
			 * インスタンスを作成してデータを読み込みます
			 */
			void* create(const void* buffer, const size_t size)
			{
				NSData* data = [[NSData alloc] initWithBytes:buffer length:size];

				return reinterpret_cast<void*>(data);
			}

			/*!
			 * インスタンスを作成してファイルを読み込みます
			 */
			void* create(const std::string& filename, const bool resource)
			{
				NSString* name = [NSString stringWithUTF8String:filename.c_str()];

				NSString* path = resource ?
					[DataController getResoucePathWithString:name] :
					[DataController getDocumentPathWithString:name] ;

				NSData* data = [[NSData alloc] initWithContentsOfFile:path];

				return reinterpret_cast<void*>(data);
			}

			/*!
			 * インスタンスを解放します
			 */
			void release(void* self)
			{
				if(self)
				{
					NSData* data = reinterpret_cast<NSData*>(self);
					[data release];
				}
			}

			/*!
			 * データをファイルに保存します
			 */
			bool write(void* self, const std::string& filename)
			{
				NSString* name = [NSString stringWithUTF8String:filename.c_str()];

				NSString* path = [DataController getDocumentPathWithString:name];

				NSData* data = reinterpret_cast<NSData*>(self);
				return [data writeToFile:path atomically:YES];
			}

			/*!
			 * データの先頭アドレスを取得します
			 */
			const void* getBuffer(void* self)
			{
				NSData* data = reinterpret_cast<NSData*>(self);
				return [data bytes];
			}

			/*!
			 * データのサイズを取得します
			 */
			size_t getSize(void* self)
			{
				NSData* data = reinterpret_cast<NSData*>(self);
				return [data length];
			}
		}
	}
}
